package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$"`0`````````````"`!0````!`8`2P````#0``0.T```````T`"``
M"``H`"``'P````8````T`8``-`&``#0```$````!``````0````$`````P``
M`30!@`$T`8`!-````!<````7````!`````$````!``````&````!@`````"6
MG```EIP````%``$```````$``/Z$`8'^A`&!_H0```*0```$$`````8``0``
M`````@``_Q`!@?\0`8'_$````.````#@````!@````0````$```!3`&``4P!
M@`%,````+````"P````$````!&1TY5```(V\`8"-O`&`C;P```$,```!#```
M``0````$9'3E4@``_H0!@?Z$`8'^A````7P```%\````!`````$O=7-R+VQI
M8F5X96,O;&0N96QF7W-O```````'````!`````%.971"4T0``#6DZ0`````$
M````!`````-085@``````````&$```!C``````````````!"````&P``````
M````````"0```$X````#`````````$,`````````/0```!````!)````````
M`#4````&````%P```$@```!1````#````$P`````````'``````````6````
M6P```$```````````````#````!A````8@````````!&````30```$4```!:
M````'@`````````+````/P```$0````T````(0```%@``````````````#8`
M````````+`````````!9````)````"<```!?````!P```%T`````````5@``
M``````!5````,P```"T```````````````````!0````*@``````````````
M7@```#@`````````/``````````Y````8`````````!2````&@```!4`````
M````4P````````````````````````!!`````````%<````O````-P``````
M``!<``````````````````````````````````````````$`````````````
M``````````````````````````````````````````````````4`````````
M``````T````````````````````````````````````````````````````1
M```````````````````````````````3````(P`````````4````!```````
M```````````````````H``````````\`````````&````"D````E````````
M`"L````````````````````*````&0`````````````````````````F````
M`````#$`````````,@```"(``````````````#X````=````$@``````````
M```````````````(``````````````!'````3P```$L`````````````````
M```@``````````(````Z````2@```#L```!4``````````X````?````+@``
M``````````````````````,W`8!$Z````FP2```*```"$0``````````$@``
M`````;,``````````!(```````"C`8(!*`````$1```;```!I0``````````
M$@```````%@``````````!(````````,```````````2````````B```````
M````$@```````68``````````!(```````,\`8`F=````#@2```*```!Z0``
M````````$@```````'D!@$R@`````!(```L```'8```````````2```````!
M<@``````````$@```````B$``````````!(```````'P```````````2````
M```"1P``````````$@```````L4!@@$,````!!$``!D```*D```````````2
M````````^0``````````$@```````;L``````````!(```````!>````````
M```2```````#5P&`'>`````0$@``"@```'\``````````!(```````$,````
M```````2````````%```````````$@```````"D``````````!(```````+0
M`8(!&`````01```:````,0``````````$@```````<T``````````!(`````
M``-H`8`FK````0P2```*````W```````````$@```````WH!@">X```&T!(`
M``H```/Z`8`2H``````2```)```"LP``````````$@```````B<`````````
M`!(```````+=```````````2```````!]P``````````$@```````X<!@!VH
M````.!(```H```"^```````````6````````T0``````````$@```````24`
M`````````!(```````./`8`2P````!P2```*````(@``````````$@``````
M`F8``````````"(```````"P```````````2```````#EP&`(V@```",$@``
M"@```Z4!@"Z(```":!(```H```.[`8!`5````902```*```!2`&!_I@````3
M$0``$@```NH``````````!(```````/&`8(!$`````01```9````E0``````
M````$@```````Y`!@!+`````'!(```H```)"`8(!,````0@1```;````Y0&"
M`2`````$$0``&@```]`!@"#$````0!(```H```$3`8(".````"`1```;```!
M5P``````````$@```````O0``````````!(```````")```````````2````
M```!^```````````$@```````GX``````````"(```````(8```````````2
M```````#W0&"`10`````$```&@```^D!@@*4`````!```!L```$`````````
M```2```````!0P``````````$@```````$D``````````!(```````/O`8`C
M]````)P2```*````'```````````$@```````#D!@$HP`````!(```````/W
M`8`_Y````'`2```*```"``&`2D``````$@``````!``!@"$$````>!(```H`
M``''```````````2```````$#`&`(7P```'L$@``"@```P$``````````!(`
M``````$U```````````2````````=```````````$@``````!!D!@@$4````
M`!```!D```0@`8`=\````M02```*```!D@&`2I``````$@``````!"T!@$'H
M```!H!(```H```!E```````````2```````#"```````````$@```````8<`
M`````````!(```````,3```````````2```````#&@``````````$@``````
M`BP``````````!(```````/J`8("E``````0```;````C@``````````$@``
M`````&\``````````!(```````([```````````2```````#)@``````````
M$@``````!#<!@##P````A!(```H```(T```````````2````````[0``````
M````$@````!L:6)P97)L+G-O`')E86QL;V,`;65M;6]V90!R;61I<@!S=')C
M:'(`<W1R;F-M<`!S=')R8VAR`&)O;W1?1'EN84QO861E<@!P97)L7V-O;G-T
M<G5C=`!A8F]R=`!U;FQI;FL`7U]L<W1A=#4P`'!E<FQ?9G)E90!?9FEN:0!?
M7W-T870U,`!?97AI=`!M96UC<'D`<&5R;%]D97-T<G5C=`!03%]D;U]U;F1U
M;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R;%]A;&QO
M8P!C;&]S961I<@!E;G9I<F]N`%]?;W!E;F1I<C,P`&UE;7-E=`!097)L7W-A
M9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`4&5R;%]G=E]F971C:'!V
M`%]?<W!R:6YT9E]C:&L`<F5A9`!03%]M96UO<GE?=W)A<`!097)L7W-Y<U]I
M;FET,P!097)L7V=E=%]S=@!097)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VYE
M=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%]?97)R;F\`
M4&5R;%]M9U]S970`;6MD:7(`<&5R;%]P87)S90!097)L7VYE=UA37V9L86=S
M`&%C8V5S<P!G971P:60`=F9P<FEN=&8`4&5R;%]A=&9O<FM?;&]C:P!M96UC
M;7``<&5R;%]R=6X`;'-E96L`;W!E;@!S<')I;G1F`&=E='5I9`!S=')L96X`
M7U]S1@!S=')C;7``;&EB;2YS;RXP`&QI8F-R>7!T+G-O+C$`7U]D97)E9VES
M=&5R7V9R86UE7VEN9F\`7U]R96=I<W1E<E]F<F%M95]I;F9O`&QI8G!T:')E
M860N<V\N,0!P=&AR96%D7V%T9F]R:P!S=')D=7``;&EB8RYS;RXQ,@!?7W!R
M;V=N86UE`%]?<'-?<W1R:6YG<P!?7V=E='!W=6ED-3``7U]S>7-C86QL`%]?
M<W1P8W!Y7V-H:P!A=&5X:70`7VQI8F-?:6YI=`!S=')T;VL`7U]R96%D9&ER
M,S``7U]S=&%C:U]C:&M?9F%I;`!M86EN`&=E=%]U<V5R;F%M95]F<F]M7V=E
M='!W=6ED`'!A<E]C=7)R96YT7V5X96,`<&%R7W-E='5P7VQI8G!A=&@`<&%R
M7VUK=&UP9&ER`'AS7VEN:70`7U]S=&%R=`!P87)?96YV7V-L96%N`%A37TEN
M=&5R;F%L<U]005)?0D]/5`!S:&%?=7!D871E`&UY7W!A<E]P;`!P87)?8F%S
M96YA;64`7U]B<W-?<W1A<G0`7U]E;F0`<&%R7V1I90!S:&%?:6YI=`!P87)?
M9&ER;F%M90!P87)?:6YI=%]E;G8`7V5D871A`'!A<E]F:6YD<')O9P!S:&%?
M9FEN86P`<&%R7V-L96%N=7``+W5S<B]P:V<O;&EB+W!E<FPU+S4N-#`N,"]P
M;W=E<G!C+6YE=&)S9"UT:')E860M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`
M``&!_O@``#\!``````&!_PP``"T!``````&!__```"A)``````&!_I@``#(3
M``````&"`2@```03``````&"`3```#<3``````&"`C@``#H3``````&"````
M``(5``````&"``0```,5``````&"``@```45``````&"``P```85``````&"
M`!````<5``````&"`!0```@5``````&"`!@```D5``````&"`!P```L5````
M``&"`"````T5``````&"`"0```X5``````&"`"@```\5``````&"`"P``!`5
M``````&"`#```!$5``````&"`#0``!,5``````&"`#@``!05``````&"`#P`
M`!45``````&"`$```!85``````&"`$0``!@5``````&"`$@``!D5``````&"
M`$P``!H5``````&"`%```!L5``````&"`%0``!T5``````&"`%@``!X5````
M``&"`%P``"`5``````&"`&```",5``````&"`&0``"05``````&"`&@``"45
M``````&"`&P``"85``````&"`'```"D5``````&"`'0``"H5``````&"`'@`
M`"P5``````&"`'P``"T5``````&"`(```"X5``````&"`(0``#,5``````&"
M`(@``#45``````&"`(P``#L5``````&"`)```#P5``````&"`)0``#T5````
M``&"`)@``#X5``````&"`)P``#\5``````&"`*```$`5``````&"`*0``$,5
M``````&"`*@``$05``````&"`*P``$45``````&"`+```$<5``````&"`+0`
M`$@5``````&"`+@``$H5``````&"`+P``$P5``````&"`,```$X5``````&"
M`,0``$\5``````&"`,@``%`5``````&"`,P``%,5``````&"`-```%45````
M``&"`-0``%85``````&"`-@``%<5``````&"`-P``%@5``````&"`.```%D5
M``````&"`.0``%H5``````&"`.@``%P5``````&"`.P``%T5``````&"`/``
M`%X5``````&"`/0``%\5``````&"`/@``&$5``````&"`/P``&(5`````'P(
M`J:0`0`$E"'_\$@`,:4X(0`0@`$`!'P(`Z9.@``@0I\`!7UH`J8]JP`".:UN
M2'SC.WA]!$-X2``!#90A_^!\"`*F0I\`!9/!`!A_R`*F/]X``I.A`!0[WFO(
MD^$`'(.^@`2#_H``D`$`)'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?Z$!!
MG/_L@`$`)(.A`!2#P0`8?`@#IH/A`!PX(0`@3H``()0A_^!\"`*F0I\`!9/!
M`!A_R`*F/]X``I.!`!`[WFM<D^$`'(.>@`R#_H`(D`$`)'^?X$!`G`!(B3\`
M!Y.A`!0OB0#X0;X`%$@``%")/P`'+XD`^$">`$2!/P`(._\`#(.___1]*0.F
M3H`$(7^?X$"0?0``09S_V(.A`!2``0`D@X$`$(/!`!A\"`.F@^$`'#@A`"!.
M@``@2``SL90A_^!\"`*F0I\`!9.!`!!\G"-YD\$`&'_(`J8_W@`"D`$`)#O>
M:L"380`,DZ$`%)/A`!Q!@@$`@3P``(#^@!R!20``@WZ`("^*``"!'``(DX<`
M`)$;``!!G@#,@1Z`))%(``"!20``B2H``"^)``!!G@`@*XD`+SE*``%`GO_L
MD4@``(DJ```OB0``0)[_Z"^#``!!G@`(2``UY4@`-C&!/H`L+XD``$&>`+"#
M_H`0@[Z`,'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?Z$!!G/_L@'Z`-$@`
M-:6#_H`4@[Z`.'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?Z$!!G/_L@'Z`
M/$@`-75+__VQ@+L``("<``"`?``$2``OZ4@`-*V!/H`D@5Z`*)%)``!+__]@
M@-Z`&#C@`!,XH``".&```#B```1,QC&"2``T03A@``%(`#)Y2__^#4O__U`L
M`P``E"'_T)/A`"Q!@@$(/2`!@I-A`!R#:0$@+YL``$&>`/R3P0`HB2,``"^)
M``!!G@#<*XD`/3O```!`G@",@_L``"^?``!!G@"X?`@"II-!`!B0`0`T?)HC
M>).!`"!\?!MXDZ$`)']]VWA_Q?-X?X3C>'_C^WA(`#+E+X,``$">`!!]/_"N
M+XD`/4&>`%R'_0`$+Y\``$">_]2``0`T@T$`&(-A`!Q\"`.F@X$`((.A`"2#
MP0`H?^/[>(/A`"PX(0`P3H``('Q^&WB-/@`!+HD``"N)`#U!E@`(0+[_\'_#
M\%!+__]<?[OH4#O>``%_O19P?__R%).Z``!+__^@@V$`'(/!`"A+__^P.\``
M`$O__S`[X```2___H#O@``"#80`<2___E)0A_]!\"`*FDZ$`)#^@`8([O0(X
MD`$`-)/!`"B!/0``D2$`'#D@``"3X0`L/\`!@GQ_&W@X@0`8?^/[>$S&,8)+
M__Z9+X,``$&>`$R!00`8@1X!(%5*$#I]*%(4@.D`!"^'``!\Z%$N0;[_S(%)
M``@OB@``E4D`!$">__0X@0`8?^/[>$S&,8)+__Y1+X,``$">_[R!(0`<@5T`
M`'TI4GDY0```0((`((`!`#2#H0`D@\$`*'P(`Z:#X0`L."$`,$Z``"!(`#0!
ME"'_,).A`,1\?1MYD^$`S#_@`8([_P(X@3\``)$A`*PY(```08(`$(D]```O
MB0``0)X`*($A`*R!7P``?2E2>3E```!`@@%0@Z$`Q(/A`,PX(0#03H``('P(
M`J:3@0#`D`$`U$@`,[E\?!MY08(`X),A`+0_(`&`DT$`N#]``8"380"\/V`!
M@)/!`,@[>TT8.UI-'#LY32!_@^-X2``S$2P#``!!@@"X.\,`#7]DVWA_P_-X
M2``P.2^#``!!OO_<?T33>'_#\WA(`#`E+X,``$&>_\A_H^MXDP$`L$@`,Q%\
M>!MX?\/S>$@`,P5\>!H4.&,``D@`,%E_QO-X?'@;>'^EZWA_),MX3,8Q@D@`
M,K$X@0`0?P/#>$@`,"4O@___?P/#>$&>`!2!(0`852D$)B^)0`!!G@!82``O
M]7\#PWA(`#(-@P$`L$O__U"``0#4@X$`P'P(`Z9+__[<?X/C>$@`,#U_H^MX
M2``QA8`!`-2#(0"T@T$`N'P(`Z:#80"\@X$`P(/!`,A+__ZL2__^=4O__ZQ\
M"`*FDP$`L)`!`-23(0"TDT$`N)-A`+R3@0#`D\$`R$@`,DF4(?_@?`@"ICS`
M`8`\H`&`D^$`'#_@`8([___TD`$`)(/___P\@`&`?_\2%#C&3-2`?P``.*5*
M,#B$32A(`#&9@`$`)#S@`8"`?P``..=0%#R@`8`\@`&`@^$`''SF.W@XI2Z(
M.(1-1#D```!\"`.F."$`($@`+H"4(?^P?`@"II`!`%2380`\DX$`0'Q\&WB3
MH0!$D\$`2#_``8*3X0!,.]X".),A`#2!7@``D4$`'#E```"300`XB20``&DI
M`#TA*0``?^0!E'_C^WA(`#%I.($`&'Q[&WA_@^-X3,8Q@DO_^WE\?1MY08(!
M)$@`,4E_@]A`0)P`_(,A`!@_0`&"5SD0.HD\```OB0``09X!P"N)`#U!G@&X
M?X/C>$@```PKB0`]09X`$(TC``$OB0``0)[_\'Q\&%`[>P`"@[H!('Q[&A1(
M`"Y1+X,``'Q]R2Y!G@&8B3P``(%!`!B!&@$@+XD``%5*$#I]2%`NF2H``$"^
M`!A(```<C3P``2^)``"=*@`!09X`#"N)`#U`GO_L.2``/3O___^9*@``C3\`
M`2^)``"=*@`!0)[_]#A@``"!(0`<@5X``'TI4GDY0```0((!,(`!`%2#(0`T
M@T$`.'P(`Z:#80`\@X$`0(.A`$2#P0!(@^$`3#@A`%!.@``@.____SN]__^-
M/P`!+XD``)T]``%`GO_T2___I#]``8*3`0`PDN$`+(,:`2"!.```+XD``$&>
M`+!_",-X.4```$@```A_JNMXA2@`!#NJ``$OB0``0)[_\#E*``-51!`Z/4`!
M@H$J`H0OB0``0)X`4#D@``%\@R-XD2H"A$@`+2E\=QMY08(`;%>Y$#I_!,-X
M?R7+>$@`+2&2^@$@.3T``3E```!5*1`Z?5=)+I.A`!B"X0`L@P$`,$O__EQ_
M`\-X2``L!2^#``!\=QMXD'H!($&>`"!7N1`Z2___Q#A@``!+__YH.(``"#N@
M``!+__]T@N$`+(,!`#`X8/__2__^Q)+A`"R3`0`P2``O390A_T!\"`*FD\$`
MN#B!`!"3X0"\/^`!@I`!`,0[_P(X?'X;>($_``"1(0"L.2```$@`+HDL`P``
M0((`9($A`!A5*00F*XE``$&>`#QM*O__+XJ@`$&>`#"!(0"L@5\``'TI4GDY
M0```0((`/(`!`,2#P0"X@^$`O'P(`Z8X(0#`3H``('_#\W@X@``"2``K77QC
M`#148]E^2___P#A@``!+__^X2``NI90A^]!\"`*F?&0;>#B@!`"3P00H/\`!
M@I/A!"P[X0`<.]X".'_C^WB!/@``D2$$'#D@``"0`00T2``LV7T_&%`Y*?__
M?5]*%'^*^$!`G0`X?3](KB^)`"]`G@`L.2$`'3E```!]*1A0?2D#ID@``!")
M(___+XD`+T">``R=0___0@#_\#B``"]_X_MX2``KE2P#``!!@@!,?X/X0$&=
M`#A_X_MX2``KK8$A!!R!7@``?2E2>3E```!`@@!<@`$$-(/!!"B#X00L?`@#
MIC@A!#!.@``@.4```)E#``!+___$@2$$'(%>``!]*5)Y.4```$""`"2``00T
M/&`!@(/!!"@X8TT8@^$$+'P(`Z8X(00P2``K0$@`+8T](`&"E"'_\#DI__0\
MP`&`@2G__#R@`8!]*1(4/(`!@(!I```XQDS4.*5*,#B$32@X(0`02``LY)0A
M__`X8```."$`$$Z``""4(?L0?8``)CT@`8!\"`*FDT$$V#]``8*3@03@.UH"
M.)/A!.Q\?!MX?)\C>#AI35PX@0`<D`$$]),A!-2!.@``D2$$O#D@``"1@03`
M3,8Q@DO_]P5\>1MX.(``+W^#XWA(`"KQ+X,``$&>`%@\8`&`?X3C>#AC36A,
MQC&"2__Z^7^#XWB!(02\@5H``'TI4GDY0```0(("((`!!/2!@03`@R$$U'P(
M`Z:#0038?8"!((.!!."#X03L."$$\$Z``""3803</V`!@'_C^W@[>TUX2``J
M(7]DVWA(`"P)?'\;>4&"`<R2H03$+AD``)+!!,@ZH```DN$$S#[``8"3`030
M/N`!@#\``8"3P03HDZ$$Y#K631@Z]U!(.QA-?&*U@``[H```B3\``"^)``!!
MG@$,09(`&'_D^WA_(\MX2``H]2^#``!!G@#`?^/[>$@`*^4Y0___?'X;>'T_
M4A1_GTA`0)P`1'T_4*XOB0`O0)X`.'Q_&A0Y7P`!?2H84'TI`Z9(```0B4/_
M_R^*`"]`G@`,GZ/__T(`__!_X_MX2``KE7Q^&WA_@^-X2``KB7QC\A0[PP`!
M*YX#_T&=`&!_Y_MX?XGC>'[HNWA_!L-X.*`$`#B````X80"\3,8Q@D@`*J4X
M@0`@.&$`O$@`*)DO@P``0)X`%($A`"A5*00F?XFH`$&>`%Q_9-MX.&```$@`
M*M5\?QMY0(+_$(*A!,2"P03(@N$$S(,!!-"#803<@Z$$Y(/!!.A+__XX09(`
M&'[$LWA_(\MX2``G[2^#``!!GO^X.\```7[?LWA+__],.(```3AA`+Q(`"=]
M+X,``$">_Y@\8`&`.($`O#AC36A,QC&"2__X]3AA`+Q(`"AI@J$$Q(+!!,B"
MX03,@P$$T(-A!-R#H03D@\$$Z$O__=R#803<2__]O)*A!,22P03(DN$$S),!
M!-"3803<DZ$$Y)/!!.A(`"IQE"'_X'P(`J8X@``OD^$`''Q_&WB0`0`D2``G
MU2P#``!!@@`(.^,``8`!`"1_X_MX@^$`'#@A`"!\"`.F2``GX)0A_^"3X0`<
M?'\;>4&"`$")/P``+XD``$&>`#1\"`*FD`$`)$@`*?DX8P`!*X,$`$&=`#"`
M`0`D?^/[>(/A`!PX(0`@?`@#IDO_^T0\8`&`@^$`'#AC31@X(0`@2``G@(`!
M`"0X8```@^$`'#@A`"!\"`.F3H``()0A_\!\"`*F/&`!@)`!`$0X8TV$DV$`
M+#]@`8"3@0`P.WM-R).A`#0_H`&`D\$`.#_``8"3X0`\/^`!@CO_`C@[WDVL
M@3\``)$A`!PY(```3,8Q@DO_])T\8`&`.&--D$S&,8)+__2-/&`!@#AC39@[
MO4U<3,8Q@DO_]'D\8`&`.&--I#^``8!,QC&"2__T97_#\WA,QC&"2__T63Q@
M`8`X8TV\.YQ-U$S&,8)+__1%?Z/K>$S&,8)+__0Y?V/;>$S&,8)+__0M?X/C
M>$S&,8)+__0A/&`!@#AC3>!,QC&"2__T$3Q@`8`X8TUH3,8Q@DO_]`$\8`&`
M.($`&#AC3>Q,QC&"2__RQ7QD&WE!@@`0?X/C>$S&,8)+__;1/&`!@#B!`!@X
M8TX`3,8Q@DO_\IU\9!MY08(`%#Q@`8`X8TX43,8Q@DO_]J4\8`&`.($`&#AC
M3B!,QC&"2__R<7QD&WE!@@!<?Z/K>$S&,8)+__9]/(`!@'_#\W@XA$Y$3,8Q
M@DO_]FF!(0`<@5\``'TI4GDY0```0((`5(`!`$2#80`L@X$`,'P(`Z:#H0`T
M@\$`.(/A`#PX(0!`3H``(#Q@`8`X@0`8.&-.,$S&,8)+__']?&0;>4&B_YQ_
M8]MX3,8Q@DO_]@E+__^,2``GS90A_]!\"`*F/&`!@)/A`"P_X`&"D`$`-#O_
M`C@X8TW(.($`&($_``"1(0`<.2```$S&,8)+__&I+`,``#D@``!!@@`<B&,`
M`"^#``!!G@`0:&,`,#$C__]]*1D0@4$`'($?``!]2D)Y.0```'TC2WA`@@`8
M@`$`-(/A`"PX(0`P?`@#IDZ``"!(`"=!E"'_@'P(`J:0`0"$D($`%)"A`!B0
MP0`<D.$`()$!`"21(0`HD4$`+$"&`"38(0`PV$$`.-AA`$#8@0!(V*$`4-C!
M`%C8X0!@V0$`:#U``8(](`&"?&0;>#DI`C@X:@'@.,```3C@```Y`0"(.4$`
M$#BA`'"!:0``D6$`?#E@``"8P0!PF.$`<9$!`'2100!X2``DC3A@`/](`"4E
M/`#__GPL"WA@`/^P/2```7PA`6Y\"`*F82D`'#C@``*3#/_@DTS_Z#\``8*3
M;/_L.Q@".)/,__@XH```D`P`!#C```"2[/_<?'H;>),L_^23C/_PDZS_])/L
M__R!6```?4D)+CE```!(`",),23__WQC`=1\>_YP?3[:>'_;\!!7W@0^?][:
M>'_;\!!_>]D0?]Y($'][&1`OFP``09P!$#[@``$_@`&".Z$`%&+W``@[G`$`
M/R#__SC@``!_9=MX?\;S>']#TWA(`"*I+X/__T&>`/A^Y;MX?Z3K>']#TWA(
M`"21+X/__T&>`-`K@P`'0)T`I#AC__A__1H4?Y_H0$"\`!1(``"0.____W^?
MZ$!!G`"$.*``"'^$XWA_X_MX2``AL2^#``!`GO_@.($`%'R$^%!\@_YP?(3P
M%'QCV10](``!82D`''U)""Z!.```?4I*>3D@``!`@@"(@6$``(`+``2"Z__<
M@PO_X'P(`Z:#*__D@TO_Z(-K_^R#B__P@ZO_](/+__B#Z__\?6%;>$Z``"!_
MWL@4?WL!U"^;__]`GO\8?Y[(`$">_Q`X8/__.(#__TO__XP\8`&`.&-.6$S&
M,8)+__VA+X3__T">_P@\8`&`.&-.2$S&,8)+__V)2``DP90A__!\"`*FD`$`
M%$@`),%(`")]+`,``$&"`!B`8P``@`$`%#@A`!!\"`.F3H``(#A@``!+___L
ME"'_P'P(`J:3P0`X/\`!@).!`#`[WDYDDZ$`-'Q\&WB3X0`\/^`!@I`!`$0[
M_P(X?\/S>#B!`!B!/P``D2$`'#D@``!,QC&"2__N47Q]&WE!@@`0B3T``"^)
M``!`G@!(?X3C>'_#\WA,QC&"2__R38$A`!R!7P``?2E2>3E```!`@@"$@`$`
M1(.!`#"#H0`T?`@#IH/!`#B#X0`\."$`0$Z``"!_@^-XDV$`+$@`(\E\>QMX
M?Z/K>$@`([U\>QH4.&,``D@`(1$\P`&`/(`!@'Q[&WA_I^MX?X7C>#C&37@X
MA$U\3,8Q@D@`(UU_9-MX?\/S>$S&,8)+__'!@V$`+$O__W"380`L2``C?90A
M_K!\"`*FD\$!2#_``8`[WDS4DX$!0($>`!`_@`&"@5X`%#N<`CB!/@`8.($`
M'(#^`!R1`0`D@1P``)$!`1PY````D4$`*($>`""!7@`DD2$`.($^`"B380$\
M/V`!@(#>``P[>TU<D.$`/)$!`$"`_@`L@1X`,)%!`$21(0!(@5X`-($^`#B3
M00$X?'H;>']CVWB0`0%4D^$!3)#!`""0X0!,D0$`+)%!`#"1(0`T3,8Q@DO_
M[-E\?QMY08(`$(D_```OB0``0)X"H).A`422X0$LDP$!,),A`31+__WA?'T;
M>4&"`IA_H^MX2``B?51C"#PX8P`!2``?T8B]``!\>1MX+X4``$&>`#`_``&`
M?S_+>#L83GQ_X_MX?P3#>$S&,8)(`"(5C+T``3O_``(OA0``0)[_Y(!A`#@O
M@P``09X!4#OA`#A(```0A'\`!"^#``!!G@$\.($`'$S&,8)+_^PM?'T;>4&B
M_^2)/0``+XD``$&>_]A+__*Y+X,``$&>_\Q_H^MX2``?H7Q]&WE!@@$`?Z/K
M>#[@`8!(`"'-.O=02'Q_&WA_(\MX2``AO7Q_&A0[XP0$?^/[>$@`'PT\X`&`
M/(`!@'QX&WA_*,MX?Z7K>'[FNW@XYTZ$.(1.C$S&,8)(`"%5?P/#>#B``<!(
M`""I+X/__T&>`2@X@0!0?P/#>$@`(.5\?1MY0((`%($A`%A5*00F+XE``$&>
M`;0\@`&`/&`!@H"Z``!_!L-X.(1.U#AC`>!,QC&"2``?S8+A`2P[X```@P$!
M,(,A`32#H0%$@2$!'(%<``!]*5)Y.4```'_C^WA`@@0\@`$!5(-!`3B#80$\
M?`@#IH.!`4"#P0%(@^$!3#@A`5!.@``@.N```%;I$#HY00$@?2I*%(/I_P`O
MGP``09X`N(D_```OB0``09X`K%;X$#HY(0`@?^/[>'\)PA1+__%I.[<``2^#
M``!_M^MX0)X`-(?X``0OGP``09X`?(D_```OB0``09X`<'_C^W@[O0`!2__Q
M-7^WZW@O@P``09[_U'_C^WA(`!X9?'T;>4&"_WQ+__YX2``<F8$C```OB0`1
M09[^T(/Z``!(`!R%@.,``#R``8`\8`&"?^7[>'\&PW@XA$Z8.&,!X$S&,8)(
M`!ZQ2__^Y#N@``!+__XP2__[C7_C^WA(`!VY?'\;>$O__MP[(```5S\0.CDA
M`"P[___\?^GZ%(1_``0O@P``09X`;#B!`!P[.0`!3,8Q@DO_Z>5\:1MY0:+_
MX(DI```OB0``09[_U$@`'6E\?1MY0(+]($O__[2#(0!L2``?M7^9&`!`GOY$
M@2$`6)*A`215*07^DL$!*"^)`<!!G@`L@J$!)(+!`2A+__X@.&``#3^@`8!(
M`!R].[U.='QY&W@XH`!32__\\'_C^WA(`!RE.($`''Q_&W@\8`&`.&-/*$S&
M,8)+_^E1?&0;>4&"`6"`>@``2__Q[7QY&WE!@@%0?R/+>$@`'PTK@P`#?'8;
M>$"=`#@\@`&`.&/__#B$3S!\>1H42``;[2^#``!`G@`<*Y8`!$&>`=A^V;(4
MB3;_^R^)`"]!G@'(2__W$7QZ&WE`@@`<?R/+>#B```!,QC&"2``<@7QY&WE`
M@@#L/(`!@#Q@`8`XA$Y$.&--R#_``8!,QC&"2__LT3O>4!1(`!MU/(`!@'QG
M&WA_R/-X.(1/7'[FNWA_!<-X?^/[>#]``8!,QC&"2``>+3M:3VQ(``!`2``:
MH8$C```OB0`10)X`1$@`&S$[O0`!?&<;>'^HZWA_R?-X?N:[>'\%PWA_1--X
M?^/[>$S&,8)(`!WI.(`!P'_C^WA(`!T]+X/__T&>_[1_`\-X2``=77_D^WA_
M8]MX3,8Q@DO_[#%_X_MX2__Y<8*A`22"P0$H@N$!+(,!`3"#(0$T@Z$!1$O_
M_+2#.@``2__^L$O_]RU\=1MY?)8C>$&@_PPPQ/_V..```'RU`=1_(\MX2``:
M?3B!`.@XH``&?R/+>$@`'&TO@P`&0)X`W($A`.B!7@`\?XE0`$">_M"A(0#L
MH5X`0'^)4`!`GO[`,-;_SCC@``!\M0'4?R/+>$@`&C%_(\MX.($`\#B@`"A(
M`!PA+X,`*$">`)`]``&`/(`!@#CA`/!^YKMX.0A0%#B$3TQ_!<-X?^/[>)M!
M`1A,QC&"2``<W4O__PB#6@`$+YH``$&^_C1_0]-X2``<]2N#``-`O?XD.&/_
M_#R``8!\>AH4.(1/.$@`&=DO@P``0+[^"']9TWA+__X`DJ$!))+!`2B2X0$L
MDP$!,),A`323H0%$2``<O3Q@`8`X8T]`3,8Q@DO_]7&4(?_`?`@"ICQ@`8"3
MX0`\/^`!@CO___23H0`TD`$`1($___R380`L?2D2%).!`#`_H`&"D\$`.#N]
M`CB!70``D4$`'#E````X@0`8@2D``#AC3:R)20!DF4D`94S&,8)+_^9=+`,`
M`$&"`!")(P``+XD`,4&>`6Q+__)]@]___#R``8!_WA(4.(1/@(!^```XP``#
M.*```4@`&A%\?!MY08(`,($\``P\H`&`@'X``#BE3X2`B0``2``:(8$\``R`
MB0``@20`"'4I`$!`@@%L@]___#R``8!_WA(4.(1/C(!^```XP``#.*```4@`
M&;U\>QMY08(`,#Q@`8`X@0`8.&--:$S&,8)+_^6U?&4;>4&"`02!.P`,@'X`
M`(")``!(`!FY/X`!@CN<`H2#___\/(`!@'__$A0XA$^0@]\``#B@``%_P_-X
M2``7\3R@`8!\9!MX.*5/J'_#\WA(`!E]@3\``#E````\8`&`F4D`93B!`!@X
M8TU<3,8Q@DO_Y4$O@P``D'P`"$&>`&@X@`'`2``:.2^#`!%!G@`0.2,``2N)
M``%!G0"@@2$`'(%=``!]*5)Y.4```$""`*R``0!$@V$`+(.!`#!\"`.F@Z$`
M-(/!`#B#X0`\."$`0$Z``"")(P`!+XD``$&^_I1+__Z,@'P`!$O_]STO@P``
MD'P`"$">_XP\8`&`.&-/L$S&,8)(`!=9/X`!@H%;``P[G`*$@'X``($\``2`
MB@``@*D``$@`&*E+__[X@'X``$@`%XU+__Z0@5P`!#T@`8"`O``(?&8;>("*
M```X:4_43,8Q@D@`%PE(`!I%E"'_X'P(`J:3X0`<?'\;>)`!`"1+__)E+X,`
M`$&>`!@OGP``09X`$(D_```OB0``0)X`&(`!`"2#X0`<."$`('P(`Z9.@``@
M?^/[>$O_[\E+_^^%/(`!@#B$3H0XH``$2``742^#``!`GO_,@`$`)'_C^WB#
MX0`<."$`('P(`Z9+_^7$E"'^@#R``8(XA`(X..``$)-!`6@Y00`8DV$!;#D#
M`!B3@0%P?.D#II.A`723P0%XD^$!?($D``"1(0%<.2```'U)4WB$Z``$E.D`
M!$(`__@XH0$8?4A3>(3J``2!*``X@,@`)'^%4$"!"``,?2DR>'TI0GA]*3IX
M52D(/I$J`$!`GO_4@2$`'(`#``B`PP`,/2E:@H%C``0Y*7F9@8,``'P?,GB`
MHP`057WP/H%!`"!5GB@^?_]8.'P(ZG@\ZEJ"?2DJ%(%!`"1__S)X?2GR%'T(
M8#A5GO`^..=YF7TI^A1\YS(4/^I:@GT(`GA_JO)X?0@Z%'U*2#B`X0`H._]Y
MF54\*#Y]2NIX52GP/G__`A1]".(4/V=:@GU*^A1_QTIX@^$`+%4<*#Y\YT`X
M.WMYF54(\#Y]2N(4?WOJ%'SG\GA]/4)X/Y]:@GSGVA157R@^?[U0.#N<>9E\
MY_H4?YSR%(/A`#!_O4IX54KP/G^]XA14_"@^?[WB%'T>4GB#@0`T/_]:@CO_
M>9E_WC@X5.?P/GT_2A1_WD)X?5\Z>#^<6H)_WDH4.YQYF5>I*#Y__^@X?]Y*
M%'T<0A2!(0`X?_]2>%>]\#Y__T(45\@H/G__0A0_B5J"@0$`/'SIZGA]*?`X
M.YQYF5?>\#Y_G%(4/0A:@GTJ.GA_J?)X5_LH/GU*XA0Y"'F9@X$`0'TI^#A]
M2MH45__P/GSH.A1]*>IX?\CZ>#]\6H)]*3H455PH/GTIXA0[>WF9?0A0.(.!
M`$151_`^?[OJ%'T(\GA_ZCIX?0CJ%#^<6H)5/2@^?4I(.'T(ZA0[G'F9@Z$`
M2%4I\#Y_G/(4?4KZ>'S^2G@_?5J"?4KB%%4=*#Y]2NH4?]Y`.(.A`$P[>WF9
M50CP/G][^A1_WCIX?3]">#^=6H)_WMH455TH/G__4#A_WNH4.YQYF8.A`%!5
M2O`^?YPZ%'__2GA]!U)X/WU:@G__XA17W2@^?__J%'SG\#B#H0!4.WMYF7][
M2A1\Z4)X5][P/C^=6H)]*=H45_TH/GTIZA1]1_)X@Z$`6'SG^#@[G'F95__P
M/G^<0A1\YU)X?\CZ>#]=6H)]"$@X53TH/GSGXA0[6GF9?.?J%(.!`%Q_6E(4
M?0KR>%4I\#Y]2M(45.@H/G_]2GA]"D(4/WQ:@H%!`&!_O3@X.WMYF53G\#Y_
M>_(4/XI:@G^^^GB!00!D?3TZ>'_>VA1_O4`X51LH/CN<>9E5"/`^?][:%'^<
M^A0]2EJ"?[]*>'S]0GA7VR@^?__B%#E*>9E_O?`X@X$`:%?>\#Y__]H4?2I*
M%'^].GA]"O)X?[U*%#^<6H)7Z2@^?[U*%'U*^#B!(0!L.YQYF7^<.A1]1T)X
M5__P/CU);MI\Y^(45[PH/GSGXA2!(0!P?]SZ>#E*ZZ%_G.IX?4I"%%>]\#X]
M"6[:?4KB%%3\*#Y]2N(4@2$`='_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<
M*#X\Z6[:?0CB%($A`'A_O/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4EN
MVGSGXA2!(0!\?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4
M@2$`@'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`(1_
MO/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"(?]SZ>#E*
MZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4@2$`C'_\ZG@Y".NA?YPZ
M>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`)!_O/)X..?KH7^<4GA\Y_H4
M?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"4?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15
M'?`^5/PH/CT);MI]2N(4@2$`F'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<
M*#X\Z6[:?0CB%($A`)Q_O/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4EN
MVGSGXA2!(0"@?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4
M@2$`I'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`*A_
MO/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"L?]SZ>#E*
MZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4@2$`L'_\ZG@Y".NA?YPZ
M>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`+1_O/)X..?KH7^<4GA\Y_H4
M?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"X?]SZ>#E*ZZ%]2NH4?YU">(.!`+Q5
M"/`^?4KJ%#TI;MI4_2@^?_M">'U*ZA0Y*>NA5/WP/C^<CQQ_9SIX?2GR%#N<
MO-Q]'NMX?YSZ%'TI.A1]'^@X54<H/G_>4#A]*3H4?][[>(#A`,!52O`^53LH
M/G_>XA1_OU-X?][:%#SGCQR#80#$?[Q0.#CGO-Q__T@X?0="%%4I\#Y__^-X
M/YN/''U'2WA__T(4.YR\W%?(*#Y__T(4?YSJ%'SG\#A]74@X@0$`R%?>\#Y\
MY^MX5_LH/GT]\WA\Y^(4/0B/''SGVA1]//`X@V$`S#D(O-Q_O?@X?4A2%'^]
MXWA7__`^/YN/''^]4A14ZB@^?[U2%'_(^WB!00#0.YR\W'T\2A1]"#@X?]SX
M.%3G\#Y]".-X/XJ/'#N<O-Q_ZCMX5[LH/G^<\A1]"$H4?_XX.%>I\#Y]2N@X
M@Z$`U'T(VA1]2O-X?/Y+>#]]CQQ]2N(451TH/CM[O-Q]2NH4@Z$`V'_>0#A_
M>_H4?/](.%4(\#Y_W_MX/YV/''T^0WA572@^?__:%#N<O-Q__^H4?]Y0.(.A
M`-Q_G#H4?2=`.%5*\#Y_QSMX/WV/''SGXA17_2@^?.?J%'T>4WB#H0#@.WN\
MW'_>^#A_>TH4?0E0.%?_\#Y_R4MX/YV/''U>^WA4_2@^?2G:%#N<O-Q]*>H4
M?]XX.(.A`.1_G$(4?4CX.%3G\#Y_R$-X/WV/''T(XA15/2@^?0CJ%'_^.WB#
MH0#H.WN\W'_>2#A_>U(4?^HX.%4I\#Y_RE-X/YV/''S^2WA5'2@^?4K:%#N<
MO-Q]2NH4?YSZ%(.A`.Q\_T@X?]Y`.%4(\#Y_WOMX/WV/''_>XA1572@^?][J
M%'T_0WB#H0#P.WN\W'][.A1__U`X?2=`.%5*\#Y__SMX/YV/''T'4WA7W2@^
M?__:%#N<O-Q__^H4?.?P.(.A`/1_G$H4?0E0.%?>\#Y\Z4MX/WV/''TIXA17
M_2@^?2GJ%'U'\WB#H0#X.WN\W'][0A1\Y_@X?4CP.%?_\#Y\YT-X/YV/''_(
M^WA5/2@^?.?:%#N<O-Q\Y^H4?YQ2%'T(2#A_RO@X@Z$`_%4I\#Y]"%-X5/LH
M/G_J2WA]".(4/[V/''T(VA1__$@X@V$!`#N]O-Q]2C@X?]WR%'U*XWA4Y_`^
M/YN/''U*\A15'B@^?4KR%'T].WB#P0$$.YR\W'_\^A1_O4`X?3PX.%4(\#Y_
MO>-X/YZ/''S^0WA_O?H455LH/E5?\#X[G+S<?]Y0.(%!`0A]/$H4?/Q`.'^]
MVA1_WN-X/XJ/''_>2A1]"OMX5ZDH/CN<O-Q_WDH4?4KH.'^<.A2!(0$,?0?X
M.%>]\#Y]1SMX/4G*8WSGXA17W"@^?.?B%($A`1!__.IX.4K!UG^<\GA]2D(4
M5][P/CT)RF-]2N(45/PH/GU*XA2!(0$4?[SR>#D(P=9_G#IX?0CZ%'T(XA14
M__`^55PH/CSIRF-]".(4@2$!&'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<
M*#X]2<IC?.?B%($A`1Q__.IX.4K!UG^<0GA]2O(4?4KB%%4>\#Y4_"@^/0G*
M8WU*XA2!(0$@?[SR>#D(P=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4
M@2$!)'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<*#X]2<IC?.?B%($A`2A_
M_.IX.4K!UG^<0GA]2O(4?4KB%%4>\#Y4_"@^/0G*8WU*XA2!(0$L?[SR>#D(
MP=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4@2$!,'_<^G@XY\'6?YQ2
M>'SGZA1\Y^(455WP/E4<*#X]2<IC?.?B%($A`31__.IX.4K!UG^<0GA]2O(4
M?4KB%%4>\#Y4_"@^/0G*8WU*XA2!(0$X?[SR>#D(P=9_G#IX?0CZ%'T(XA14
M__`^55PH/CSIRF-]".(4@2$!/'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<
M*#X]2<IC?.?B%($A`4!__.IX.4K!UG^<0GA]2O(4?4KB%%4>\#Y4_"@^/0G*
M8WU*XA2!(0%$?[SR>#D(P=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4
M@2$!2'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<*#X]2<IC@2$!3'SGXA0Y
M2L'6?_SJ>'^<0GA]2O(451[P/CT)RF.!(0%0?4KB%%3\*#XY",'6/2G*8WU*
MXA0Y*<'6?[SR>'^<.GA]"/H4?2GJ%(.A`514__`^?0CB%%5'*#Y_W/IX?0@Z
M%#^]RF.`X0%8?YQ2>#N]P=952O`^?2GB%'_=\A15'"@^?_U2>#SGRF-]*>(4
M?[U">#CGP=95"/`^?][J%%4\*#Y]74)X?.=B%'^]2GA_GN(4?.?Z%'SGZA17
MGB@^?.?R%%4I\#Z0XP``?09"%'U%4A1]:^(4?2D"%(#A`5R`Q```?.<R>3C`
M``"10P`0D0,`#)%C``21(P`(0((`)(-!`6B#80%L@X$!<(.A`72#P0%X@^$!
M?#@A`8!.@``@?`@"II`!`81(``M1E"'_\'P(`J8X8`!@D`$`%$@`"(T\@&=%
M/*#OS3S`F+H\X!`R/0##TCE```!@A",!8*6KB6#&W/Y@YU1V80CA\)"#``"0
MHP`$D,,`")#C``R1`P`0D4,`%)%#`!B10P!<@`$`%#@A`!!\"`.F3H``()0A
M_]!\"`*F5*D8.%2H'WZ300`8?)HC>)-A`!R3@0`@?+PK>)/A`"Q\?QMXD`$`
M-#M_`!R3P0`H@4,`%(!C`%Q]*5(4@/\`&'U*2!`O@P``?4I1$)$_`!1\ZCA0
M?0@Z%)$?`!A!G@!<(2,`0'R^*WA_A4@`09T`X'Q[&A1_Q?-X?T33>$@`"BF!
M/P!<?3Y*%"^)`$"1/P!<09X`Y(`!`#2#00`8@V$`''P(`Z:#@0`@@\$`*(/A
M`"PX(0`P3H``().A`"0OG``_?YWC>']>TWA`G0!(?\3S>#B@`$!_8]MX.[W_
MP$@`"<E_X_MX2__P-2^=`#\[W@!`09W_W#N<_\!7B=&^'4G_P#DI``%5*3`R
M?XKB%']:2A1_1--X?V/;>'^%XWA(``F)DY\`7(`!`#2#H0`D@T$`&'P(`Z:#
M80`<@X$`((/!`"B#X0`L."$`,$Z``"!]/DMX?'L:%'_%\WA_1--X2``)28$_
M`%Q]/DH4+XD`0)$_`%Q`GO\D?^/[>).A`"1_GN!0?UKR%$O_[Y5+__\TE"'_
MT'P(`J8Y0/^`DV$`'#MD`!R3P0`H?'X;>)/A`"Q\GR-XD`$`-).!`""3H0`D
M@Z0`%(.$`!A7J>Z^.*D``7U;2:XOA0`X?'LJ%$"=`40@I0!`.(```$@`!@%_
MX_MX2__O+7]CVW@XH``X.(```$@`!>E7J<(^5X5&/E>&A#Y7A\(^5ZA&/E>J
MA#Z;GP!7?^/[>)N_`%N9/P!:F+\`5)C?`%68_P!6F1\`6)E?`%E+_^[=B3\`
M`'_C^WB9/@``H3\``)D^``&!/P``52G"/ID^``*!/P``F3X``XD_``29/@`$
MH3\`!)D^``6!/P`$52G"/ID^``:!/P`$F3X`!XD_``B9/@`(H3\`")D^``F!
M/P`(52G"/ID^``J!/P`(F3X`"XD_``R9/@`,H3\`#)D^``V!/P`,52G"/ID^
M``Z!/P`,F3X`#XD_`!"9/@`0H3\`$)D^`!&!/P`052G"/ID^`!*!/P`0F3X`
M$X`!`#2#80`<@X$`('P(`Z:#H0`D@\$`*(/A`"PX(0`P2``'#""E`#@X@```
M2``$P4O__MB4(?_@?`@"ID*?``63P0`8?\@"IC_>``*0`0`D.]X[7(%>@`2)
M*@`D+XD``$">`&R3@0`0.2```9/A`!R#GH`8@_Z`%)DJ`"1_G^!`0)P`,).A
M`!0[___\@3\`!#N_``0[_P`(?2D#IDZ`!"%_G^!`?[_K>$&<_^2#H0`4@3Z`
M'"^)``!!G@`H@'Z`#$@`!3V#@0`0@^$`'(`!`"2#P0`8."$`('P(`Z9.@``@
M@`$`)(.!`!"#X0`<?`@#IH/!`!@X(0`@3H``()0A_^!\"`*F0I\`!9/!`!A_
MR`*F/]X``I`!`"0[WCJ4@)Z`!(DD```OB0``0)X`6($^@`@Y0``!DZ$`%"^)
M``"3X0`<F40``$&>`!"`?H`,.(0`!$@`!2V#_H``@[Z`$'^?Z$!`G0`8A3__
M_'TI`Z9.@`0A?Y_H0$&=__"#H0`4@^$`'(`!`"2#P0`8."$`('P(`Z9.@``@
ME"'_P'P(`J:3P0`X/\`!@I!A`!P[W@(XD($`&#AA`!R0H0`4.($`&#BA`!2!
M/@``D2$`+#D@``"0`0!$D^$`/$@`!&D\H`&`/&`!@#BE2I`X8TI`?*0K>$@`
M`O$](`&"B^D!*"^?``!!G@%8/2`!@I.A`#0Y*?_T@Z$`'($I__Q]*1(4/4`_
M_X$)```X?0`$84K__SN]``-_@U!`B2@#UF$I``.9*`/609T!H%1C$#H_X`&"
M2``$58#A`!R!00`8/2`!@B^'``&`R0$0@*H``#T``8`](`&`.0A0&#DI4!P[
M_P*$?&L;>)"C``"1`P`$D,,`")$C``R0?P`$0)T!1%3I$#HY`P`,.2G_^%4I
M\+XY*0`!?2D#IH4J``25*``$0@#_^#CG``-4Z1`Z/(`!@(!_``PY0```?Z7K
M>#B$&.Q]9EMX..```'U+22Y(``*-+X,``$&>`+B`?P`,2``#/3B!`"A\?1MX
M/&`!@#AC3;Q,QC&"2__.Z2^#``!!G@!\@'\`#$@`!*5(``$1@2$`+(%>``!]
M*5)Y.4```'^CZWA`@@#`@`$`1(.A`#2#P0`X?`@#IH/A`#PX(0!`3H``($@`
M`GD]0`&"+X,``)!J`I"3H0`T09X`D$@``U$](`&".2G_](%)__Q]2A(4@4H`
M`)OJ!GY+__Z$@'\`""^#``!!G@`82__J!4O__WB`?P`,2``"Z4O__T0\8`&`
M.($`*#AC35Q,QC&"2__.-2^#``"0?P`(09[_3$O_Z=%+__]$.2``$$O__N@\
M@`&"/&`!@#B$_I@X8T]83,8Q@D@``*U(``/I.&```4@``F$`````````````
M```]8`&"@6L``'UI`Z9.@`0@/6`!@H%K``1]:0.F3H`$(#U@`8*!:P`(?6D#
MIDZ`!"`]8`&"@6L`#'UI`Z9.@`0@/6`!@H%K`!!]:0.F3H`$(#U@`8*!:P`4
M?6D#IDZ`!"`]8`&"@6L`&'UI`Z9.@`0@/6`!@H%K`!Q]:0.F3H`$(#U@`8*!
M:P`@?6D#IDZ`!"`]8`&"@6L`)'UI`Z9.@`0@/6`!@H%K`"A]:0.F3H`$(#U@
M`8*!:P`L?6D#IDZ`!"`]8`&"@6L`,'UI`Z9.@`0@/6`!@H%K`#1]:0.F3H`$
M(#U@`8*!:P`X?6D#IDZ`!"`]8`&"@6L`/'UI`Z9.@`0@/6`!@H%K`$!]:0.F
M3H`$(#U@`8*!:P!$?6D#IDZ`!"`]8`&"@6L`2'UI`Z9.@`0@/6`!@H%K`$Q]
M:0.F3H`$(#U@`8*!:P!0?6D#IDZ`!"`]8`&"@6L`5'UI`Z9.@`0@/6`!@H%K
M`%A]:0.F3H`$(#U@`8*!:P!<?6D#IDZ`!"`]8`&"@6L`8'UI`Z9.@`0@/6`!
M@H%K`&1]:0.F3H`$(#U@`8*!:P!H?6D#IDZ`!"`]8`&"@6L`;'UI`Z9.@`0@
M/6`!@H%K`'!]:0.F3H`$(#U@`8*!:P!T?6D#IDZ`!"`]8`&"@6L`>'UI`Z9.
M@`0@/6`!@H%K`'Q]:0.F3H`$(#U@`8*!:P"`?6D#IDZ`!"`]8`&"@6L`A'UI
M`Z9.@`0@/6`!@H%K`(A]:0.F3H`$(#U@`8*!:P",?6D#IDZ`!"`]8`&"@6L`
MD'UI`Z9.@`0@/6`!@H%K`)1]:0.F3H`$(#U@`8*!:P"8?6D#IDZ`!"`]8`&"
M@6L`G'UI`Z9.@`0@/6`!@H%K`*!]:0.F3H`$(#U@`8*!:P"D?6D#IDZ`!"`]
M8`&"@6L`J'UI`Z9.@`0@/6`!@H%K`*Q]:0.F3H`$(#U@`8*!:P"P?6D#IDZ`
M!"`]8`&"@6L`M'UI`Z9.@`0@/6`!@H%K`+A]:0.F3H`$(#U@`8*!:P"\?6D#
MIDZ`!"`]8`&"@6L`P'UI`Z9.@`0@/6`!@H%K`,1]:0.F3H`$(#U@`8*!:P#(
M?6D#IDZ`!"`]8`&"@6L`S'UI`Z9.@`0@/6`!@H%K`-!]:0.F3H`$(#U@`8*!
M:P#4?6D#IDZ`!"`]8`&"@6L`V'UI`Z9.@`0@/6`!@H%K`-Q]:0.F3H`$(#U@
M`8*!:P#@?6D#IDZ`!"`]8`&"@6L`Y'UI`Z9.@`0@/6`!@H%K`.A]:0.F3H`$
M(#U@`8*!:P#L?6D#IDZ`!"`]8`&"@6L`\'UI`Z9.@`0@/6`!@H%K`/1]:0.F
M3H`$(#U@`8*!:P#X?6D#IDZ`!"`]8`&"@6L`_'UI`Z9.@`0@2``!`$@``/Q(
M``#X2```]$@``/!(``#L2```Z$@``.1(``#@2```W$@``-A(``#42```T$@`
M`,Q(``#(2```Q$@``,!(``"\2```N$@``+1(``"P2```K$@``*A(``"D2```
MH$@``)Q(``"82```E$@``)!(``",2```B$@``(1(``"`2```?$@``'A(``!T
M2```<$@``&Q(``!H2```9$@``&!(``!<2```6$@``%1(``!02```3$@``$A(
M``!$2```0$@``#Q(```X2```-$@``#!(```L2```*$@``"1@````8````&``
M``!@````8````&````!@````8````#V``8(]:_Z`@`S_^#EKM*!\"0.F?`M:
M%(&,__Q]8%H43H`$(&````!@````8````&````!@````8````&````!\"`*F
MD`$`!)0A__!+__;=."$`$(`!``1\"`.F3H``('!S7W-T<FEN9W,@;6ES<VEN
M9PH`<&5R;'AS:2YC`````8!00`&`31@``````8!.%`&`3A@!@%`T`8!-8`&`
M4#P``````8!0(`&`4"@``````$-!0TA%```N````+BX``"5S+R5S````1'EN
M84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4
M`````%!!4E]414U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S``!0
M15),-4Q)0@````!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.
M251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P``
M`%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]4
M35!$25(```!005)?5$U01$E2``!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!
M3%]#3$5!3@`````Q````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D`$Q$
M7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````<&%R+0`````E<R5S)7,E
M<P`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@
M9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@
M)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R
M(&]P97)A=&EO;BD*``!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R
M96%D```E<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP
M+25U+25U)7,`````&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)3
M24].`````#$N,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O
M<GD`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T
M;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`````M90``+2T``%5315(`````
M55-%4DY!344`````5$5-4$1)4@!435``+W9A<B]T;7`O````979A;"`G97AE
M8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T
M(')U;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM
M>2`H)%!!4E]-04=)0RP@)$9)3$5?;V9F<V5T7W-I>F4L("1C86-H95]N86UE
M7W-I>F4I.PI"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["B1&
M24Q%7V]F9G-E=%]S:7IE(#T@-#L@("`C('!A8VLH(DXB*0HD8V%C:&5?;F%M
M95]S:7IE(#T@-#`["B102UI)4%]-04=)0R`](")02UPP,#-<,#`T(CL*?0IS
M=6(@9FEN9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]($!?.PIM>2`D8VAU;FM?
M<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D9F@[
M"FUY("1P;W,@/2`H)'-I>F4M,2D@+2`H)'-I>F4M,2D@)2`D8VAU;FM?<VEZ
M93L@("`@("`C($Y/5$4Z("1P;W,@:7,@82!M=6QT:7!L92!O9B`D8VAU;FM?
M<VEZ90IW:&EL92`H)'!O<R`^/2`P*2!["G-E96L@)&9H+"`D<&]S+"`P.PIR
M96%D("1F:"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L96YG=&@H)%!!4E]-04=)
M0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X]
M(#`I('L*<F5T=7)N("1P;W,@*R`D:3L*?0HD<&]S("T]("1C:'5N:U]S:7IE
M.PI]"G)E='5R;B`M,3L*?0IM>2`H)'!A<E]T96UP+"`D<')O9VYA;64L($!T
M;7!F:6QE+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%
M04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E
M;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ
M0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN
M9R!F:6QE<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P
M=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP
M*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N
M92`G35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U0
M3$%412`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I
M<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P
M+`HI.PIM>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M
M<"`\/"(N+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M
M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N
M"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D
M9FEL96YA;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA
M;64["FUY("1C;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*
M<WES=&5M*"1C;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P<F]C
M97-S('1O('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]"D)%
M1TE.('L*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E
M<FYA;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@
M)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@)7-Y<R`]("@*<&%T:%]S
M97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@
M("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G
M+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/
M4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O9VYA;64H*3L*;W5T<RAQ
M<5M<)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*7W-E=%]P87)?=&5M<"@I
M.PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D14Y6>U!!4E]414U0?2)=
M*3L*;7D@*"1S=&%R=%]P;W,L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI.PI-
M04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL97-S
M("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;0V%N
M)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E
M("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)0SL*
M?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H*E]&2"D["FEF("@D
M;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T(&9I;F0@;6%G:6,@<W1R
M:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@
M34%'24,["GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC(&%T('!O<VET:6]N("1M
M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S
M971?<VEZ92`M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIR96%D(%]&2"P@)&)U
M9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&YE(")<,$-!0TA%(B`F
M)B`D8G5F(&YE(")<,$-,14%.(BD@>PIO=71S*").;R!C86-H92!M87)K97(@
M9F]U;F0B*3L*;&%S="!-04=)0SL*?0IO=71S*'%Q6T-A8VAE(&UA<FME<B`B
M)&)U9B(@9F]U;F1=*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O
M9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L("1&24Q%7V]F9G-E=%]S
M:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@)&)U9BD["F]U=',H(D]F
M9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O9F9S970B*3L*<V5E:R!?
M1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92`M("1O9F9S970L
M(#`["B1S=&%R=%]O9E]&24Q%7W-E8W1I;VX@/2!T96QL(%]&2#L*;7D@)7)E
M<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B
M=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@
M)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68[
M"F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@
M)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@
M)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M
M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D
M97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@
M)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D[
M"B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M
M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN
M86UE.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F
M(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*
M)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L
M(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I
M(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@
M)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@
M)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M
M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!
M4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F
M:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T
M92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H
M)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]
M"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]
M+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M
M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;
M0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I
M92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M
M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@
M4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O
M<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*
M<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T
M(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE
M;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA
M;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E
M('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T
M;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA
M;65].PI]"FEF("@D8G5F(&YE("102UI)4%]-04=)0RD@>PIO=71S*'%Q6TYO
M('II<"!F;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@:6X@9FEL92`B)'!R;V=N
M86UE(ETI.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?<&]S(#T@*'1E;&P@7T9(
M*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@;V8@>FEP"GT*;7D@0'!A
M<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO9V9H+"`D8V%C:&5?;F%M
M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5].R`C('-A;FET:7IE("A2
M155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M*0II9B`H(21S=&%R=%]P
M;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI
M('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS
M=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V
M97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U
M<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI
M9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY
M($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM
M*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=)
M)RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG
M*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H
M:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T
M(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G
M.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L
M<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@
M)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@
M;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@
M>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C
M;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=]
M.PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]
M*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?
M=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM
M>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D
M<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#
M86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)
M3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^
M<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI
M=F4@(B1P87(B73L*?0IM>2`E;65T85]P87(@/2!D;R!["FEF("@D>FEP(&%N
M9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M
M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*
M)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I
M;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE
M('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68[
M"G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A
M<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D4$M:25!?34%'24,["F-L;W-E("1P
M:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U)$
M5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q
M6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H
M*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1S
M=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*<F5A9"!?1D@L("1L;V%D97(L("1S
M=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@96QS92!["FQO8V%L("0O(#T@=6YD
M968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%3E9[4$%27U9%4D)!5$E-
M?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!0
M05(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^
M87!P;'DH7"1L;V%D97(L("0P*3L*?0HD9F@M/G!R:6YT*"1L;V%D97(I"F]R
M(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO861E<B!T;R`B)&]U="(Z("0A73L*
M:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y
M.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D
M=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH
M)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[
M87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB
M97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,[
M"FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?
M<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#
M;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT
M('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R
M="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN
M960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L*
M*"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C
M:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N
M*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]
M"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM
M>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]
M("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I
M;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN
M97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N
M86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<
M421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@
M(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI
M.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D
M9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD968["F]P96X@
M;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B
M)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1T:#X["F-L;W-E("1T:#L*4$%2
M.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D
M960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@
M/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT
M96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD
M9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y
M*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-2
M0S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT
M96YT*2DL"B1C;VYT96YT*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!E;6)E
M9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=.PIO=71S*'%Q6U=R:71T96X@87,@
M(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I('L*)'II<"T^=W)I=&54;T9I;&5(
M86YD;&4H)&9H*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+"F]R(&1I92!Q<5M%
M<G)O<B!W<FET:6YG('II<"!P87)T(&]F("(D;W5T(ET["GT*:68@*"1M971A
M7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN="@B7#!#3$5!3B(I.PI]"F5L<V4@
M>PII9B`H(61E9FEN960@)&-A8VAE7VYA;64I('L*;7D@)&-T>"`]($1I9V5S
M=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H+"`P+"`P.PHD8W1X+3YA9&1F:6QE
M*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD8V%C:&5?;F%M92`]("1C='@M/FAE
M>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@
M,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@D
M8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H("1C86-H95]N86UE*3L*)&9H+3YP
M<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#2$4B*3L*?0IM>2`D;V9F<V5T(#T@
M)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I.PHD9F@M/G!R:6YT*'!A8VLH
M)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)0RD["B1F:"T^8VQO<V4@;W(@9&EE
M('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B)&]U="(Z("0A73L*8VAM
M;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@
M)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2
M.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ
M1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I
M;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV
M93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH
M9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@
M)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D
M>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP+3YR96%D1G)O;49I;&5(
M86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH
M*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P<F]G
M;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@V-"`J(#$P,C0I
M.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD4$%2.CI,:6)#86-H97LD
M<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT<F%C=&EN9R!Z:7`N+BXB*3L*
M:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@(R!S:&]U;&0@8F4@<V5T
M(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE;6)E<B`H("1Z:7`M/FUE
M;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$:7)E8W1O<GD["FUY("1M
M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE.PIN97AT('5N;&5S<R`D
M;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH/SHD0V]N9FEG.CI#;VYF
M:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M
M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D
M14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N
M86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2![
M"F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR
M96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U=',H
M<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(N
M+BY=*3L**"1M96UB97(M/F5X=')A8W14;T9I;&5.86UE9"@D9&5S=%]N86UE
M*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!E
M>'1R86-T:6YG('II<"!M96UB97(@=&\@(B1D97-T7VYA;64B73L*8VAM;V0H
M,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H<'5X(CL*?0I]"GT*?0IU
M;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97TI('L*9&EE(#P\("(N
M(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!;&EB+G!A<B!=(%L@+4ED
M:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@6R!P<F]G<F%M+G!L(%T*
M)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP87(*+@HD14Y6>U!!4E]0
M4D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H:69T*$!!4D=6*3L*?0IS
M=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`7SL*<F5Q=6ER92!&:6QE
M.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A=&@L("1E>'0I(#T@1FEL
M93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@*"=<+BXJ)RDI.PIR97%U
M:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK<&%T:"@D<&%T:"D@=6YL
M97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W:71H(&5R<F]R"GT*<W5B
M(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB.PIR97%U:7)E($1Y;F%,
M;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER92!S=')I8W0["G)E<75I
M<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q=6ER92!#87)P.PIR97%U
M:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO.PIR97%U:7)E($5X<&]R
M=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*<F5Q=6ER92!&8VYT;#L*
M<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.E-P96,["G)E<75I
M<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR97%U:7)E($E/.CI(86YD
M;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@0V]M<')E<W,Z.EIL:6([
M"G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!.PIR
M97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E(%!!4CHZ
M1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["G)E<75I<F4@
M4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q=6ER92!A='1R:6)U=&5S
M.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![(')E<75I<F4@5VEN,S(@
M?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL('T["F5V86P@>R!R97%U
M:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE=F%L('L@<F5Q=6ER92!4
M:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L('L@<F5Q=6ER92!097)L
M24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*979A;"![(')E<75I<F4@
M=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*:68@*&1E9FEN960@)$5.
M5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+RD@>PHD
M<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A8V@@;7D@)'!A=&@@*`HH
M;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!435!$25(@5$5-4$1)4B!4
M14U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@+B`I"BD@>PIN97AT('5N
M;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T:"!A;F0@+7<@)'!A=&@[
M"FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L('LH)'!W=6ED*2`](&=E
M='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF("@@9&5F:6YE9"@F5VEN
M,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]("97:6XS,CHZ3&]G:6Y.
M86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@>PHD=7-E<FYA;64@/2`D
M<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%3E9[55-%4DY!345]('Q\
M("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*;7D@)'-T;7!D:7(@/2`B)'!A
M=&@D<WES>U]D96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*
M;6MD:7(@)'-T;7!D:7(L(#`W-34["FUY("1C86-H95]D:7(["FEF("@D14Y6
M>U!!4E]#3$5!3GTI('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*?0IE;'-E
M('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;
M0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PII9B`H*&UY("1M86=I8U]P
M;W,@/2!F:6YD7W!A<E]M86=I8R@D9F@I*2`^/2`P*2!["G-E96L@)&9H+"`D
M;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#
M2$4B*2P@,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B=68L(&QE;F=T:"@B7#!#
M04-(12(I.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("1M
M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM(&QE;F=T:"@B7#!#04-(
M12(I"BT@)&-A8VAE7VYA;65?<VEZ92P@,#L*<F5A9"`D9F@L("1B=68L("1C
M86-H95]N86UE7W-I>F4["B1B=68@/7X@<R]<,"\O9SL*)&-A8VAE7V1I<B`]
M(")C86-H92TD8G5F(CL*?0I]"F-L;W-E("1F:#L*?0II9B`H(21C86-H95]D
M:7(I('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*)$5.5GM005)?0TQ%04Y]
M(#T@,3L*?0HD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U])&-A8VAE7V1I<B([
M"FUK9&ER("1S=&UP9&ER+"`P-S4U.PHD14Y6>U!!4E]414U0?2`]("1S=&UP
M9&ER.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!]
M(&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S
M('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L
M;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA
M;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["FUY("1F:#L*
M*&]P96X@)&9H+"`G/CIR87<G+"`D=&5M<&YA;64I"B8F("AP<FEN="`D9F@@
M)&-O;G1E;G1S*0HF)B`H8VQO<V4@)&9H*0IO<B!D:64@<7%;17)R;W(@=W)I
M=&EN9R`B)'1E;7!N86UE(CH@)"%=.PIC:&UO9"`D;6]D92P@)'1E;7!N86UE
M(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F=6QL;F%M
M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL;F%M93L*
M?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[4$%27U!2
M3T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI+RD@>PHD
M<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H)$5.5GM0
M05)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?5$5-4'TI
M(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@<FEN9&5X
M*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`K(#$I.PI]"FEF("@A)$5.5GM0
M05)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]
M*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D<')O9VYA;64I('L*
M<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G;F%M921S>7-[7V5X
M97TB*2!["B1P<F]G;F%M92`N/2`D<WES>U]E>&5].PIR971U<FX["GT*?0IF
M;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421S>7-[<&%T:%]S97!]7$4O+"`D
M14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N
M9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D<WES>U]D
M96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P
M<F]G;F%M921S>7-[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64D<WES>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I
M<B1S>7-[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1S
M>7-[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92![
M"B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI
M;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M
M92`]("(N)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@*&1E
M9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E9FEN
M960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W9&`I
M.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D<WES
M>U]D96QI;7U<12DO)'!W9"1S>7-[7V1E;&EM?2\["B1%3E9[4$%27U!23T=.
M04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N=B!["FEF("@@
M)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T=7)N.PI](&5L
M<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O<B`H<7<H(%-0
M05=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-12`I("D@>PID
M96QE=&4@)$5.5GLB4$%27R1?(GT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@
M0TQ%04X@1$5"54<@+RD@>PHD14Y6>R)005)?)%\B?2`]("1%3E9[(E!!4E]'
M3$]"04Q?)%\B?2!I9B!E>&ES=',@)$5.5GLB4$%27T=,3T)!3%\D7R)].PI]
M"FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y]
M.PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H
M*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN
M(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)
M3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W
M*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X
M<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I
M;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R
M.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG
M="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@
M;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G
M;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<
M9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@
M)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%
M6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2
M.PHQ.PI?7T5.1%]?"@`!&P,[```!"````"#__X4@```!(/__A8P```%0__^&
M*````9#__X>(```"3/__B+````*T__^)>````N3__XLP```#?/__BZ@```.D
M__^.+````_S__X[4```$*/__C^P```1@__^0)```!'C__Y`T```$D/__DP@`
M``4<__^32```!43__Y/````%A/__E:P```6X__^6.```!>#__Y;4```%_/__
MF+@```9`__^8\```!F3__YG\```&H/__H,P```=,__^C-```!X3__Z.X```'
M\/__LB@```@H__^RF```"$C__[0L```(I/__M<P```(`__^VE````=#__[<L
M```'M/__N:0```&\````$``````!>E(`!'Q!`1L,`0`````L````&/__@_@`
M``!L`$$.($$)00!"G@)%G0.?`4,107],!D%"W][=#@`````````\````2/__
MA#0```"<`$$.($$)00!"G@)%G`2?`40107]#G0-/W40&04+?WMP.`$$.()P$
MG0.>`I\!$4%_````*````(C__X20```!8`!!#B!!"4$`0IP$0IX"19L%$4%_
M09T#09\!```````0````M/__M^````5``````````"P```#(__^TO````)@`
M00X@00E!`$*>`D<107]$G0-"GP%/W4'?0]X.`$$&00```$@```#X__^SQ```
M`,@`00X@00E!`$*>`D<107]"G`1"GP%&G0-*W4;<0=]#W@X`009!00X@G`2>
M`I\!$4%_0MQ!WT$&04+>#@````!D```!1/__A30```$H`$(.,$&?`4.;!42>
M`DH)00!!F@9!$4%_0IP$0IT#3]I!VT$&04'<0=U!WD/?#@!!#C";!9X"GP%(
MF@:<!)T#$4%_1MK<W09!00K;0=Y!"T+;WD*;!4+;`````"P```&L__^%]```
M`,@`00XP0@E!`)T#2!%!?YX"GP%C"@9!0M_>W0X`00L``````)0```'<__^&
MC````;@`00[0`4&=`T*?`5$*W]T.`$$+00E!`$&<!$$107]$F0="F@9"FP5"
MG@)4F`A;V$'9VMO>0MQ!!D%!F0>:!IL%G`2>`A%!?T;90=I!!D%!VT'<0=Y!
MF`B9!YH&FP6<!)X"$4%_0MC9VMO<W@9!00E!`$&8"$$107]!F0=!F@9!FP5!
MG`1!G@(`````)````G3__X>L````>`!!#B!$"4$`GP%#$4%_5`9!0=\.````
M`````%0```*<__^'_````H0`00Y01!%!?YL%G`1#G0.>`DN?`9D'F@8"1`H&
M04;?WMW<V]K9#@!!"TF8"$&7"6,*UT'800M(U]A"EPF8"$370=A#EPE!F`@`
M```H```"]/__BB@```"H`$$.P`%$"4$`G@*?`4(107]8"@9!0=_>#@!!"P``
M`#0```,@__^*I````1@`00ZP"$0)00">`D*?`4<107]E"@9!0=_>#@!!"TX*
M!D%!W]X.`$$+````%````UC__XN$````.`!"#A!+#@``````%````W#__XND
M````$`!!#A!"#@``````B````XC__XN<```"U`!!#O`)1`E!`)H&1)P$GP%*
M$4%_F0<%1@Q5"@E&#$(&04(&1D/?W-K9#@!!"T&;!4F5"T*6"D*7"4*8"$.>
M`D&=`P)#"M5!UD'70=A!VT'=0=Y!"U;50=9!UT'80=M!W4'>09L%0=M"E0M!
ME@I!EPE!F`A!FP5!G0-!G@(````D```$%/__C>0```!``$$.($,)00"?`4(1
M07](WPX`009!````````/```!#S__XW\````>`!!#B!!GP%&"4$`01%!?TC?
M#@!!!D%!#B"?`43?#@!!#B"?`1%!?T3?#@!!!D$``````#````1\__^.-```
M`>P`00Y`11%!?YL%1)P$G0-"G@)"GP$"7`H&043?WMW<VPX`00L````D```$
ML/__C^P```",`$$.,$,)00"?`4(107]:"M\.`$$&04$+````&```!-C__Y!0
M````G`!!#H`!0A%!?P```````$````3T__^0T````>0`10[0@`1%"4$`F`B:
M!DV;!9X"$4%_EPF9!YP$G0.?`0)%"@9!2-_>W=S;VMG8UPX`00L`````(```
M!3C__Y)P````.`!!#A!"$4%_1PH.`$$&04$+````````.```!5S__Y*$```!
M#`!!#D!""4$`G@)$G`2=`T*?`4@107]4"@9!0]_>W=P.`$$+0IL%5=M"FP4`
M````J```!9C__Y-4```&T`!!#M`"0@E!`)X"1)P$4)L%3)H&21%!?Y\!2)T#
M09<)09@(09D'`EK70MA!V4'=2@9!1-_>W-O:#@!!#M`"EPF8")D'F@:;!9P$
MG0.>`I\!$4%_=M?8V=U%EPF8")D'G0-;E0M"E@I#U4'62)4+E@H"6PK50=9!
MUT'80=E!W4$+`D#5UM?8V=U!E0M!E@I!EPE!F`A!F0=!G0,``````#0```9$
M__^9>````F@`00Y`0PE!`)\!2)T#$4%_FP6<!$*>`@)E"@9!1-_>W=S;#@!!
M"P``````+```!GS__YNH````A`!!#B!""4$`GP%"$4%_2PK?#@!!!D%!"TW?
M#@!!!D$`````.```!JS__Z]P```";`!!#D!""4$`G@)-$4%_GP%-G0,"3MU"
M!D%"W]X.`$$.0)X"GP$107]%G0,`````-```!NC__YO````.<`!!#H`#4)H&
MFP6<!)T#G@*?`0,#APK?WMW<V]H.`$$+00E!`$$107\````<```'(/__J?@`
M``!P`$$.$$,107]6#@!!!D$``````%@```=`__^J2````90`00XP1`E!`)H&
M0YL%G`1"GP%/$4%_G@)2"@9!1-_>W-O:#@!!"T&=`US=0@9!1=_>W-O:#@!!
M#C":!IL%G`2>`I\!$4%_3)T#````````-```!YS__ZN````!H`!!#C!#"4$`
MFP5"G@)"GP%$$4%_G`2=`P)3"@9!1-_>W=S;#@!!"P``````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````#_____
M`````/____\``````````````````````````````````````8(!#`&!_H0!
M@?Z$`8`/B`&`#X@!@?Z$`8'^A`&`3,`!@@$8`8(!(`&"`0P!@@)8`8'_$`&!
M_H0!@!+<`8'^A`&`3*`!@?Z(`8("7``````!@(^,`8'^B`&!_I`!@?Z0````
M``````$````!`````0```DX````!```"6`````$```*4`````0```KH````/
M```$0P````P!@!*@````#0&`3*`````$`8`!>`````4!@`K`````!@&`!)``
M```*```$BP````L````0````%0`````````#`8(```````(```,`````%```
M``<````7`8`/H'`````!@?_T````!P&`#TP````(```#5`````D````,````
M````````````````````````````````````````````````````````````
M``````&!_Q````````````&`2V`!@$MD`8!+:`&`2VP!@$MP`8!+=`&`2W@!
M@$M\`8!+@`&`2X0!@$N(`8!+C`&`2Y`!@$N4`8!+F`&`2YP!@$N@`8!+I`&`
M2Z@!@$NL`8!+L`&`2[0!@$NX`8!+O`&`2\`!@$O$`8!+R`&`2\P!@$O0`8!+
MU`&`2]@!@$O<`8!+X`&`2^0!@$OH`8!+[`&`2_`!@$OT`8!+^`&`2_P!@$P`
M`8!,!`&`3`@!@$P,`8!,$`&`3!0!@$P8`8!,'`&`3"`!@$PD`8!,*`&`3"P!
M@$PP`8!,-`&`3#@!@$P\`8!,0`&`3$0!@$Q(`8!,3`&`3%`!@$Q4`8!,6`&`
M3%P*4$%2+G!M"@`````!@@)8`8!03$=#0SH@*$YE=$)31"!N8C,@,C`Q.3`S
M,3DI(#<N-"XP`$=#0SH@*&YB,R`R,#$Y,#,Q.2D@-RXT+C``)$YE=$)31#H@
M8W)T,"Y3+'8@,2XW(#(P,3@O,3$O,C8@,3<Z,S<Z-#4@:F]E<F<@17AP("0`
M)$YE=$)31#H@8W)T,"UC;VUM;VXN8RQV(#$N,C,@,C`Q."\Q,B\R."`R,#HQ
M,CHS-2!C:')I<W1O<R!%>'`@)``D3F5T0E-$.B!C<G1I+E,L=B`Q+C0@,C`Q
M-"\P,R\P-B`Q.3HP-3HU.2!M871T($5X<"`D`"1.971"4T0Z(&-R=&)E9VEN
M+F,L=B`Q+C$W(#(P,3@O,3(O,C@@,3@Z,3<Z,3$@8VAR:7-T;W,@17AP("0`
M)$YE=$)31#H@8W)T96YD+E,L=B`Q+C,@,C`Q-"\P,B\R-R`Q-CHT-SHT."!M
M871T($5X<"`D`"1.971"4T0Z(&-R=&XN4RQV(#$N,B`R,#$T+S`R+S(W(#$V
M.C0W.C0X(&UA='0@17AP("0`00````]G;G4``0````<$`0`N<VAS=')T86(`
M+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A>``N
M:&%S:``N9'EN<WEM`"YD>6YS='(`+G)E;&$N9'EN`"YR96QA+G!L=``N:6YI
M=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE
M`"YC=&]R<P`N9'1O<G,`+FIC<@`N9&%T82YR96PN<F\`+F9I>'5P`"YG;W0R
M`"YD>6YA;6EC`"YG;W0`+F1A=&$`+G-D871A`"YS8G-S`"YB<W,`+F-O;6UE
M;G0`+F=N=2YA='1R:6)U=&5S````````````````````````````````````
M```````````````````````+`````0````(!@`$T```!-````!<`````````
M``````$`````````$P````<````"`8`!3````4P````8```````````````$
M`````````"8````'`````@&``60```%D````%```````````````!```````
M```W````!0````(!@`%X```!>````Q@````%``````````0````$````/0``
M``L````"`8`$D```!)````8P````!@````$````$````$````$4````#````
M`@&`"L````K````$BP```````````````0````````!-````!`````(!@`],
M```/3````%0````%``````````0````,````5P````0```!"`8`/H```#Z``
M``,`````!0```!<````$````#````&$````!````!@&`$J```!*@````(```
M````````````!`````````!G`````0````8!@!+````2P```.>``````````
M`````!``````````;0````$````&`8!,H```3*`````@```````````````$
M`````````',````!`````@&`3,```$S```!`_```````````````!```````
M``![`````0````(!@(V\``"-O````0P```````````````0`````````B0``
M``$````"`8".R```CL@```?4```````````````$`````````),````!````
M`P&!_H0``/Z$````"```````````````!`````````":`````0````,!@?Z,
M``#^C`````@```````````````0`````````H0````$````#`8'^E```_I0`
M```$```````````````$`````````*8````!`````P&!_I@``/Z8````$P``
M````````````"`````````"S`````0````,!@?ZL``#^K`````0`````````
M``````0`````````N@````$````#`8'^L```_K````!@```````````````$
M`````````,`````&`````P&!_Q```/\0````X`````8`````````!`````@`
M``#)`````0````,!@?_P``#_\````!````````````````0````$````7```
M``$````#`8(````!``````$````````````````$`````````,X````!````
M`P&"`0```0$`````"0``````````````!`````````#4`````0````,!@@$,
M``$!#`````@```````````````0`````````VP````@````#`8(!&``!`10`
M```0```````````````(`````````.$````(`````P&"`2@``0$4```!;```
M````````````"`````````#F`````0```#````````$!%````#L`````````
M``````$````!````'P````$````````````!`4\```%6```````````````!
M`````````.]O___U`````````````0*E````$````````````````0``````
G```!`````P````````````$"M0```/\```````````````$`````
