################################################################################
#    iGMT: Interactive Mapping of Geoscientific Datasets.                      #
#                                                                              #
#    Copyright (C) 1998 - 2000 Thorsten W. Becker, Alexander Braun             #
#                                                                              #
#    This program is free software; you can redistribute it and/or modify      #
#    it under the terms of the GNU General Public License as published by      #
#    the Free Software Foundation; either version 2 of the License, or         #
#    (at your option) any later version.                                       #
#                                                                              #
#    This program is distributed in the hope that it will be useful,           #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#    GNU General Public License for more details.                              #
#                                                                              #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program; see the file COPYING.  If not, write to          #
#    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,          #
#    Boston, MA 02111-1307, USA.                                               #
#                                                                              #
################################################################################

# this script deals with the WSM format of the World Stress Map
# by Zoback et al. (1992) and Mueller et al. (1997)
# for details about the format and the types that have been lumped together
# here, refer to http://www-wsm.physik.uni-karlsruhe.de/pub/contribute/fieldswsm.html 
#
# it reads the original data and prints output to stdout that can be used
# for plotting with GMT
# the user can supply sort criteria as follows

# input parameters are 
#
# pregime: 1 means plot only extensional mechanism derived 
#            compressional stress axis (default)
#          2 means plot only strike-slip mechanism derived..
#          3 compressional
#          4 undetermined
#
# pquality: 1 means plot only A class data points
#           2 means A and B (default)
#           and so on to 4
#
#
# type:     t1: if 1, plot focal mechanism derived data, if 0 not
#               (default)
#           t2: switch to include borehole breakouts
#           t3: hydro frac
#           t4: overcoring
#           t5: geological indicators
#           tswitches add up
# 

BEGIN{
# separator
  FS=";";
# check if sort criteria are set, if not use defaults
  if(pregime == 0)
    pregime=1;# compressional mechanism derived
# A and B quality data
  if(pquality == 0)
    pquality=2;

# only focal mechanism derived data  
  if(t1 == 0 && t2 == 0 && t3 == 0 && t4 == 0 && t5 == 0){
    t1=1;t2=0;t3=0;t4=0;t5=0;
  }
}
{    
  #skip header line
  if (NR == 1) next;

# coordinates
  lon=$3;
  lat=$2;
  

# azimuth 
  azi=$4;

# sort out according to type of measurement      

  type=substr($5,1,2);



  if(type == "FM"){ 
    if(t1 == 0)next;
  }else if(type == "BO"){
    if(t2 == 0)next;
  }else if(type == "HF"){
    if(t3==0)next;
  }else if(type == "OC"){
    if(t4 == 0)next;
  }else if(type == "GF"){
    if(t5 == 0)next;
  }else { 
    next;
  }
  

  quality=substr($7,1,1);
# sort out quality
  if(quality == "A"){
    ;
  }else if(quality == "B"){
    if(pquality < 2)next;
  }else if(quality == "C"){
    if(pquality < 3)next;
  }else if(quality == "D"){
    if(pquality < 4)next;
  }else{
    next;
  }


# assign type
# assign regime  
  reg=substr($8,1,2);

  if(reg == "NF"){
    regime=1;
  }else if(reg == "NS")
    regime=2;
  else if(reg == "SS")
    regime=3;
  else if(reg == "TS")
    regime=4;
  else if(reg == "TF")
    regime=5;
  else 
    regime=-1;
  
#  print(type,quality,reg);

# output if fits criteria


  if(pregime == 1){ # extensional mechanism output
    if(regime == 1) # pure
      print(lon,lat,1,0,azi);
    else if(regime == 2) # with strike slip component
      print(lon,lat,1,-0.5,azi);
  }else if(pregime == 2){# pure strike slip mech. out
    if(regime == 3)
      print(lon,lat,1,-1,azi);
  }else if(pregime == 3){# compressional mech. out
    if(regime == 4) # with ss
      print(lon,lat,0.5,-1,azi);
    else if(regime == 5) # pure 
      print(lon,lat,0,-1,azi);
  }else if(pregime == 4 && regime== -1){#undetermined, no mechanism
    print(lon,lat,0,-1,azi);
  }
  
}
