-- $Id: harmbase2.sql 6957 2018-12-26 18:33:31Z flaterco $

-- Harmbase 2:  harmonic constant management package.
-- Copyright (C) 2004  David Flater.
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.


-- Schema for harmbase application:  maintenance and management of harmonic
-- constant sets for XTide.

-- This works in Postgres 11.1.  You must have done 'createdb harmbase2' at
-- the Unix shell prompt before running this script.


-- Select harmbase database
\connect harmbase2


-- Congen definitions and speeds for recognized constituents.
create table constituents (
  name text primary key,
  definition text not null,
  speed double precision not null check (speed > 0)
);

-- Congen input rev. 2004-08-16 with changes per "flips" investigation:
--    3KM5 definition changed 2016-12
--    KJ2-IHO added 2018-01
COPY constituents (name, definition, speed) FROM stdin;
J1	Basic 1 1 1 -1 0 -90    0 -1 0 0 0 0     76	15.5854433
K1	Compound 0 1 0 0 0 0 0 0	15.041068599999999
K2	Compound 0 0 0 0 0 0 0 1	30.082137299999999
L2	Compound 0 0 0 0 0 0 1 0	29.5284789
M1	Basic 1 -1 1 0 0 -90    1 -1 0 0 1 0     206	14.4966939
M2	Compound 0 0 0 1 0 0 0 0	28.984104200000001
M3	Basic 3 -3 3 0 0 0      3 -3 0 0 0 0     149	43.4761563
M4	Compound 0 0 0 2 0 0 0 0	57.968208400000002
M6	Compound 0 0 0 3 0 0 0 0	86.952312599999999
M8	Compound 0 0 0 4 0 0 0 0	115.9364169
N2	Compound 0 0 0 0 0 1 0 0	28.439729499999999
2N2	Basic 2 -4 2 2 0 0      2 -2 0 0 0 0     78	27.8953548
O1	Compound 1 0 0 0 0 0 0 0	13.9430356
OO1	Basic 1 2 1 0 0 -90     -2 -1 0 0 0 0    77	16.139101700000001
P1	Compound 0 0 1 0 0 0 0 0	14.958931400000001
Q1	Compound 0 0 0 0 0 0 0 0 1	13.398660899999999
2Q1	Basic 1 -4 1 2 0 90     2 -1 0 0 0 0     75	12.854286200000001
R2	Basic 2 0 1 0 -1 180    0 0 0 0 0 0      1	30.041066699999998
S1	Basic 1 0 0 0 0 0       0 0 0 0 0 0      1	15
S2	Compound 0 0 0 0 1 0 0 0	30
S4	Compound 0 0 0 0 2 0 0 0	60
S6	Compound 0 0 0 0 3 0 0 0	90
T2	Basic 2 0 -1 0 1 0      0 0 0 0 0 0      1	29.958933300000002
LDA2	Basic 2 -1 0 1 0 180    2 -2 0 0 0 0     78	29.455625300000001
MU2	Basic 2 -4 4 0 0 0      2 -2 0 0 0 0     78	27.968208400000002
NU2	Basic 2 -3 4 -1 0 0     2 -2 0 0 0 0     78	28.512583100000001
RHO1	Basic 1 -3 3 -1 0 90    2 -1 0 0 0 0     75	13.4715145
MK3	Compound 0 1 0 1 0 0 0 0	44.025172900000001
2MK3	Compound 0 -1 0 2 0 0 0 0	42.927139799999999
MN4	Compound 0 0 0 1 0 1 0 0	57.423833700000003
MS4	Compound 0 0 0 1 1 0 0 0	58.984104199999997
2SM2	Compound 0 0 0 -1 2 0 0 0	31.015895799999999
MF	Basic 0 2 0 0 0 0       -2 0 0 0 0 0     74	1.0980331000000001
MSF	Compound 0 0 0 -1 1 0 0 0	1.0158958
MM	Basic 0 1 0 -1 0 0      0 0 0 0 0 0      73	0.54437469999999999
SA	Basic 0 0 1 0 0 0       0 0 0 0 0 0      1	0.041068599999999997
SSA	Basic 0 0 2 0 0 0       0 0 0 0 0 0      1	0.082137299999999996
SA-IOS	Doodson 0 0 1 0 -1 0    0	0.041066699999999998
MF-IOS	Doodson 0 2 0 0  0 0    0	1.0980331000000001
S1-IOS	Doodson 1 0 0 0 1 -90   2\n  0  0 -2 .0  0.3534     0  1  0 .50 0.0264	15.000002
OO1-IOS	Doodson 1  2  1  0  0 -90    8\n  -2 -1  0 .50 0.0037    -2  0  0 .0  0.1496    -2  1  0 .0  0.0296\n  -1  0  0 .25 0.0240R1  -1  1  0 .25 0.0099R1   0  1  0 .0  0.6398\n  0  2  0 .0  0.1342     0  3  0 .0  0.0086	16.139101700000001
R2-IOS	Doodson 2 0 1 0 -1 180    2\n  0  0  2 .50 0.2535     0  1  2 .0  0.0141	30.041066699999998
A7	Basic 0 3 0 -1 0 0      -2 0 0 0 0 0     74	1.6424078
2MK5	Compound 0  1  0  2  0  0  0  0	73.009277100000006
2MK6	Compound 0  0  0  2  0  0  0  1	88.050345699999994
2MN2	Compound 0  0  0  2  0 -1  0  0	29.5284789
2MN6	Compound 0  0  0  2  0  1  0  0	86.407937899999993
2MS6	Compound 0  0  0  2  1  0  0  0	87.968208399999995
2NM6	Compound 0  0  0  1  0  2  0  0	85.863563200000002
2SK5	Compound 0  1  0  0  2  0  0  0	75.041068600000003
2SM6	Compound 0  0  0  1  2  0  0  0	88.984104200000004
3MK7	Compound 0  1  0  3  0  0  0  0	101.9933813
3MN8	Compound 0  0  0  3  0  1  0  0	115.39204220000001
3MS2	Compound 0  0  0  3 -2  0  0  0	26.952312599999999
3MS4	Compound 0  0  0  3 -1  0  0  0	56.952312599999999
3MS8	Compound 0  0  0  3  1  0  0  0	116.9523126
ALP1	Doodson 1 -5 3 1 0 90     2\n  -1  0  0 .75 0.0360R1   0 -1  0 .00 0.1906	12.3827651
BET1	Doodson 1 -1 -1 1 0 -90   1\n  0 -1  0 .00 0.2266	14.4145567
CHI1	Basic 1 -1 3 -1 0 -90   0 -1 0 0 0 0     76	14.5695476
H1	Doodson 2 -2 1 0 1 180    2\n  0 -1  0 .50 0.0224     1  0 -1 .50 0.0447	28.943037499999999
H2	Doodson 2 -2 3 0 -1 0     1\n  0 -1  0 .50 0.0217	29.025170899999999
KJ2	Basic 2 1 2 -1 0 0      0 -2 0 0 0 0     79	30.626512000000002
ETA2	Doodson 2 1 2 -1 0 0      7\n  0 -1  0 .50 0.0187     0  1  0 .0  0.4355     0  2  0 .0  0.0467\n  1  0  0 .75 0.0747R2   1  1  0 .75 0.0482R2   1  2  0 .75 0.0093R2\n  2  0  0 .50 0.0078	30.626512000000002
KQ1	Basic 1 3 1 -1 0 -90    -2 -1 0 0 0 0    77	16.6834764
UPS1	Doodson 1 3 1 -1 0 -90    5\n  -2  0  0 .00 0.0611     0  1  0 .00 0.6399     0  2  0 .00 0.1318\n  1  0  0 .25 0.0289R1   1  1  0 .25 0.0257R1	16.6834764
M10	Compound 0 0 0 5 0 0 0 0	144.9205211
M12	Compound 0 0 0 6 0 0 0 0	173.90462529999999
MK4	Compound 0  0  0  1  0  0  0  1	59.066241499999997
MKS2	Compound 0  0  0  1 -1  0  0  1	29.0662415
MNS2	Compound 0  0  0  1 -1  1  0  0	27.423833699999999
EPS2	Doodson 2 -5 4 1 0 0   3\n  -1 -1  0 .25 0.0075R2  -1  0  0 .25 0.0402R2   0 -1  0 .50 0.0373	27.423833699999999
MO3	Compound 1 0 0 1 0 0 0 0	42.927139799999999
MP1	Basic 1 -2 3 0 0 -90    0 -1 0 0 0 0     76	14.025172899999999
TAU1	Doodson 1 -2 3 0 0 -90    5\n  -2  0  0 .0  0.0446    -1  0  0 .25 0.0426R1   0 -1  0 .50 0.0284\n  0  1  0 .50 0.2170     0  2  0 .50 0.0142	14.025172899999999
MPS2	Compound 0 0 1 1 0 0 0 0 0 0 -1	28.943035600000002
MSK6	Compound 0  0  0  1  1  0  0  1	89.066241500000004
MSM	Basic 0 1 -2 1 0 0      0 0 0 0 0 0      73	0.47152110000000003
MSN2	Compound 0  0  0  1  1 -1  0  0	30.544374699999999
MSN6	Compound 0  0  0  1  1  1  0  0	87.423833700000003
NLK2	Compound 0 0 0 0 0 1 1 -1	27.886071099999999
NO1	Compound -1 0 0 0 0 1 0 0	14.4966939
OP2	Compound 1  0  1  0  0  0  0  0	28.901966900000001
OQ2	Doodson 2 -5 2 3 0 0   2\n  -1  0  0 .25 0.1042R2   0 -1  0 .50 0.0386	27.350980100000001
PHI1	Basic 1 0 3 0 0 -90     0 0 0 0 0 0      1	15.1232059
KP1	Compound 0 0 -1 0 0 0 0 1	15.1232059
PI1	Basic 1 0 -2 0 1 90     0 0 0 0 0 0      1	14.917864700000001
TK1	Basic 1 0 -2 0 1 90     0 0 1 0 0 0      227	14.917864700000001
PSI1	Basic 1 0 2 0 -1 -90    0 0 0 0 0 0      1	15.082135299999999
RP1	Basic 1 0 2 0 -1 90     0 0 0 0 0 0      1	15.082135299999999
S3	Basic 3 0 0 0 0 0       0 0 0 0 0 0      1	45
SIG1	Basic 1 -4 3 0 0 90     2 -1 0 0 0 0     75	12.927139800000001
SK3	Compound 0  1  0  0  1  0  0  0	45.041068600000003
SK4	Compound 0 0 0 0 1 0 0 1	60.082137299999999
SN4	Compound 0  0  0  0  1  1  0  0	58.439729499999999
SNK6	Compound 0 0 0 0 1 1 0 1	88.521866799999998
SO1	Basic 1 2 -1 0 0 -90    0 -1 0 0 0 0     76	16.056964399999998
SO3	Compound 1  0  0  0  1  0  0  0	43.943035600000002
THE1	Basic 1 1 -1 1 0 -90    0 -1 0 0 0 0     76	15.512589699999999
2PO1	Compound -1  0  2  0  0  0  0  0	15.974827100000001
2NS2	Compound 0  0  0  0 -1  2  0  0	26.879459000000001
MLN2S2	Compound 0  0  0  1 -2  1  1  0	26.952312599999999
2ML2S2	Compound 0  0  0  2 -2  0  1  0	27.496687300000001
SKM2	Compound 0  0  0 -1  1  0  0  1	31.098033099999999
2MS2K2	Compound 0  0  0  2  1  0  0 -2	27.803933900000001
MKL2S2	Compound 0  0  0  1 -2  0  1  1	28.5947204
M2(KS)2	Compound 0  0  0  1 -2  0  0  2	29.1483788
2SN(MK)2	Compound 0  0  0 -1  2  1  0 -1	29.373487999999998
2KM(SN)2	Compound 0  0  0  1 -1 -1  0  2	30.708649300000001
NO3	Compound 1  0  0  0  0  1  0  0	42.3827651
2MLS4	Compound 0  0  0  2 -1  0  1  0	57.496687299999998
ML4	Compound 0  0  0  1  0  0  1  0	58.512583100000001
N4	Compound 0  0  0  0  0  2  0  0	56.879458999999997
SL4	Compound 0  0  0  0  1  0  1  0	59.528478900000003
MNO5	Compound 1  0  0  1  0  1  0  0	71.366869300000005
2MO5	Compound 1  0  0  2  0  0  0  0	71.911243999999996
MSK5	Compound 0  1  0  1  1  0  0  0	74.025172900000001
3KM5	Compound 0  1  0  1  0  0  0  1	74.107310100000007
2MP5	Compound 0  0  1  2  0  0  0  0	72.927139800000006
3MP5	Compound 0  0 -1  3  0  0  0  0	71.993381299999996
MNK5	Compound 0  1  0  1  0  1  0  0	72.4649024
2NMLS6	Compound 0  0  0  1 -1  2  1  0	85.392042200000006
MSL6	Compound 0  0  0  1  1  0  1  0	88.512583100000001
2ML6	Compound 0  0  0  2  0  0  1  0	87.496687300000005
2MNLS6	Compound 0  0  0  2 -1  1  1  0	85.936416899999998
3MLS6	Compound 0  0  0  3 -1  0  1  0	86.480791600000003
2MNO7	Compound 1  0  0  2  0  1  0  0	100.35097349999999
2NMK7	Compound 0  1  0  1  0  2  0  0	100.9046319
2MSO7	Compound 1  0  0  2  1  0  0  0	101.911244
MSKO7	Compound 1  0  0  1  1  0  0  1	103.00927710000001
2MSN8	Compound 0  0  0  2  1  1  0  0	116.40793789999999
2(MS)8	Compound 0  0  0  2  2  0  0  0	117.96820839999999
2(MN)8	Compound 0  0  0  2  0  2  0  0	114.8476675
2MSL8	Compound 0  0  0  2  1  0  1  0	117.4966873
4MLS8	Compound 0  0  0  4 -1  0  1  0	115.46489579999999
3ML8	Compound 0  0  0  3  0  0  1  0	116.4807916
3MK8	Compound 0  0  0  3  0  0  0  1	117.0344499
2MSK8	Compound 0  0  0  2  1  0  0  1	118.05034569999999
2M2NK9	Compound 0  1  0  2  0  2  0  0	129.88873609999999
3MNK9	Compound 0  1  0  3  0  1  0  0	130.43311080000001
4MK9	Compound 0  1  0  4  0  0  0  0	130.9774855
3MSK9	Compound 0  1  0  3  1  0  0  0	131.99338130000001
4MN10	Compound 0  0  0  4  0  1  0  0	144.37614640000001
3MNS10	Compound 0  0  0  3  1  1  0  0	145.39204219999999
4MS10	Compound 0  0  0  4  1  0  0  0	145.93641690000001
3MSL10	Compound 0  0  0  3  1  0  1  0	146.4807916
3M2S10	Compound 0  0  0  3  2  0  0  0	146.9523126
4MSK11	Compound 0  1  0  4  1  0  0  0	160.9774855
4MNS12	Compound 0  0  0  4  1  1  0  0	174.37614640000001
5MS12	Compound 0  0  0  5  1  0  0  0	174.9205211
4MSL12	Compound 0  0  0  4  1  0  1  0	175.46489579999999
4M2S12	Compound 0  0  0  4  2  0  0  0	175.93641690000001
M1C	Basic 1 -1 1 0 0 0   1 -1 0 0 0 0   144	14.4920521
3MKS2	Compound 0 0 0 3 -1 0 0 -1	26.870175400000001
OQ2-HORN	Compound 1 0 0 0 0 0 0 0 1	27.341696500000001
MSK2	Compound 0 0 0 1 1 0 0 -1	28.901966900000001
MSP2	Compound 0 0 -1 1 0 0 0 0 0 0 1	29.025172900000001
2MP3	Compound 0 0 -1 2 0 0 0 0	43.009277099999998
4MS4	Compound 0 0 0 4 -2 0 0 0	55.936416899999998
2MNS4	Compound 0 0 0 2 -1 1 0 0	56.4079379
2MSK4	Compound 0 0 0 2 1 0 0 -1	57.886071100000002
3MN4	Compound 0 0 0 3 0 -1 0 0	58.512583100000001
2MSN4	Compound 0 0 0 2 1 -1 0 0	59.528478900000003
3MK5	Compound 0 -1 0 3 0 0 0 0	71.911243999999996
3MO5	Compound -1 0 0 3 0 0 0 0	73.009277100000006
3MNS6	Compound 0 0 0 3 -1 1 0 0	85.392042200000006
4MS6	Compound 0 0 0 4 -1 0 0 0	85.936416899999998
2MNU6	Compound 0 0 0 2 0 0 0 0 0 1	86.480791600000003
3MSK6	Compound 0 0 0 3 1 0 0 -1	86.870175399999994
MKNU6	Compound 0 0 0 1 0 0 0 1 0 1	87.578824600000004
3MSN6	Compound 0 0 0 3 1 -1 0 0	88.512583100000001
M7	Compound 0 0 0 3 0 0 0 0 0 0 0 1	101.4490066
2MNK8	Compound 0 0 0 2 0 1 0 1	116.49007520000001
2(MS)N10	Compound 0 0 0 2 2 1 0 0	146.40793790000001
MNUS2	Compound 0 0 0 1 -1 0 0 0 0 1	27.496687300000001
2MK2	Compound 0 0 0 2 0 0 0 -1	27.886071099999999
KJ2-IHO	Basic  2 1 2 -1 0 180    0 -2 0 0 0 0   79	30.6265120000000017
\.


-- Known constituent aliases, relative to different input formats.
create table aliases (
  format text,
  alias  text,
  name   text not null references constituents,
  primary key (format, alias) -- makes them both not null
);

COPY aliases (format, alias, name) FROM stdin;
N	SM	MSF
N	OQ2	OQ2-HORN
N	LABDA2	LDA2
N	3MSN10	3MNS10
N	4MSN12	4MNS12
G	OQ2	OQ2-HORN
G	LAM2	LDA2
G	4MSN12	4MNS12
G	MQ3	NO3
G	3M(SK2	3MKS2
G	3M2S2	3MS2
G	2MV6	2MNU6
G	MVS2	MNUS2
GI	SA	SA-IOS
GI	MF	MF-IOS
GI	S1	S1-IOS
GI	OO1	OO1-IOS
GI	R2	R2-IOS
NOS	LAM2	LDA2
NOS	RHO	RHO1
NOS	SIGMA1	SIG1
NOS	2MLNS6	2MNLS6
NOS	2MS8	2(MS)8
NOS	2MN8	2(MN)8
NOS	M2KS2	M2(KS)2
NOS	2SNMK2	2SN(MK)2
NOS	2KMSN2	2KM(SN)2
NOS	THETA1	THE1
NOS	OO2	OQ2-HORN
NOS	RP1	PSI1
NOS	KJ2	KJ2-IHO
\.


-- To limit the number of joins, unions and inserts that must be
-- performed, the table data_sets was defined as a denormalized
-- combination of reference stations and subordinate stations.  For
-- each row, the columns specific to the other subclass are nulled
-- out.  The views reference_stations and subordinate_stations defined
-- below project the subclasses as they should be seen, omitting the
-- extraneous columns.

-- The table inheritance feature implemented in PostgreSQL 8.2.3
-- addresses the concern that led to the denormalization, but as
-- currently implemented it has undesirable consequences for
-- constraints.

-- The canonical test to determine whether a data set is a current
-- station is to check whether the units are knots or knots^2.

create table data_sets (

  -- Pseudo key for data sets.  (There may be multiple data sets for
  -- the same station or with null station IDs.)
  index              serial primary key,

  -- Station name.  For currents, the depth should be indicated in the
  -- name field (e.g., Blah Blah Current, depth 22).  This is
  -- particularly important if multiple depths are supported for the
  -- same station, as there will be no other way to tell them apart.
  name               text not null,
  original_name      text,

  -- Name of the authority that assigned the station ID (e.g., NOS).
  station_id_context text,

  -- An identifier for the tide station (e.g., 1611347) defined by
  -- the authority specified in station_id_context.
  station_id         text,

  -- Degrees and decimal degrees (not minutes and seconds).
  lat                double precision
    check (lat >= -90  and lat <= 90),
  lng                double precision
    check (lng >= -180 and lng <= 180), -- XTide allows both -180 and +180

  -- Timezone:  a zoneinfo path like :America/New_York that affects
  -- which time zone results are rendered into.  Not to be confused
  -- with meridian, which actually affects the calibration of the
  -- results relative to real time.  See comments for meridian below.
  timezone           text not null,

  -- Default choices for countries can be found in
  -- libtcd/tide_db_default.h; however, you can use others.
  country            text,

  -- libtcd 2.0 does not include a state field; this is converted to an
  -- X-field if it is non-null.
  state              text,

  -- Units for the amplitudes of the constants, the datum, and any
  -- level_add corrections.  Units of knots^2 indicate a hydraulic
  -- current.  In this case, it should be understood that only the
  -- amplitudes of the constants are such as to give results in knots
  -- squared.  The datum and any level_add corrections are in plain
  -- knots; i.e., the square root of the amplitude is taken before the
  -- datum and any corrections are added in.
  units              text not null -- for datum and constituent amplitudes
    check (units = 'feet' or units = 'meters' or units = 'knots'
    or units = 'knots^2'),

  -- Directions of flood and ebb in degrees true.
  min_dir            double precision
    check (min_dir >= 0 and min_dir < 360),
  max_dir            double precision
    check (max_dir >= 0 and max_dir < 360),

  -- Messages that MUST be delivered to the user of a data set at the
  -- time of use.  XTide currently appends these to the location name,
  -- so keep it short.
  legalese           text,

  -- Messages that MAY be delivered to the user of a data set at the
  -- time of use.  XTide will print the notes in modes where this is
  -- convenient but ignore them when it is not.
  notes              text,

  -- Messages that are generally NOT delivered to the user at the time
  -- of use but can be called up using "About this station."
  comments           text,

  -- Where you got the data set.
  source             text,

  -- Public domain, Non-commercial use only, DoD use only, ...
  restriction text not null,

  -- The date on which the data set was imported into Harmbase2.
  date_imported      date default current_date,

  -- Source-specific fields that are not worth creating columns for.
  xfields            text,


  -- Attributes for reference stations only.

  -- Meridian:  the standard time to which the phases of the constants
  -- are adjusted, in hours and minutes east of UTC.  Most of the time
  -- it will either be local standard time or UTC, but sometimes the
  -- local standard time is wrong.  Meridian affects the calibration
  -- of the predictions relative to real time.  Not to be confused
  -- with timezone, which only affects which time zone the results are
  -- rendered into.  For example, adjusting the meridian will change
  -- high tide from 5:00 EST to 4:00 EST, while adjusting the time
  -- zone will change it from 5:00 EST to 4:00 CST (actually the same
  -- "real time" in two different time zones).
  meridian             interval hour to minute,

  -- Default choices for datum kinds can be found in
  -- libtcd/tide_db_default.h; however, you can use others.
  -- This can be null for currents.
  datumkind            text,

  datum                double precision,
  months_on_station    smallint,
  last_date_on_station date,


  -- Attributes for sub stations only.

  -- Certain sub stations in the Pacific use reference stations across
  -- the international date line and the predictions are literally
  -- calculated from the previous or next day's tides at the reference
  -- station.  (Confirmed by comparison with NOS web site.)  At one
  -- time, Postgres would truncate intervals exceeding 24 hours unless
  -- they were defined as day to minute.  This problem no longer
  -- exists in PostgreSQL 8.2.3.

  ref_index integer    references data_sets,
  min_time_add         interval hour to minute,
  min_level_add        double precision,
  min_level_multiply   double precision
    check (min_level_multiply > 0),
  max_time_add         interval hour to minute,
  max_level_add        double precision,
  max_level_multiply   double precision
    check (max_level_multiply > 0),
  flood_begins         interval hour to minute,
  ebb_begins           interval hour to minute,


  -- TIDE_RECORD has an attribute confidence of type byte that is
  -- set to 10 in build_tide_db.c.  I don't know what its semantics
  -- are so I'm leaving it out.


  -- Table constraints
  check ((lat is null and lng is null) or
         (lat is not null and lng is not null)),
  check ((datum is null and meridian is null) or
         (datum is not null and meridian is not null)),
  check (datum is not null or ref_index is not null)
);


-- Views to filter data_sets by station type.  Note that the canonical
-- test for whether a station is a reference station or a subordinate
-- station is to check whether datum is null.

create view reference_stations as
  select index, name, original_name, station_id_context, station_id, lat, lng,
         timezone, country, state, units, min_dir, max_dir, legalese,
         notes, comments, source, restriction, date_imported, xfields,
         -- end of common columns
         meridian, datumkind, datum, months_on_station, last_date_on_station
    from data_sets
    where datum is not null;

create view subordinate_stations as
  select index, name, original_name, station_id_context, station_id, lat, lng,
         timezone, country, state, units, min_dir, max_dir, legalese,
         notes, comments, source, restriction, date_imported, xfields,
         -- end of common columns
         ref_index, min_time_add, min_level_add, min_level_multiply,
         max_time_add, max_level_add, max_level_multiply, flood_begins,
         ebb_begins
    from data_sets
    where datum is null;


-- Harmonic constants for reference stations.
create table constants (
  index integer          references data_sets on delete cascade,
  name  text             references constituents,
  phase double precision not null -- Degrees
    check (phase >= 0 and phase < 360),
  amp   double precision not null -- In units said in data_sets
    check (amp >= 0.00005), -- Don't junk up database with zeros!

  primary key (index, name) -- makes them both not null
);
