/* 
* MyGoGrinder - a program to practice Go problems
* Copyright (c) 2004-2010 Tim Kington
*   timkington@users.sourceforge.net
* Portions Copyright (C) Ruediger Klehn (2015)
*   RuediRf@users.sourceforge.net
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
*
*/
// with a little help of usagi ( kgsusagi @ sf.net )
package GoGrinder;

import java.awt.*;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;

import GoGrinder.ui.*;
import GoGrinder.sgf.*;
import GoGrinder.GS;
import GoGrinder.ui.ProbFrame;
import java.io.*;

/**
 *
 * @author  tkington
 * @author  Ruediger Klehn
 */
public class SGFController extends Controller {
    public static final int PLAY = 0;
    public static final int NAV_SOLN = 1;
    
    private SGFNode root;
    private SGFNode curNode;
    private WidgetPanel widgetPanel;
    private JTextArea commentArea;
    private JLabel statusBar;
    private int mode;
    private int playerColor;
    private int curColor;
    private boolean editTried = false;
    private boolean enabled = true;
    static final String NL = Main.NEW_LINE;
    private boolean hasResult;
    private boolean result;
    private boolean hasPlayedSound;
    public static String currFile = "nix";
    private static String msg;
    public ProbData currentProblem; // for startProblem(), restartProblem ()
    
    public void changeCharacterSet(){ // SGFController useController
      ChangeCharacterSet chCharSet = new ChangeCharacterSet(currentProblem,        // probdata, caption
                                                           "Change text decoding for the current problem"); // ,useController
      currentProblem = chCharSet.forProblem(); 
      // is it possible to return directly to the same node? -> in SGFController: private SGFNode curNode;
      // possibly we need to give numbers to the nodes and remember the number of the current node
      // and ... remember orientation
    }
    
    public boolean startProblem(ProbData prob) { // true for successful loading

        currentProblem = prob;
        // if text display is not to our liking, we can redo prob.getSGF() with a choosen Charset
        msg = ""; // for the error messages
        int pathToSettingsLength = Main.pathToSettings.length() + 1;
        String filename = prob.getFile().getPath().substring(pathToSettingsLength);  // this is just for the display
        currFile = Main.pathToSettings + Main.SLASH + filename; // this is to have the complete path for messages etc
// if(DEBUG2)d.b.g("filename = " + filename); // shortened for display
// if(DEBUG2)d.b.g("currFile = " + currFile); // full path
        statusBar.setText(((Main.PORTABLE) ? " Portable " : (Main.TEST) ? " Test " : " My ") + filename); // display  
        // this reads "My problems/SUBDIR/NAME.sgf" resp. "Portable problems/SUBDIR/NAME.sgf" 
        // resp. "Test problems/SUBDIR/NAME.sgf" (the extra leading space is for optical reasons)
                                                       // or "AllUsersProblems/SUBDIR/NAME.sgf"
                                                       // or "MyGoGrinderProblems/SUBDIR/NAME.sgf"
                                                       // or "SOURCEPATH/NAME.sgf"
                    // in settings, names would change: MyStats, AllUsersStats (or SharedStats) , MyGoGrinderStats
                    // edited files could be in SETTINGS/PATH-WITHOUT-SPECIAL-CHARACTERS/ (...NAME.sgf)
                    // and would replace the original files
        boolean parseSuccess = true;
        String parseThisCode = "";
        String codeFromFile = "";
        do{ // the loop tests for !parseSuccess, so on first pass, we use the code from the file
          codeFromFile = prob.getSGF();
          parseThisCode = (parseSuccess) ? codeFromFile : Main.SGF_DEFAULT_CODE;
// if(DEBUG7) d.b.p(currFile +"\n" + codeFromFile +"\n"+prob.getUseCharset());
          
          try { // this seems to be a job for the parser...
            SGFParser.checkingSGF = true;
            ArrayList recs = SGFParser.parse(parseThisCode); // recs - rectangles? (boards/problems)
            // recs - a really intuitive name and - is type <String>? - no, seems to be <SGFNode> - should be <MainBranch>
           // if(recs.size() == 0 || recs == null) or if(SGFParser.isDefectSGF()) errorhandling without Exception
            if(recs.size() > 1){ // no need to write this to the log file now
                String parserMsg = Messages.getString("contains_mult_probs");
                JOptionPane.showMessageDialog(null, prob.getFile().getPath()
                                            + "\n " + parserMsg);
                String stackTop = "(~) " + new Exception().getStackTrace()[0];
                SGFLog.SGFLog2(0, parserMsg, "", stackTop);
            }
            root = (SGFNode)recs.get(0); // (1) etc are thrown away; could go through the subproblems (T.K. planned this)
            int size = root.getSize();
            if(size != 0)
                board = new Board(size);
            else board = new Board(19);
            root.validatePoints(board.getSize()); // this would also be the job of the parser (I think)
            if(root.getBounds() == null){
              msg = "(No stones? - Check the file!)"; // Messages.getString("no_points")
              int sgfSize = parseThisCode.length();
              String codeExample = "";
              if (sgfSize > 200){
                codeExample = parseThisCode.substring(0, 190) + " ... (code example shortened from " + sgfSize + " to 190 bytes)";
              }
              else {
              codeExample = parseThisCode;  
              }
              throw new SGFParseException(msg, codeExample);
            }
            parseSuccess = true;
          }
          catch(OutOfMemoryError e) { 
          // I want to have the stacktrace handed over to an SGFParseException (as the exc. happened while parsing)
            String stackTop = "" + e.getStackTrace()[0];
            msg = Messages.getString("out_of_memory");
            JOptionPane.showMessageDialog(null, msg+ "\n " + Messages.getString("file_colon") + " \"" + filename + "\""); 
//            NumberFormatException?
            Main.logSilent(msg);
            // ExceptionHandler.logCommonProblem(e, msg);
            return false;
          }

 // ####################################################################################
          catch(SGFParseException e) { // ### WE DON'T COME HERE, WHEN VALIDATING!! same with splitting ###
            String displayString = "";
            String htmlStartMsg = "<html>"; // code enclosed by <span style=""></span>, 
            String htmlEndMsg = "</html>"; // complete msg enclosed by <html></html> (lowercase is o.k.)
            String htmlBr = "<br>";
            parseSuccess = false;
            if (!SGFParseException.newExc){
              msg = e.getMessage();
              Main.logSilent(e, filename); 
              if(msg.length() > 250){
                msg = msg.substring(0, 240) + Messages.getString("dot_dot_dot");
              displayString = Messages.getString("err_parsing_sgf_skipping") + htmlBr
                            + Messages.getString("file_colon") + " \"" + filename + "\"" + htmlBr
                            + msg;
              }
            }
            else displayString = SGFLog.displayString;
            SGFParseException.newExc = false;
            
            String message = "";
            String question1 = htmlBr + "How do we continue?";
            String question2 = htmlBr + "Do you want to try a text edit?"; // if move defect file is activated, edit is deactivated
            
            String caption = "Error parsing SGF!";
            String [] optionsNoEdit = {"Skip", "Use default code"}; // "Move file"
            String [] optionsEdit = {"Skip", "Use default code", "Text edit"}; // "Move file"
                                                   
            //boolean allowTextEdit = true; // need change: false, if file not found // or no text editor configured
            int btnClicked = 0;
            
            if (!GS.getAllowTextEdit() || GS.getParserMoveDefectFile()){ // if move... is active, edit... is not active
                message = htmlStartMsg + displayString + question1 + htmlEndMsg;
                btnClicked = JOptionPane.showOptionDialog(panel, message, 
                                                          caption, JOptionPane.OK_CANCEL_OPTION, 
                                                          JOptionPane.QUESTION_MESSAGE, null, 
                                                          optionsNoEdit, optionsNoEdit[0]);
            }
            // java api constant-values: YES / OK // [ESC] / [x](3+0//-1) / Go on | NO (1) / Text edit | CANCEL (2) / Select
            // here this gives: buttons left to right = 0, 1, 2 ; ESC/[x] = -1
            else { // here the optionDialog could have a 4th button "help edit" with some 
                   // hints, why a problem doesn't load; (it's easier to describe than 
                   // to do an automatic repair). 
                   // "Important: Don't use an office program for this!" and
                   // give names of the usual text editors (Win=notepad, Mac=..., Linux=gedit or kate)
              message = htmlStartMsg + displayString + question2 + htmlEndMsg;
              btnClicked = JOptionPane.showOptionDialog(panel, message, 
                                                        caption, JOptionPane.YES_NO_CANCEL_OPTION, 
                                                        JOptionPane.QUESTION_MESSAGE, null, 
                                                        optionsEdit, optionsEdit[0]);
            }                  // further: if we move that file to a safe place (e.g. SETTINGS/defect),
                               // we could delete it from the selection arraylist

            // edit: copy file to Backup/RelPath/FileName + Msg "Backup copied
            //   to .../NAME-DATE-TIME-Seconds.sgf" (NAME-2014.04.12-17h23.34.sgf) - so no overwriting of the original file
            // noEdit: move file to DefectFiles/RelPath/FileName + Msg "File moved to ..."

            if (btnClicked == 3 || btnClicked == 0 || btnClicked == -1) 
              return false; // and then skip!
            // OK_OPTION /YES_OPTION / [ESC] / [x] / just continue
            // - should lead to 'move file to $GGSETTINGS/brokenfiles/' or similar
            if (btnClicked == 2 ){ // NO_OPTION / text edit // and then we shouldn't continue with the next file, but
              // after editing, the problem should be loaded new to evaluate the result of the edits 
              // (but only, when the editor has been called successfully)
        
              boolean success = EditorStarter.editorStarter(GS.getSGFEditor2Str(), getCurrentProblemPath());
               // and then DON'T skip! // EditorStarter doesn't block the probFrame!
               // editTried = success; // here we should make a temp-backup of the file, 
               // and if the file date changed, we move the temp-backup to settings-backup (else, we delete that temp-backup)
              return false; // true;  should happen AFTER editing!
            }
            else if (btnClicked == 1){ // CANCEL_OPTION 
              parseSuccess = false; // we go once more through the loop, but the sgf code is replaced by the default code
              // exit or change selection is now simple
            }
          }
          catch(Exception e) {
            msg = Messages.getString("err_parsing_sgf_skipping") + " (any Exception) " + NL
                                   + Messages.getString("file_colon") + " \"" + filename + "\"" + NL
                                   + "We use the default code instead.";
            JOptionPane.showMessageDialog(panel, msg + "\n" + e.toString());
            ExceptionHandler.logCommonProblem(e, msg);
            parseSuccess = false;
          }
          if (!parseSuccess) 
            statusBar.setText(" SGF error, using default code.");
        }while(!parseSuccess);
        SGFParser.checkingSGF = false;
// ########## END LOOP #################
        panel.setBoard(board);
        board.setPanel(panel);
        if(!root.markPathsRIGHT())
            root.markPathsWV();

        boolean doFlip = GS.getFlip();
        boolean doFlipColors = GS.getFlipColors();
        
        int n = Main.rand.nextInt(4);
        boolean flip = ((n & 1) == 1) && doFlip;
        boolean flipC = ((n & 2) == 2) && doFlipColors;
        int rot = doFlip ? Main.rand.nextInt(4) : 0;
        
        root.flip(board.getSize(), flip, rot, flipC);
        Rectangle bounds;
        try{ 
          bounds = root.getBounds();  // ??: getBounds() - does this give the view port?
        }
        catch(SGFParseException e){
          JOptionPane.showMessageDialog(panel, e.getMessage()); // just zurechtgeflickt (~hacked)
          return false;
        }
        if (bounds == null) {bounds = new Rectangle(5, 5);} 
                            // interestingly, this doesn't give small boards - what does this rect do???
          // tinkered to work, when sgf data = "(;)" (valid sgf) or similar
        //  GobanPanel expects 1-based width & height
        bounds.width++;
        bounds.height++;
        panel.setSize(bounds, null);
        
        //  Process nodes until we come to one with moves as children
        root.execute(board);
        root.updateState(board);
        commentArea.setText(root.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
        commentArea.setCaretPosition(0);
        while(!root.hasChildMoves()) {
            SGFNode newRoot = (SGFNode)root.getFirstChild();
            if(newRoot == null)
                break;
            
            root = newRoot;
            root.setParent(null);
            root.execute(board);
            root.updateState(board);
            commentArea.setText(root.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
        }
        
        setCurColor(playerColor = root.getNextPlayer());
        root.setPlayer(playerColor * -1);
        hasResult = false;
        result = false;
        hasPlayedSound = false;
        
        curNode = root;
        
        panel.hideGhost();
        return true;
    }
    
    public void mouseClicked(int x, int y, int modifiers)
    {
        if((modifiers & MouseEvent.BUTTON3_MASK) != 0) {
          if(GS.getRightClickAdvance())
            nextProblem();
            return;
        }
        
        switch(mode)
        {
            case PLAY:
                playMouseClicked(x, y);
                break;
            case NAV_SOLN:
                navSolnMouseClicked(x, y);
                break;
        }
    }
    
    public void mouseWheelMoved(int numClicks) {
      numClicks *= -1;
      for(int i = 0; i < numClicks; i++)
        goBack();
    }
    
    private void playMouseClicked(int x, int y) {
        if(!enabled)
            return;
        
        if(!board.isLegalMove(curColor, x, y)) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        
        SGFNode next = curNode.getNextNode(new Point(x,y));
        
        if(next == null) {
            next = new SGFNode(curNode, x, y);
            curNode.addChild(next);
        }
        
        doNode(next);
        
        if(!next.isRight())
        {
          SGFNode resp = next.getResponse();
          if(resp != null) {
              enabled = false;
              SwingUtilities.invokeLater(new Responder(resp));
          }
        }
    }
    
    private void doNode(SGFNode n) {
        if(n.isRight()) {
            setResult(true);
        }
        else if(!n.isOnRightPath()) {
            setResult(false);
        }
        
        if(GS.getClickSoundEnabled())
            Main.clickSound.play();
        
        if(n.isRight()) {
            widgetPanel.setSolved(result, true);
            
            if(!hasPlayedSound && GS.getSoundEnabled()) {
                Main.rightSound.play();
                hasPlayedSound = true;
            }
        }
        else {
            if(GS.getShowWrongPath()) {
                if(!n.isOnRightPath()) {
                    widgetPanel.setSolved(result, false);
                    if(!hasPlayedSound && GS.getSoundEnabled()) {
                        Main.wrongSound.play();
                        hasPlayedSound = true;
                    }
                }
                else widgetPanel.clearSolved();
            }
            else {
                if(!n.hasChildrenFromFile()) {
                    widgetPanel.setSolved(result, false);
                    if(!hasPlayedSound && GS.getSoundEnabled()) {
                        Main.wrongSound.play();
                        hasPlayedSound = true;
                    }
                }
                else widgetPanel.clearSolved();
            }
        }

        n.execute(board);
        n.updateState(board);
        commentArea.setText(n.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
        curNode = n;
        setCurColor(curColor *= -1);
        
        if(GS.getAutoAdv() && n.isRight() && mode == PLAY) {
            enabled = false;
            SwingUtilities.invokeLater(new AutoAdvancer());
        }
    }
    
    private void navSolnMouseClicked(int x, int y) {
        SGFNode next = curNode.getNextNode(new Point(x,y));
        
        if(next == null) {
            if(!board.isLegalMove(curNode.getPlayer() * -1, x, y)) {
                Toolkit.getDefaultToolkit().beep();
                return;
            }
            
            next = new SGFNode(curNode, x, y);
            curNode.addChild(next);
        }
        
        doNode(next);
    }
    
    public void goBack()
    {
        do {
            SGFNode parent = (SGFNode)curNode.getParent();
            if(parent == null) {
                Toolkit.getDefaultToolkit().beep();
                return;
            }

            board.undoLast();
            curNode = parent;
            setCurColor(curColor *= -1);
            curNode.updateState(board);
            commentArea.setText(curNode.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
            
            if(GS.getShowWrongPath()) {
                if(!curNode.isOnRightPath())
                    widgetPanel.setSolved(result, false);
                else widgetPanel.clearSolved();
            }
            else widgetPanel.clearSolved();
            
            if(mode == NAV_SOLN || !curNode.isFromFile())
                break;
        } while(curNode.getPlayer() == playerColor);
    }
    
    public void restart()
    {
        mode = PLAY;
        panel.setNavMode(false);
        widgetPanel.clearSolved();
        SGFNode parent;
        hasPlayedSound = false;
        while((parent = (SGFNode)curNode.getParent()) != null) {
            board.undoLast();
            curNode = parent;
            setCurColor(curColor *= -1);
            curNode.updateState(board);
            commentArea.setText(curNode.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
        }
    }
    
    public void navigateSolution() {
        mode = NAV_SOLN;
        panel.setNavMode(true);
        setResult(false);
        
        widgetPanel.clearSolved();
        SGFNode parent;
        while((parent = (SGFNode)curNode.getParent()) != null) {
            board.undoLast();
            curNode = parent;
            setCurColor(curColor *= -1);
            curNode.updateState(board);
            commentArea.setText(curNode.getComment()); // ########### BEFORE: REMOVE ESCAPE CHARACTER "\" 
        }
    }
    
    public void setCommentArea(JTextArea a) { commentArea = a; }
    public void setWidgetPanel(WidgetPanel p) { widgetPanel = p; }
    public void setStatusBar(JLabel s) { statusBar = s; }
    
    public void loadProblem(int previousThisNext){ // previous = -1, reload same = 0, next = 1
        mode = PLAY;
        panel.setNavMode(false);
        widgetPanel.clearSolved();
        widgetPanel.enableNavButton(true);
        Selection selSets = GS.getSelectedSets();
        if (selSets == null)JOptionPane.showMessageDialog(null,"BAAAHHH! nextProblem, selSets (SelectedSets) = null");
          // when the grind.dat was damaged, we got an error on startup (but we should do something with this!!)
        ProbData p;
        if (previousThisNext == -1)
            p = selSets.getPrevProblem(); // todo: remember next problem's colors, sides, board orientation
        else if (previousThisNext == 1)
            p = selSets.getNextProblem(); // todo: remember previous problem's colors, sides, board orientation
        else{ // previousThisNext == 0
          p = selSets.reloadProblem(); // ####### // todo: don't switch colors, sides, board orientation
        }
        if (p == null)JOptionPane.showMessageDialog(null,"BAAAHHH! nextProblem, probdata = null");
        while(!startProblem(p)){ // ! -> not successful loaded
          if (previousThisNext == -1){
              p = selSets.getPrevProblem();
          }
          else if (previousThisNext == 1){
              p = selSets.getNextProblem();
          }    
          else {// previousThisNext == 0  // MIST!! Huston! We have a problem!
              msg = "File error with the choosen decoding!"; // ... Choose another please!";
              new SGFParseException(msg, p.charsetW.getCharset()); // could tell, if forFile, inFile, forFolder
              p = selSets.getNextProblem();
          }
        } // while
        if(editTried) { 
        // at the moment, editTried is always false, as the editor starter does not wait for the editor to finish 
          editTried = false;
          restart();
        }
        widgetPanel.setProblem(p);
        widgetPanel.setNumText(Messages.getString("SGFContr.solvedColon") + selSets.getCurIndex()
                                + " " + Messages.getString("of") 
                                + " " + selSets.getNumSelected());
        widgetPanel.setPctRight((double)selSets.getNumRight() / selSets.getNumSolved());      
    }
    
    public void nextProblem() {  // previous = -1, reload same = 0, next = 1
    // (next/previous problem: nearly the same code)
      loadProblem(1);
      return;
    }
    
    public void prevProblem() {  // previous = -1, reload same = 0, next = 1
     // (next/previous problem: nearly the same code)
      loadProblem(-1);
    }
    
    public boolean findProblem(String sub) {
        Selection selSets = GS.getSelectedSets();
        ProbData old = selSets.getCurProb();
        
        ProbData p = selSets.findProblem(sub);
        if(p == null)
            return false;
        
        mode = PLAY;
        panel.setNavMode(false);
        widgetPanel.clearSolved();
        widgetPanel.enableNavButton(true);
        
        if(!startProblem(p)) {
            p = old;
            startProblem(p);
        }
        
        widgetPanel.setProblem(p);
        widgetPanel.setNumText(selSets.getCurIndex() + " " + Messages.getString("of") 
                                + " " + selSets.getNumSelected());
        widgetPanel.setPctRight((double)selSets.getNumRight() / selSets.getNumSolved());
        
        return true;
    }
    
    public void setResult(boolean r) {
        if(!hasResult) {
            result = r;
            hasResult = true;
            GS.getSelectedSets().probDone(r);
        }
    }
    
    private void setCurColor(int c) {
        curColor = c;
        widgetPanel.setToPlay(curColor);
        panel.setGhostColor(curColor);
    }
    
    public String getCurrentProblemPath() { return GS.getSelectedSets().getCurrentProblemPath(); }
    
    class Responder implements Runnable{
        SGFNode node;
        
        Responder(SGFNode n) { this.node = n; }
        
        public void run() {
            try { Thread.sleep(250); } catch(Exception e) { /* */ }
            doNode(node);
            enabled = true;
        }
    }
    
    class AutoAdvancer implements Runnable {
        public void run() {
            try { Thread.sleep(500); } catch(Exception e) { /* */ }
            enabled = true;
            nextProblem();
        }
    }
}
