#!/bin/sh

# Copyright (C) 2023-2024 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr/pkg"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/tmp/pkgbuild/net/kea/work.riscv64/kea-2.6.3/src/share/database/scripts/pgsql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/tmp/pkgbuild/net/kea/work.riscv64/kea-2.6.3/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

VERSION=$(pgsql_version "$@")

if [ "$VERSION" != "18.0" ]; then
    printf 'This script upgrades 18.0 to 19.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${VERSION}"
    exit 0
fi

psql "$@" >/dev/null <<EOF
START TRANSACTION;

-- This line starts the schema upgrade to version 19.0.

-- Convert ddns-use-conflict-resolution to ddns-conflict-resolution-mode
SELECT set_config('kea.disable_audit', 'true', false);
UPDATE dhcp4_global_parameter
SET name = 'ddns-conflict-resolution-mode', value = 'check-with-dhcid', parameter_type = 4
WHERE name = 'ddns-use-conflict-resolution' and value = '1';

UPDATE dhcp4_global_parameter
SET name = 'ddns-conflict-resolution-mode', value = 'no-check-with-dhcid', parameter_type = 4
WHERE name = 'ddns-use-conflict-resolution' and value = '0';

UPDATE dhcp6_global_parameter
SET name = 'ddns-conflict-resolution-mode', value = 'check-with-dhcid', parameter_type = 4
WHERE name = 'ddns-use-conflict-resolution' and value = '1';

UPDATE dhcp6_global_parameter
SET name = 'ddns-conflict-resolution-mode', value = 'no-check-with-dhcid', parameter_type = 4
WHERE name = 'ddns-use-conflict-resolution' and value = '0';

-- Clear lease6_relay_id table.
DELETE FROM lease6_relay_id;

-- Change lease6_relay_id:lease_addr to INET.
ALTER TABLE lease6_relay_id ALTER COLUMN lease_addr TYPE INET USING '::';

-- Add a constraint on lease6_relay_id that any lease_addr must map to
-- a lease6 address.
ALTER TABLE lease6_relay_id
    ADD CONSTRAINT fk_lease6_relay_id_addr FOREIGN KEY (lease_addr)
    REFERENCES lease6 (address) ON DELETE CASCADE ON UPDATE NO ACTION;

-- Clear lease6_remote_id table.
DELETE FROM lease6_remote_id;

-- Change lease6_remote_id:lease_addr to INET.
ALTER TABLE lease6_remote_id ALTER COLUMN lease_addr TYPE INET USING '::';

-- Add a constraint on lease6_remote_id that any lease_addr must map to
-- a lease6 address.
ALTER TABLE lease6_remote_id
    ADD CONSTRAINT fk_lease6_remote_id_addr FOREIGN KEY (lease_addr)
    REFERENCES lease6 (address) ON DELETE CASCADE ON UPDATE NO ACTION;

-- Update the schema version number.
UPDATE schema_version
    SET version = '19', minor = '0';

-- This line concludes the schema upgrade to version 19.0.

-- Commit the script transaction.
COMMIT;

EOF
