package TDS::Cache::Daily;
# $Id: Daily.pm,v 1.33 2000/12/20 14:59:41 tom Exp $
################################################################

=head1 NAME

TDS::Cache::Daily - daily cache class

=head1 SYNOPSIS

 use TDS::Cache::Daily;

 $d = new TDS::Cache::Daily(year=>1999, month=>10, day=>22);
 if ($d->Read){
    print $d->html;
 }
 ...
 $d->Write($html);

=cut


################################################################

use strict;
use Time::Local;

use DateTime::Date;

use TDS::System;
use TDS::DirInfo;
use TDS::Cache::Diary;

use vars qw(@ISA @EXPORT
	    $SurvivalDays);

@ISA = qw(DateTime::Date TDS::Cache::Diary);
@EXPORT = qw(attributes);

=head1 MEMBER VARIABLES

 year,month,day    date
 reverse           if it's reverse mode
 html              HTML text as cache

=cut

attributes qw(reverse);

$SurvivalDays = 30 unless defined $SurvivalDays;

################################################################
sub initialize($)
{
    my $self = shift;

    $self->DateTime::Date::initialize;
    $self->TDS::Cache::Diary::initialize;
}
################################################################

=head2 $d->IsFresh

whether if the cache is fresh.
conditions for beging regarded as fresh:

 * if already exist, newer than correspond tdf file, and
 * of this month, or recent few days(CanSurvive)

=cut

sub IsFresh ($)
{
    my $self = shift;

    return 0 unless $self->SUPER::IsFresh;
    return 0 unless $self->CanSurvive;
    return 1;
}

#sub NewerThanTdf($)
sub GetDataFilename($)
{
    my $self = shift;
    
    return sprintf("%s/%04d/%02d/%02d.tdf",
		   GetDiaryDir(),
		   $self->year, $self->month, $self->day);
}

################################################################
sub WriteCache($$)
{
    my ($self, $line) = @_;

    return unless $self->CanSurvive;

    $self->SUPER::WriteCache($line);
}

################################################################	     

=head2 $d->CanSurvive

where if it's fresh enough to survive as cache.

conditions for surviving:

 * of this month, or recent few days(CanSurvive)

=cut

sub CanSurvive ($)
{
    my $self = shift;

    return 1 if $SurvivalDays < 0;    # always survival
    
    my ($now_year, $now_month) = ($TDS::Status->start_time->year,
				  $TDS::Status->start_time->month);

    # check survival days
    my $time = $TDS::Status->start_time->Dup;
    my $svv_d = $SurvivalDays;
    $svv_d--;
    $time->Decrement("${svv_d}D");
    return 1 if ($time <= $self);
#    my $survival = ($time <= $self);

    # even if before survival days,
    # in this month, it CAN survive.
    return 1 if $self->year == $now_year && $self->month == $now_month;

    # if can NOT be survive, remove it
    $self->RemoveCache;
    return 0;

}
################################################################
sub GetCacheFilename($)
{
    my $self = shift;

    my $prefix = ($self->reverse) ? 'r' : 'c';
    return
	$self->create_cache_filename(sprintf("%s%04d%02d%02d",
					     $prefix,
					     $self->year,
					     $self->month,
					     $self->day));
}

1;
    
 
