class PlasterError(Exception):
    """
    A base exception for any error generated by plaster.
    """


class InvalidURI(PlasterError, ValueError):
    """
    Raised by :func:`plaster.parse_uri` when failing to parse a ``config_uri``.

    :ivar uri: The user-supplied ``config_uri`` string.

    """

    def __init__(self, uri, message=None):
        if message is None:
            message = f'Unable to parse config_uri "{uri}".'
        super().__init__(message)
        self.message = message
        self.uri = uri


class LoaderNotFound(PlasterError, ValueError):
    """
    Raised by :func:`plaster.get_loader` when no loaders match the requested
    ``scheme``.

    :ivar scheme: The scheme being matched.
    :ivar protocols: Zero or more :term:`loader protocol` identifiers that
        were requested when finding a loader.

    """

    def __init__(self, scheme, protocols=None, message=None):
        if message is None:
            scheme_msg = f'scheme "{scheme}"'
            if protocols is not None:
                scheme_msg += ', protocol "{}"'.format(", ".join(protocols))
            message = f"Could not find a matching loader for the {scheme_msg}."
        super().__init__(message)
        self.message = message
        self.scheme = scheme
        self.protocols = protocols


class MultipleLoadersFound(PlasterError, ValueError):
    """
    Raised by :func:`plaster.get_loader` when more than one loader matches the
    requested ``scheme``.

    :ivar scheme: The scheme being matched.
    :ivar protocols: Zero or more :term:`loader protocol` identifiers that
        were requested when finding a loader.
    :ivar loaders: A list of :class:`plaster.ILoaderInfo` objects.

    """

    def __init__(self, scheme, loaders, protocols=None, message=None):
        if message is None:
            scheme_msg = f'scheme "{scheme}"'
            if protocols is not None:
                scheme_msg += ', protocol "{}"'.format(", ".join(protocols))
            loader_list = ", ".join(
                loader.scheme for loader in sorted(loaders, key=lambda v: v.scheme)
            )
            message = (
                "Multiple plaster loaders were found for {}. "
                "Please specify a more specific config_uri. "
                "Matched loaders: {}"
            ).format(scheme_msg, loader_list)
        super().__init__(message)
        self.message = message
        self.scheme = scheme
        self.protocols = protocols
        self.loaders = loaders
