/*
 * control socket
 *
 * $Id: control.c,v 1.10 2002/03/26 10:46:59 taka Exp $
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/stat.h>

#include "queue.h"
#include "aime.h"
#include "cmd.h"
#include "bufsize.h"
#include "dencode.h"
#include "control.h"

#ifndef SUN_LEN
#define SUN_LEN(s) \
	(sizeof (*(s)) - sizeof ((s)->sun_path) + strlen ((s))->sun_path)
#endif

static int control_socket_fd = -1;
#define BUFFERSIZE (CONTROL_BUFFER_LINE_SIZE * 2)
char en_de_buf[BUFFERSIZE + 1];
struct uis
{
	char readbuf[BUFFERSIZE + 1];
	int readlen;

	int  fd;

	SLIST_ENTRY (uis)	entry;
};
static SLIST_HEAD (,uis) uiheads = SLIST_HEAD_INITIALIZER (0);


void
control_write (const char* cmd)
{
	int len;
	struct uis* ui;

	len = strlen (cmd);
	if (len > BUFFERSIZE) {
		error_printf ("can't send so long message\n");
		return;
	};

	encode (cmd, en_de_buf);
	strcat (en_de_buf, "\n");
	cmd = en_de_buf;

	len = strlen (cmd);
	ui = SLIST_FIRST (&uiheads);
	while (ui) {
		int ret;
		struct uis* next = SLIST_NEXT (ui, entry);

		ret = write (ui->fd, cmd, len);
		if (ret == -1) {
			close (ui->fd);
			del_fd_watch (ui->fd);
			SLIST_REMOVE (&uiheads, ui, uis, entry);
			free (ui);
		}
		ui = next;
	}	
}
static void
control_read (int fd, void* data)
{
	struct uis* ui = (struct uis*)data;
	char line[BUFFERSIZE];
	int ret;
	char* str;
	int len;
	char* l;

	ret = read (fd, ui->readbuf + ui->readlen,
		    sizeof (ui->readbuf) - ui->readlen);
	if (ret == -1 || ret == 0) {
		DPRINTF ("ui is dead.\n");
		del_fd_watch (fd);
		SLIST_REMOVE (&uiheads, ui, uis, entry);
		close (fd);
		free (ui);
		return;
	}
	ui->readlen += ret;

	memcpy (line, ui->readbuf, sizeof (line));
	line[ui->readlen] = '\0';

	l = line;
	while ((str = index (l, '\n')) != NULL) {
		*str = '\0';
		len = str - l + 1;

		memmove (ui->readbuf, ui->readbuf + len, ui->readlen - len);
		ui->readlen -= len;

		decode (l, en_de_buf);
		cmd_exec (en_de_buf);

		l = str + 1;
	}		

	if (ui->readlen == sizeof (ui->readbuf)) {
		error_printf ("can't find newline\n");
		ui->readlen = 0;
	}
}

static void
control_accept (int fd, void* data)
{
	int newfd;
	struct sockaddr_un u;
	struct uis* ui;
	int ulen = sizeof (u);

	newfd = accept (fd, (struct sockaddr*)&u, &ulen);
	if (newfd == -1) {
		perror ("accept");
		return;
	}

	ui = malloc (sizeof (struct uis));
	if (!ui) {
		NOMEMORY ("can't allocate uis\n");
		close (newfd);
		return;
	}

	ui->fd = newfd;
	ui->readlen  = 0;

	add_fd_watch (ui->fd, FD_WATCH_READ, control_read, ui);
	SLIST_INSERT_HEAD (&uiheads, ui, entry);
}
	

int
control_init (const char* name)
{
	int fd;
	struct sockaddr_un s;
	int ret;
	struct stat statbuf;

	if (control_socket_fd != -1) return -1;

	if (!name) name = CONTROL_DEFAULT_SOCKET_NAME;

	fd = socket (PF_UNIX, SOCK_STREAM, 0);
	if (fd < 0) {
		perror ("socket");
		return -1;
	}

	strncpy (s.sun_path, name, sizeof (s.sun_path));
	s.sun_path[sizeof (s.sun_path) - 1] = '\0';
	s.sun_family = AF_UNIX;

	ret = bind (fd, (struct sockaddr*)&s, SUN_LEN (&s));
	if (ret == 0) {
		/* success!! */
		ret = listen (fd, 5);
		if (ret == -1) {
			perror ("listen");
			close (fd);
			return -1;
		}
		control_socket_fd = fd;
		goto END;
	}
	close (fd);


	if (errno != EADDRINUSE) {
		perror ("bind");
		return -1;
	}

	/*
	 * address already in use.
	 * check other ximserver is running or not.
	 *
	 * first check file is socket or not.
	 */
	ret = stat (name, &statbuf);
	if (ret == -1) {
		perror ("stat");
		return -1;
	}
	if ((statbuf.st_mode & S_IFMT) != S_IFSOCK) {
		/* not socket!! */
		return -1;
	}

	/*
	 * already running server?
	 */
	fd = socket (PF_UNIX, SOCK_STREAM, 0);
	if (fd == -1) {
		perror ("socket");
		return -1;
	}
	ret = connect (fd, (struct sockaddr*)&s, SUN_LEN (&s));
	if (ret >= 0) {
		/* succeeded */
		fd_set readset;
		struct timeval t = {2, 0};

#define ECHO "echo\n"
		ret = write (fd, ECHO, strlen (ECHO));
		if (ret == -1) {
			close (fd);
			return -1;
		}

		FD_ZERO (&readset);
		FD_SET (fd, &readset);
		ret = select (fd + 1, &readset, NULL, NULL, &t);
		if (ret == -1) {
			/* error */
			close (fd);
			return -1;

		} else if (ret == 0) {
			/* timeout */

		} else {
			/* there is reply */
			char buf[16];

			ret = read (fd, buf, sizeof (buf));
			if (ret == -1) {
				close (fd);
				return -1;
			}
			buf[ret] = '\0';

			if (strcmp ("\n", buf) == 0) {
				close (fd);
				return -1;
			}
		}
	}
	close (fd);

	ret = unlink (name);
	if (ret == -1) {
		perror ("unlink");
		return -1;
	}
	fd = socket (PF_UNIX, SOCK_STREAM, 0);
	if (fd == -1) {
		perror ("socket");
		return -1;
	}
	ret = bind (fd, (struct sockaddr*)&s, SUN_LEN (&s));
	if (ret == -1) {
		perror ("bind");
		close (fd);
		return -1;
	}
	ret = listen (fd, 5);
	if (ret == -1) {
		perror ("listen");
		close (fd);
		return -1;
	}
	control_socket_fd = fd;

 END:
	add_fd_watch (control_socket_fd, FD_WATCH_READ, control_accept, NULL);

	{
		struct sigaction act, oact;

		act.sa_handler = SIG_IGN;
		sigemptyset (&act.sa_mask);
		act.sa_flags = SA_RESTART;
		sigaction (SIGPIPE, &act, &oact);
	}
	return 0;
}

void
control_mode (int mode, int onoff)
{
	char buffer[16];
	const char* modename[INPUT_MODE_NUM]
		= {"ascii", "hira", "kata", "zen_ascii", "kana",};

	if (mode >= 0 && mode < INPUT_MODE_NUM) {
		snprintf (buffer, sizeof (buffer), "mode %s", modename[mode]);
		control_write (buffer);
	}

	if (onoff >= 0) control_write (onoff ? "on" : "off");
}
