/*
 * deal imconnection
 *
 * $Id: imconnection.c,v 1.25 2002/03/26 10:46:59 taka Exp $
 */
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <stdlib.h>
#include <string.h>

#include "aime.h"
#include "xim.h"
#include "imconnection.h"
#include "ximpacket.h"
#include "ximic.h"
#include "conversion.h"
#include "ximproc.h"
#include "byteorder.h"
#include "pewindow.h"
#include "control.h"


static struct imrequest imrequests[] =
{
	{"CONNECT",          XIM_CONNECT,          0, ximp_connect},
	{"CONNECT-REPLY",    XIM_CONNECT_REPLY,    0, ximp_bad_protocol},
	{"DISCONNECT",       XIM_DISCONNECT,       0, ximp_disconnect},
	{"DISCONNECT-REPLY", XIM_DISCONNECT_REPLY, 0, ximp_bad_protocol},
	{"AUTH-REQUIRED",    XIM_AUTH_REQUIRED,    0, ximp_bad_protocol},
	{"AUTH-REPLY",       XIM_AUTH_REPLY,       0, ximp_bad_protocol},
	{"AUTH-NEXT",        XIM_AUTH_NEXT,        0, ximp_bad_protocol},
	{"AUTH-SETUP",       XIM_AUTH_SETUP,       0, ximp_bad_protocol},
	{"AUTH-NG",          XIM_AUTH_NG,          0, ximp_bad_protocol},
	{"ERROR",            XIM_ERROR,            0, ximp_error},
	{"OPEN",             XIM_OPEN,             0, ximp_open},
	{"OPEN-REPLY",       XIM_OPEN_REPLY,       0, ximp_bad_protocol},
	{"CLOSE",            XIM_CLOSE,            0, ximp_close},
	{"CLOSE-REPLY",      XIM_CLOSE_REPLY,      0, ximp_bad_protocol},

	{"REGISTER-TRIGGERKEYS", XIM_REGISTER_TRIGGERKEYS, 0,
	 ximp_bad_protocol},
	{"TRIGGER-NOTIFY",       XIM_TRIGGER_NOTIFY,       0, NULL},/*ignore*/
	{"TRIGGER-NOTIFY-REPLY", XIM_TRIGGER_NOTIFY_REPLY, 0,
	 ximp_bad_protocol},
	{"SET-EVENT-MASK",       XIM_SET_EVENT_MASK,       0,
	 ximp_bad_protocol},
	{"ENCODING-NEGOTIATION", XIM_ENCODING_NEGOTIATION, 0,
	 ximp_encoding_negotiation},
	{"ENCODING-NEGOTIATION-REPLY", XIM_ENCODING_NEGOTIATION_REPLY, 0,
	 ximp_bad_protocol},
	{"QUERY-EXTENSION",  XIM_QUERY_EXTENSION,  0, ximp_query_extension},
	{"QUERY-EXTENSION-REPLY", XIM_QUERY_EXTENSION_REPLY, 0,
	 ximp_bad_protocol},
	{"SET-IM-VALUES",       XIM_SET_IM_VALUES,       0, NULL},/* ignore */
	{"SET-IM-VALUES-REPLY", XIM_SET_IM_VALUES_REPLY, 0, ximp_bad_protocol},
	{"GET-IM-VALUES",       XIM_GET_IM_VALUES,       0, ximp_get_imvalues},
	{"GET-IM-VALUES-REPLY", XIM_GET_IM_VALUES_REPLY, 0, ximp_bad_protocol},
	{"CREATE-IC",           XIM_CREATE_IC,           0, ximp_createic},
	{"CREATE-IC-REPLY",     XIM_CREATE_IC_REPLY,     0, ximp_bad_protocol},
	{"DESTROY-IC",          XIM_DESTROY_IC,          0, ximp_destroyic},
	{"DESTROY-IC-REPLY",    XIM_DESTROY_IC_REPLY,    0, ximp_bad_protocol},
	{"SET-IC-VALUES",       XIM_SET_IC_VALUES,       0, ximp_set_icvalues},
	{"SET-IC-VALUES-REPLY", XIM_SET_IC_VALUES_REPLY, 0, ximp_bad_protocol},
	{"GET-IC-VALUES",       XIM_GET_IC_VALUES,       0, ximp_get_icvalues},
	{"GET-IC-VALUES-REPLY", XIM_GET_IC_VALUES_REPLY, 0, ximp_bad_protocol},
	{"SET-IC-FOCUS",        XIM_SET_IC_FOCUS,        0, ximp_set_icfocus},
	{"UNSET-IC-FOCUS",      XIM_UNSET_IC_FOCUS,      0,
	 ximp_unset_icfocus},
	{"FORWARD-EVENT",       XIM_FORWARD_EVENT,       0,
	 ximp_forward_event},

	{"SYNC",                XIM_SYNC,                0, ximp_sync},
	{"SYNC-REPLY",          XIM_SYNC_REPLY,          0, NULL},
	{"COMMIT",              XIM_COMMIT,              0, ximp_bad_protocol},
	{"RESET-IC",            XIM_RESET_IC,            0, ximp_resetic},

	{"RESET-IC-REPLY",   XIM_RESET_IC_REPLY,   0, ximp_bad_protocol},
	{"GEOMETRY",         XIM_GEOMETRY,         0, ximp_bad_protocol},
	{"STR-CONVERSION",   XIM_STR_CONVERSION,   0, ximp_bad_protocol},
	{"STR-CONVERSION-REPLY", XIM_STR_CONVERSION_REPLY, 0,
	 ximp_bad_protocol},
	{"PREEDIT-START",    XIM_PREEDIT_START,    0, ximp_bad_protocol},
	{"PREEDIT-START-REPLY", XIM_PREEDIT_START_REPLY, 0,
	 NULL}, /* ignore */
	{"PREEDIT-DRAW",     XIM_PREEDIT_DRAW,     0, ximp_bad_protocol},
	{"PREEDIT-CARET",    XIM_PREEDIT_CARET,    0, ximp_bad_protocol},
	{"PREEDIT-CARET-REPLY", XIM_PREEDIT_CARET_REPLY, 0,
	 ximp_bad_protocol},
	{"PREEDIT-DONE",     XIM_PREEDIT_DONE,     0, ximp_bad_protocol},
	{"STATUS-START",     XIM_STATUS_START,     0, ximp_bad_protocol},
	{"STATUS-DRAW",      XIM_STATUS_DRAW,      0, ximp_bad_protocol},
	{"STATUS-DONE",      XIM_STATUS_DONE,      0, ximp_bad_protocol},
	{"PREEDITSTATE",     XIM_PREEDITSTATE,     0, ximp_bad_protocol},
	{NULL, 0, 0, NULL},
};

struct imrequest extrequests[] =
{
	{"XIM_EXT_SET_EVENT_MASK", XIM_EXT_SET_EVENT_MASK, 0,
	 ximp_bad_protocol},
	{"XIM_EXT_FORWARD_KEYEVENT", XIM_EXT_FORWARD_KEYEVENT, 0, },
	{"XIM_EXT_MOVE", XIM_EXT_MOVE, 0, },
	{NULL, 0, 0, NULL},
};
struct imrequest*
get_extrequest_table (void)
{
	return extrequests;
}

#define IMC_GET8(imc,y)  get8  ((imc)->buffer + (y), (imc)->byte_order)
#define IMC_GET16(imc,y) get16 ((imc)->buffer + (y), (imc)->byte_order)
#define IMC_GET32(imc,y) get32 ((imc)->buffer + (y), (imc)->byte_order)

#define XIM_MAJOR_PROTOCOL_VERSION (1)
#define XIM_MINOR_PROTOCOL_VERSION (0)

struct imconnection*
imc_allocate (void)
{
	struct imconnection* imc;
	imc = malloc (sizeof (struct imconnection));
	if (!imc)
		NOMEMORY ("can't allocate imconncection");
	return imc;
}

void
imc_free (struct imconnection* p)
{
	struct ximim* xim;
	struct ximpacket* x;

	if (!p) return;

	xim = SLIST_FIRST (&p->imlist);
	while (xim) {
		SLIST_REMOVE_HEAD (&p->imlist, entry);
		ximim_destroy (xim);
		xim = SLIST_FIRST (&p->imlist);
	}

	x = SIMPLEQ_FIRST (&p->packetlist);
	while (x) {
		SIMPLEQ_REMOVE_HEAD (&p->packetlist, x, entry);
		ximpacket_destroy (x);
		x = SIMPLEQ_FIRST (&p->packetlist);
	}

	XDestroyWindow (p->display, p->comwindow);
	free (p);
}

Window
imc_comwindow (struct imconnection* p)
{
	return p->comwindow;
}

static void
shiftbuffer (struct imconnection* imc, int nbytes)
{
	memmove (imc->buffer, imc->buffer + nbytes, imc->buffer_len - nbytes);
	imc->buffer_len -= nbytes;
}

static int
checkHeader (struct imconnection* imc, int* major, int* minor, int* l)
{
	int len;
	if (imc->buffer_len < 4) return -1;

	len = IMC_GET16 (imc, 2) * 4;
	if (imc->buffer_len >= len + 4) {
		*major = IMC_GET8 (imc, 0);
		*minor = IMC_GET8 (imc, 1);
		*l     = len;
		shiftbuffer (imc, 4);
		return 0;
	}
	return -1;
}

static struct imrequest* requesttable[XIM_DISPATCHER_MAX];
static int
convert_imrequests (void)
{
	struct imrequest* r;
	for (r = imrequests; r->name != NULL; r++) {
		if (r->major < 0
		    || r->major > NUMBER_OF_ELEMENTS (requesttable)) {
			return -1;
		}
		requesttable[r->major] = r;
	}
	for (r = extrequests; r->name != NULL; r++) {
		if (r->major < 0
		    || r->major > NUMBER_OF_ELEMENTS (requesttable)) {
			return -1;
		}
		requesttable[r->major] = r;
	}
	return 0;
}
static void
print_trace (const char* prompt, unsigned char* buf, int len)
{
#if 0
	int i;
	DPRINTF ("%s", prompt);
	for (i = 0; i < len; i++) DPRINTF ("%02x ", buf[i]);
	DPRINTF ("\n");
#endif
}
static int
maindispatcher (struct imconnection* imc)
{
	int ret;
	int major, minor, len;

	ret = checkHeader (imc, &major, &minor, &len);
	if (ret == -1) return 0;

	if (debug_mode) print_trace ("recv: ", imc->buffer, imc->buffer_len);
	if (major < 0
	    || major > NUMBER_OF_ELEMENTS (requesttable)) return 0;

	DPRINTF ("recv xim protocol[%s:(%d %d)]\n",
		 requesttable[major]->name,
		 requesttable[major]->major,
		 requesttable[major]->minor);
	if (requesttable[major]->proc) {
		ret = requesttable[major]->proc (imc, major, minor, len);
		if (ret == -1) return -1;
	}
	imc->buffer_len = 0;
	return 0;
}
static int
send_packet (struct imconnection* imc, struct ximpacket* x)
{
	XClientMessageEvent e;
	int length = x->buffer_len;
	unsigned char* data = x->buffer;

	if (length < XTRANSPORT_SIZE) {
		/* use event */
		while (length > XTRANSPORT_UNIT_SIZE) {
			e.type   = ClientMessage;
			e.window = imc->clientwindow;
			e.format = 8;
			e.message_type = xim_get_moredata_atom ();
			memcpy (e.data.b, data, XTRANSPORT_UNIT_SIZE);

			xim_xsendevent (imc->display, imc->clientwindow, False,
					NoEventMask, (XEvent*)&e);
			data   += XTRANSPORT_UNIT_SIZE;
			length -= XTRANSPORT_UNIT_SIZE;
		}
		e.type         = ClientMessage;
		e.window       = imc->clientwindow;
		e.format       = 8;
		e.message_type = xim_get_protocol_atom ();
		memset (e.data.b, 0, XTRANSPORT_UNIT_SIZE);
		memcpy (e.data.b, data, length);
		xim_xsendevent (imc->display, imc->clientwindow, False,
				NoEventMask, (XEvent*)&e);

	} else {
		/* use property */
		e.type         = ClientMessage;
		e.window       = imc->clientwindow;
		e.format       = 32;
		e.message_type = xim_get_protocol_atom ();
		e.data.l[0]    = length;
		e.data.l[1]    = xim_get_comm_atom ();

		xim_xchangeproperty (imc->display, imc->clientwindow,
				     xim_get_comm_atom (), XA_STRING, 8,
				     PropModeAppend, data, length);
		xim_xsendevent (imc->display, imc->clientwindow, False,
				NoEventMask, (XEvent*)&e);
	}
	XFlush (imc->display);
	return 0;
}
void
imc_send_packets (struct imconnection* imc)
{
	struct ximpacket* x;
	x = SIMPLEQ_FIRST (&imc->packetlist);
	while (x) {
		struct ximpacket* next = SIMPLEQ_NEXT (x, entry);

		DPRINTF ("send xim protocol[%s:(%d %d %d(%lu))]\n",
			 requesttable[(int)x->buffer[0]]->name,
			 requesttable[(int)x->buffer[0]]->major,
			 requesttable[(int)x->buffer[0]]->minor,
			 get16 (x->buffer + 2, imc->byte_order),
			 x->buffer_len
			);

		send_packet (imc, x);
		print_trace ("send: ", x->buffer, x->buffer_len);

		SIMPLEQ_REMOVE_HEAD (&imc->packetlist, x, entry);
		ximpacket_destroy (x);
		x = next;
	}
}

static int
initdispatcher (struct imconnection* imc)
{
	int ret;
	int major, minor, len;
	int num_auth;
	struct ximpacket* ximp;

	ximp = ximpacket_allocate ();
	if (ximp == NULL) return -1;
	
	if (imc->byte_order == BYTE_ORDER_UNKNOWN) {
		if (imc->buffer_len < 5) goto NG;
		if (imc->buffer[0] != XIM_CONNECT) goto NG;

		switch (imc->buffer[4]) {
		case 'B':
			imc->byte_order = BYTE_ORDER_MSB_FIRST;
			break;
		case 'l':
			imc->byte_order = BYTE_ORDER_LSB_FIRST;
			break;
		default:
			goto NG;
		}
	}

	ret = checkHeader (imc, &major, &minor, &len);
	if (ret == -1) {
		ximpacket_destroy (ximp);
		return 0;
	}
	imc->buffer_len = 0;

	if (major != XIM_CONNECT || minor != 0 || len < 8) goto NG;

	imc->major_protocol = IMC_GET16 (imc, 2);
	imc->minor_protocol = IMC_GET16 (imc, 4);

	if (imc->major_protocol > XIM_MAJOR_PROTOCOL_VERSION
	    || (imc->major_protocol == XIM_MAJOR_PROTOCOL_VERSION
		&& imc->minor_protocol > XIM_MINOR_PROTOCOL_VERSION)) {
		goto NG;
	}

	num_auth = IMC_GET16 (imc, 6);
	if (num_auth > 0) {
		/* never implemented? */
		goto NG;
	} else {
		/* send reply */
		ret = 0;
		ret |= ximpacket_putheader (ximp, XIM_CONNECT_REPLY, 0, 4,
					    imc->byte_order);
		ret |= ximpacket_put16 (ximp, XIM_MAJOR_PROTOCOL_VERSION,
					imc->byte_order);
		ret |= ximpacket_put16 (ximp, XIM_MINOR_PROTOCOL_VERSION,
					imc->byte_order);
		if (ret == -1) {
			ximpacket_destroy (ximp);
			return -1;
		}
		SIMPLEQ_INSERT_TAIL (&imc->packetlist, ximp, entry);
	}

	imc->dispatch = maindispatcher;
	return 0;

 NG:
	imc->state = IMC_STATE_CLOSE;
	ret = ximpacket_simple_request (ximp, XIM_AUTH_NG, 0);
	if (ret == -1) {
		ximpacket_destroy (ximp);
		return -1;
	}
	SIMPLEQ_INSERT_TAIL (&imc->packetlist, ximp, entry);
	return -1;
}

static int
xinput (struct imconnection* imc, XEvent* ev)
{
	XClientMessageEvent* e = (XClientMessageEvent*)ev;
	if (e->format == 32) {
		unsigned long offset = 0;
		unsigned long remain;

		do {
			Atom type;
			int format;
			unsigned long nitems;
			char* data;

			XGetWindowProperty (e->display, e->window,
					    e->data.l[1], offset,
					    IMC_BUFFER_SIZE - imc->buffer_len,
					    True, AnyPropertyType, &type,
					    &format, &nitems, &remain,
					    (unsigned char**)&data);

			if (!data) return -1;

			if (format == 8) {
				memcpy (imc->buffer + imc->buffer_len, data,
					nitems);
				imc->buffer_len += nitems;
				offset += nitems;
			} else {
				return -1;
			}
			XFree (data);
		} while (remain > 0);

	} else if (e->format == 8) {
		if (imc->buffer_len + XTRANSPORT_UNIT_SIZE
		    >= IMC_BUFFER_SIZE) {
			return -1;
		}
		memcpy (imc->buffer + imc->buffer_len, e->data.b,
			XTRANSPORT_UNIT_SIZE);
		imc->buffer_len += XTRANSPORT_UNIT_SIZE;

	} else {
		return -1;
	}
	return 0;
}

int
imc_xevent (struct imconnection* imc, XEvent* e)
{
	int ret;
	if (imc->display != e->xany.display) return 0;

	if (e->xany.window == imc->comwindow) {
		switch (e->type) {
		case ClientMessage:
			ret = xinput (imc, e);
			if (ret == -1) break;

			ret = imc->dispatch (imc);
			if (ret == -1) break;
			imc_send_packets (imc);
			break;
		default:
			break;
		}
	} else if (e->xany.window == imc->clientwindow) {
		switch (e->type) {
		case DestroyNotify:
			/* remove this imconnection */
			DPRINTF ("receive DestroyNotify: %lx\n",
				 imc->clientwindow);
			control_mode (-1, 0);
			return -1;
			break;
		default:
			break;
		}
	} else {
		struct ximim* xim;
		SLIST_FOREACH (xim, &imc->imlist, entry) {
			struct ximic* xic;
			SLIST_FOREACH (xic, &xim->iclist, entry) {
				ret = pewindow_proc_xevent (xic->pew, e);
				if (ret == 1) break;
			}
		}
	}
	return 0;
}

void
imc_init (struct imconnection* imc, Display* d, Window p, Window cl)
{
	imc->display      = d;
	imc->clientwindow = cl;
	imc->comwindow    = XCreateSimpleWindow (d, p, 0, 0, 1, 1, 0, 0, 0);
	
	imc->buffer_len = 0;
	imc->dispatch   = initdispatcher;
	imc->byte_order = BYTE_ORDER_UNKNOWN;

	SIMPLEQ_INIT (&imc->packetlist);
	SLIST_INIT (&imc->imlist);

	imc->state = IMC_STATE_INIT;

	XSelectInput (d, cl, StructureNotifyMask);
}

int
imc_module_initialize (void)
{
	return convert_imrequests ();
}
