/*
 * xim initialize routines and X11 utilities
 *
 * $Id: xim.c,v 1.20 2002/03/26 10:47:00 taka Exp $
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xproto.h>

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "aime.h"
#include "xim.h"
#include "imconnection.h"
#include "queue.h"
#include "byteorder.h"
#include "ximic.h"
#include "conversion.h"
#include "pewindow.h"
#include "fontset.h"
#include "control.h"

#define SERVERTRANSPORT_MAJORVERSION (0)
#define SERVERTRANSPORT_MINORVERSION (2)

static struct ximserver
{
	Display*	display;
	Window		top_window;
	struct xatoms
	{
		Atom		server;
		Atom		locales;
		Atom		transport;
		Atom		xconnect;
		Atom		protocol;
		Atom		moredata;
		Atom		comm;
	} atoms;

	SLIST_HEAD (,imconnection)	imconnectionlist;
}ximserv;

static int ignore_error_flag = 0;
static int
errorhandler (Display* display, XErrorEvent* xee)
{
	if (ignore_error_flag) return False;

	fprintf (stderr, "XError Event is received.\n");

	fprintf (stderr, "resource_id  = %ld\n", xee->resourceid);
	fprintf (stderr, "any.window   = %lx\n", ((XEvent*)xee)->xany.window);
	fprintf (stderr, "serial       = %ld\n", xee->serial);
	fprintf (stderr, "error_code   = %d\n",  xee->error_code);
	fprintf (stderr, "request_code = %d\n",  xee->request_code);
	fprintf (stderr, "minor_code   = %d\n",  xee->minor_code);

	while (1) sleep (10000);
}

static void
init_xatoms (void)
{
	Display* dpy = ximserv.display;

	ximserv.atoms.server    = XInternAtom (dpy,"@server=" PACKAGE, False);
	ximserv.atoms.locales   = XInternAtom (dpy, "LOCALES",         False);
	ximserv.atoms.transport = XInternAtom (dpy, "TRANSPORT",       False);

	ximserv.atoms.xconnect  = XInternAtom (dpy, "_XIM_XCONNECT", False);
	ximserv.atoms.protocol  = XInternAtom (dpy, "_XIM_PROTOCOL", False);
	ximserv.atoms.moredata  = XInternAtom (dpy, "_XIM_MOREDATA", False);
	ximserv.atoms.comm      = XInternAtom (dpy, "_" PACKAGE "_COMM",
					       False);
}

static int
init_property (void)
{
	Atom type;
	Atom xim_servers;
	int  format;
	unsigned long nitems;
	unsigned long nbytes;
	unsigned char* props;
	int valuechange = 1;
	int mode = PropModePrepend;
	Atom newdata;
	int ret;

	xim_servers = XInternAtom (ximserv.display, "XIM_SERVERS", False);
	XGetWindowProperty (ximserv.display,
			    DefaultRootWindow (ximserv.display),
			    xim_servers, 0, 256 * 32, False,
			    AnyPropertyType, &type, &format, &nitems,
			    &nbytes, &props);

	if (type != XA_ATOM || format != 32) {
		/*
		 * it is not allowed for this Atom (xim_servers)
		 * so replace it.
		 */
		mode = PropModeReplace;
	} else {
		Atom* atoms = (Atom*)props;
		int i;

		for (i = 0; i < nitems; i++) {
			if (atoms[i] == ximserv.atoms.server) {
				/* property is already ready */
				mode = PropModeAppend;
				valuechange = 0;
				break;
			}
		}
	}
	if (nitems) XFree (props);

	/*
	 * although no need to change property (valuechange == 0),
	 * change property forcibly to raise PropertyNotify
	 */
	newdata = ximserv.atoms.server;
	XChangeProperty (ximserv.display, DefaultRootWindow (ximserv.display),
			 xim_servers, XA_ATOM, 32, mode,
			 (unsigned char*)&newdata,
			 valuechange ? 1 : 0);

	XSetSelectionOwner (ximserv.display, ximserv.atoms.server,
			    ximserv.top_window, CurrentTime);
	ret = XGetSelectionOwner (ximserv.display, ximserv.atoms.server);
	if (ret == ximserv.top_window) {
		return 0;
	} else {
		return -1;
	}
}

static void
notify_locale (XEvent* e)
{
	static unsigned char buf[] = "@locale=ja_JP";
	XChangeProperty (ximserv.display, e->xselectionrequest.requestor,
			 e->xselectionrequest.property,
			 e->xselectionrequest.target,
			 8, PropModeReplace, buf, strlen (buf));
}
static void
notify_transport (XEvent* e)
{
	static unsigned char buf[] = "@transport=X/";
	XChangeProperty (ximserv.display, e->xselectionrequest.requestor,
			 e->xselectionrequest.property,
			 e->xselectionrequest.target,
			 8, PropModeReplace, buf, strlen (buf));
}

static int
new_xim_connection (XClientMessageEvent* ev)
{
	Window client_window;
	struct imconnection* imc;

	if (ev->type != ClientMessage
	    || ev->display      != ximserv.display
	    || ev->window       != ximserv.top_window
	    || ev->message_type != ximserv.atoms.xconnect
	    || ev->format       != 32) {
		return -1;
	}

	client_window = ev->data.l[0];
	DPRINTF ("new imconnection: %lx\n", client_window);

	/* check cleint_window is valid or not */
	{
		XWindowAttributes xwa;
		int ret;
		int origvalue = ignore_error_flag;
		ignore_error_flag = 1;
		ret = XGetWindowAttributes (ximserv.display, client_window,
					    &xwa);
		ignore_error_flag = origvalue;
		if (!ret) return -1;
	}

	imc = imc_allocate ();
	if (imc == NULL) return -1;

	imc_init (imc, ximserv.display, ev->window, client_window);
	SLIST_INSERT_HEAD (&ximserv.imconnectionlist, imc, entry);

	/* send XClentMessageEvent */
	{
		XClientMessageEvent r;

		r.type         = ClientMessage;
		r.window       = client_window;
		r.message_type = ximserv.atoms.xconnect;
		r.format       = 32;
		r.data.l[0]    = imc_comwindow (imc);
		r.data.l[1]    = SERVERTRANSPORT_MAJORVERSION;
		r.data.l[2]    = SERVERTRANSPORT_MINORVERSION;
		r.data.l[3]    = XTRANSPORT_SIZE;
		XSendEvent (ximserv.display, client_window, False, NoEventMask,
			    (XEvent*)&r);
		XFlush (ximserv.display);
	}

	return 0;
}

static void
top_window_proc_xevent (XEvent* e)
{
	switch (e->type) {
	case SelectionRequest:
	{
		XSelectionEvent xse;

		XSelectionRequestEvent* xsre = &e->xselectionrequest;
		if (xsre->selection == ximserv.atoms.server) {
			if (xsre->target == ximserv.atoms.locales) {
				notify_locale (e);
			} else if (xsre->target == ximserv.atoms.transport) {
				notify_transport (e);
			} else {
				fprintf (stderr,
					 "i don't know such property %s\n",
					 XGetAtomName (ximserv.display,
						       xsre->target));
			}
		}

		xse.type      = SelectionNotify;
		xse.serial    = xsre->serial;
		xse.display   = xsre->display;
		xse.requestor = xsre->requestor;
		xse.selection = xsre->selection;
		xse.target    = xsre->target;
		xse.time      = xsre->time;
		xse.property  = xsre->property;

		XSendEvent (ximserv.display, xsre->requestor, False,
			    NoEventMask, (XEvent*)&xse);
		XFlush (ximserv.display);
		break;
	}
	case SelectionClear:
	{
		XSelectionClearEvent* xsce = &e->xselectionclear;

		/* FIXHERE */
		if (xsce->selection == ximserv.atoms.server) {
			;
		}
		break;
	}
	case ClientMessage:
		new_xim_connection ((XClientMessageEvent*)e);
		break;
	default:
		break;
	}
}

static void
xim_read (int fd, void* data)
{
	XEvent e;
	while (XPending (ximserv.display)) {
		XAnyEvent* xany;
		XNextEvent (ximserv.display, &e);
		xany = (XAnyEvent*)&e;

		if (xany->window == ximserv.top_window) {
			top_window_proc_xevent (&e);
		} else {
			struct imconnection* c;

			c = SLIST_FIRST (&ximserv.imconnectionlist);
			while (c) {
				struct imconnection* n = SLIST_NEXT (c, entry);
				int ret;

				ret = imc_xevent (c, &e);
				if (ret == -1) {
					SLIST_REMOVE (&ximserv.
						      imconnectionlist,
						      c, imconnection, entry);
					imc_free (c);
				}
				c = n;
			}
		}
	}
}

int
xim_init (void)
{
	int ret;

	ximserv.display = XOpenDisplay (NULL);
	if (!ximserv.display) {
		fprintf (stderr, "can't open display\n");
		return -1;
	}
	init_xatoms ();
	ignore_error_flag = 0;
	XSetErrorHandler (errorhandler);

	SLIST_INIT (&ximserv.imconnectionlist);

	ret = fontset_initialize_module (ximserv.display);
	if (ret == -1) {
		fprintf (stderr, "can't initialize fontset module\n");
		return -1;
	}

	pewindow_initialize_module (ximserv.display);

	ximserv.top_window = XCreateSimpleWindow (
		ximserv.display, DefaultRootWindow (ximserv.display),
		0, 0, 1, 1, 0, 0, 0);

	ret = init_property ();
	if (ret == -1) return -1;

	XFlush (ximserv.display);

	add_fd_watch (XConnectionNumber (ximserv.display), FD_WATCH_READ,
		      xim_read, NULL);

	/* initialize several modules */
	ret = imc_module_initialize ();
	if (ret == -1) {
		fprintf (stderr, "can't initialize imconnection module\n");
		return -1;
	}
	ret = conversion_module_initialize ();
	if (ret == -1) {
		fprintf (stderr, "can't initialize conversion module\n");
		return -1;
	}


	return 0;
}

int
xim_get_moredata_atom (void)
{
	return ximserv.atoms.moredata;
}
int
xim_get_protocol_atom (void)
{
	return ximserv.atoms.protocol;
}
int
xim_get_comm_atom (void)
{
	return ximserv.atoms.comm;
}

int
xim_xsendevent (Display* d, Window w, int propagate, long event_mask,
		XEvent* e)
{
	int oldv = ignore_error_flag;
	int ret;

	ignore_error_flag = 1;
	ret = XSendEvent (d, w, propagate, event_mask, e);
	ignore_error_flag = oldv;

	return ret;
}
int
xim_xchangeproperty (Display* d, Window w, Atom property, Atom type,
		     int format, int mode, unsigned char* data, int nelements)
{
	int oldv = ignore_error_flag;
	int ret;

	ignore_error_flag = 1;
	ret = XChangeProperty (d, w, property, type, format, mode, data,
			       nelements);
	ignore_error_flag = oldv;

	return ret;
}

static void
swap_bytes (char* buf, const char* format)
{
	const char* p = format;
#define SWAP(fmt,a,b) do {fmt tmp = b; b=a;a=tmp;} while (0)
	while (*p) {
		switch (*p) {
		case 'x':
			buf ++;
			break;
		case 's':
			SWAP (char, buf[0], buf[1]);
			buf += 2;
			break;
		case 'l':
			SWAP (char, buf[0], buf[3]);
			SWAP (char, buf[1], buf[2]);
			buf += 4;
			break;
		}
		p++;
	}
#undef SWAP
}
int
xim_convert_xEvent_to_XEvent (Display* d, const char* buffer,
			      unsigned int serial, XEvent* e, int border)
{
	int one = 1;
	char* onep = (char*)&one;
	int swapping = 0;
	xEvent xev;

	memcpy (&xev, buffer, sizeof (xev));

	if (*onep == 1) {
		/* LSB */
		if (border != BYTE_ORDER_LSB_FIRST) swapping = 1;
	} else {
		/* MSB */
		if (border != BYTE_ORDER_MSB_FIRST) swapping = 1;
	}

	/* make common part */
	if (swapping) swap_bytes ((char*)&xev, "xxs");
	e->xany.type       = xev.u.u.type & 0x7f;
	e->xany.serial     = (serial << 16) | xev.u.u.sequenceNumber;
	e->xany.display    = d;
	e->xany.send_event = xev.u.u.type > 127;

	/* make each format accroding to event type*/
	switch (e->xany.type) {
	case KeyPress:
	case KeyRelease:
		if (swapping) swap_bytes ((char*)&xev, "xxxxllllsssssxx");
		e->xkey.window      = xev.u.keyButtonPointer.event;
		e->xkey.root        = xev.u.keyButtonPointer.root;
		e->xkey.subwindow   = xev.u.keyButtonPointer.child;
		e->xkey.time        = xev.u.keyButtonPointer.time;
		e->xkey.x      = cvtINT16toInt (xev.u.keyButtonPointer.eventX);
		e->xkey.y      = cvtINT16toInt (xev.u.keyButtonPointer.eventY);
		e->xkey.x_root = cvtINT16toInt (xev.u.keyButtonPointer.rootX);
		e->xkey.y_root = cvtINT16toInt (xev.u.keyButtonPointer.rootY);
		e->xkey.state       = xev.u.keyButtonPointer.state;
		e->xkey.keycode     = xev.u.u.detail;
		e->xkey.same_screen = xev.u.keyButtonPointer.sameScreen;
		break;
	default:
		/* deal only KeyPress KeyRelease */
		return -1;
		break;
	}
	
	return 0;
}
int
xim_convert_XEvent_to_xEvent (Display* d, const XEvent* e,
			      unsigned int* serial, xEvent* x, int border)
{
	int one = 1;
	char* onep = (char*)&one;
	int swapping = 0;

	if (*onep == 1) {
		/* LSB */
		if (border != BYTE_ORDER_LSB_FIRST) swapping = 1;
	} else {
		/* MSB */
		if (border != BYTE_ORDER_MSB_FIRST) swapping = 1;
	}

	x->u.u.type = e->xany.type;
	*serial = (e->xany.serial >> 16) & 0xffff;
	x->u.u.sequenceNumber = (e->xany.serial & 0xffff);
	if (e->xany.send_event) x->u.u.type &= 0x80;
	if (swapping) swap_bytes ((char*)x, "xxs");

	switch (e->xany.type) {
	case KeyPress:
	case KeyRelease:
		x->u.keyButtonPointer.event      = e->xkey.window;
		x->u.keyButtonPointer.root       = e->xkey.root;
		x->u.keyButtonPointer.child      = e->xkey.subwindow;
		x->u.keyButtonPointer.time       = e->xkey.time;
		x->u.keyButtonPointer.eventX     = e->xkey.x;
		x->u.keyButtonPointer.eventY     = e->xkey.y;
		x->u.keyButtonPointer.rootX      = e->xkey.x_root;
		x->u.keyButtonPointer.rootY      = e->xkey.y_root;
		x->u.keyButtonPointer.state      = e->xkey.state;
		x->u.u.detail                    = e->xkey.keycode;
		x->u.keyButtonPointer.sameScreen = e->xkey.same_screen;
		if (swapping) swap_bytes ((char*)x, "xxxxllllsssssxx");
		break;
	default:
		return -1;
		break;
	}

	return 0;
}
