/*
 * deal InputMethod and InputContext
 *
 * $Id: ximic.c,v 1.21 2002/03/26 10:47:00 taka Exp $
 */
#include <string.h>
#include <stdlib.h>
#include <X11/Xlib.h>

#include "xim.h"
#include "imconnection.h"
#include "ximpacket.h"
#include "ximic.h"
#include "ximproc.h"
#include "byteorder.h"
#include "conversion.h"
#include "pewindow.h"
#include "aime.h"
#include "fontset.h"

enum
{
	TYPE_BYTE                = 1,
	TYPE_WORD                = 2,
	TYPE_LONG                = 3,
	TYPE_CHAR                = 4,
	TYPE_WINDOW              = 5,
	TYPE_XIMSTYLES           = 10,
	TYPE_XRECTANGLE          = 11,
	TYPE_XPOINT              = 12,
	TYPE_XFONTSET            = 13,
	TYPE_XIMHOTKEYTRIGGERS   = 15,
	TYPE_XIMHOTKEYSTATE      = 16,
	TYPE_XIMSTRINGCONVERSION = 17,
	TYPE_XIMPREEDITSTATE     = 18,
	TYPE_XIMRESETSTATE       = 19,
	TYPE_NESTEDLIST          = 0x7fff,
};


static struct input_style_table
{
	int x_style;
	int style;
}inputstyles[] =
{
	{XIMPreeditNothing   | XIMStatusNothing, ROOT_WINDOW},
	{XIMPreeditPosition  | XIMStatusArea,    OVER_THE_SPOT},
	{XIMPreeditPosition  | XIMStatusNothing, OVER_THE_SPOT},
#if 0
	{XIMPreeditArea      | XIMStatusArea,    OFF_THE_SPOT},
#endif
	{XIMPreeditCallbacks | XIMStatusNothing, ON_THE_SPOT},
	{XIMPreeditCallbacks | XIMStatusCallbacks, ON_THE_SPOT},
	{0, 0},
};

static int
getqueryinputstyle (struct ximpacket* x, int id, int btype)
{
	int numstyles = NUMBER_OF_ELEMENTS (inputstyles) - 1;
	int numbyte;
	int i;
	int ret = 0;

	numbyte = 4 + numstyles * 4;

	ret |= ximpacket_put16 (x, id,        btype);
	ret |= ximpacket_put16 (x, numbyte,   btype);
	ret |= ximpacket_put16 (x, numstyles, btype);
	ret |= ximpacket_put16 (x, 0,         btype);
	for (i = 0; i < numstyles; i++) {
		if (disable_on_the_spot
		    && inputstyles[i].style == ON_THE_SPOT) continue;
		ret |= ximpacket_put32 (x, inputstyles[i].x_style, btype);
	}
	return numbyte + 4;
}

static struct ximattribute
{
	const char*	name;
	int	type;
	int (*get_im_value_proc) (struct ximpacket* x, int id, int btype);
}ximattr[] =
{
	{XNQueryInputStyle, TYPE_XIMSTYLES, getqueryinputstyle},
	{NULL},
};

int
ximim_im_putattributes (struct ximpacket* x, int type)
{
	int ret = 0;
	int i;
	int len = 0;

	/* check len */
	for (i = 0; ximattr[i].name != NULL; i++) {
		len += 2 * 3;
		len += strlen (ximattr[i].name);
		len += PAD4 (strlen (ximattr[i].name) + 2);
	}

	ret |= ximpacket_put16 (x, len, type);
	for (i = 0; ximattr[i].name != NULL; i++) {
		ret |= ximpacket_put16 (x, i, type);
		ret |= ximpacket_put16 (x, ximattr[i].type, type);
		ret |= ximpacket_put16 (x, strlen (ximattr[i].name), type);
		ret |= ximpacket_putbytes (x, ximattr[i].name,
					   strlen (ximattr[i].name), type);
		ret |= ximpacket_putpad (x, strlen (ximattr[i].name) + 2);
	}

	return ret;
}
int
ximim_im_get_values (struct ximpacket* x, char* data, int len, int border)
{
	int start, end;
	int ret = 0;

	start = ximpacket_position (x);
	ret = ximpacket_put16 (x, 0, border);
	if (ret == -1) return -1;

	while (len >= 2) {
		unsigned int id;

		id = get16 (data, border);
		data += 2;
		len -= 2;

		if (id >= NUMBER_OF_ELEMENTS (ximattr)) {
			/* error */
			ximpacket_init_buffer (x);
			return ximp_bad_length (x, id, 0, border);
		}

		if (ximattr[id].get_im_value_proc) {
			ret = ximattr[id].get_im_value_proc (x, id, border);
			if (ret == -1) {
				ximpacket_init_buffer (x);
				return ximp_bad_length (x, id, 0, border);
			}
		}
	}

	end = ximpacket_position (x);
	ximpacket_reput16 (x, end - start, start, border);

	return 0;
}

enum {
	ICATTR_NONE,
	ICATTR_PREEDIT_ATTRIBUTE,
	ICATTR_STATUS_ATTRIBUTE,
};
static struct ps_attribute*
get_preedit_attribute (struct ximic* xic, int nest)
{
	if (nest == ICATTR_PREEDIT_ATTRIBUTE) {
		return &xic->preedit_attr;
	} else if (nest == ICATTR_STATUS_ATTRIBUTE) {
		return &xic->status_attr;
	} else {
		return NULL;
	}
}
#define SEND_BAD_LENGTH							\
	do {								\
		int ret;						\
		ximpacket_init_buffer (x);				\
		ret = ximp_bad_length (x, xic->imid, xic->icid, border); \
		if (ret == -1) return -1;				\
		else           return 1;				\
	} while (0)
#define SEND_BAD_NEST							\
	do {								\
		int retret;						\
		ximpacket_init_buffer (x);				\
		retret = ximpacket_error_packet (x, XIM_ERROR_BADSOMETHING,\
					      xic->imid, xic->icid,	\
					      "bad nest", border);	\
		if (retret == -1) return -1;				\
		else              return 1;				\
	} while (0)
	
 
#define CHECK_LEN(len,valid) 						\
	if ((len) != (valid)) {						\
		SEND_BAD_LENGTH;					\
	}
#define CHECK_NEST(nest,valid) 						\
	if ((nest) != (valid)) {					\
		SEND_BAD_NEST;						\
	}
		
static int
ic_set_attributes (struct ximic* xic, struct ximpacket* x, char* data,
		   int nest, int len, int border);
static int
ic_get_attributes (struct ximic* xic, struct ximpacket* x, char* data,
		   int nest, int len, int border);
static int
setinputstyle (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	struct input_style_table* t;
	int xstyle;

	CHECK_LEN  (len, 4);
	CHECK_NEST (nest, ICATTR_NONE);

	if (xic->common_attr.set_mask & ATTR_MASK_INPUT_STYLE) {
		int ret;
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADSTYLE,
					      xic->imid, xic->icid,
					      "style is already specified",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
	
	xstyle = get32 (buf, border);
	for (t = inputstyles; t->x_style != 0; t++) {
		if (disable_on_the_spot && t->style == ON_THE_SPOT) continue;
		if (t->x_style == xstyle) {
			xic->common_attr.input_style = t->style;
			xic->common_attr.changed_mask |= ATTR_MASK_INPUT_STYLE;
			xic->common_attr.set_mask     |= ATTR_MASK_INPUT_STYLE;
			DPRINTF (" [%lx]setIC input style=%d\n",
				 xic->imc->clientwindow,
				 xic->common_attr.input_style);
			break;
		}
	}
	if (t->x_style == 0) {
		int ret;
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADSTYLE,
					      xic->imid, xic->icid,
					      "such style is not supported",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
	return 0;

}
static int
setclientwindow (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		 char* buf, int len, int border)
{
	CHECK_LEN (len, 4);
	CHECK_NEST (nest, ICATTR_NONE);

	if (xic->common_attr.set_mask & ATTR_MASK_CLIENT_WINDOW) {
		int ret;
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADCLIENTWINDOW,
					      xic->imid, xic->icid,
					      "can't change clientwindow",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
	xic->common_attr.client_window = get32 (buf, border);
	xic->common_attr.changed_mask |= ATTR_MASK_CLIENT_WINDOW;
	xic->common_attr.set_mask     |= ATTR_MASK_CLIENT_WINDOW;
	DPRINTF (" [%lx]setIC client_window=%lx\n",
		 xic->imc->clientwindow, xic->common_attr.client_window);
	return 0;
}
static int
setfocuswindow (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		char* buf, int len, int border)
{
	Window w;
	CHECK_LEN (len, 4);
	CHECK_NEST (nest, ICATTR_NONE);

	w = get32 (buf, border);
	if (! (xic->common_attr.set_mask & ATTR_MASK_FOCUS_WINDOW)
	    || xic->common_attr.focus_window != w) {
		xic->common_attr.changed_mask |= ATTR_MASK_FOCUS_WINDOW;
	}
	xic->common_attr.focus_window = w;
	xic->common_attr.set_mask |= ATTR_MASK_FOCUS_WINDOW;
	DPRINTF (" [%lx]setIC focus_window=%lx\n",
		 xic->imc->clientwindow, xic->common_attr.focus_window);
	return 0;
}
static int
setpreeditattributes (struct ximic* xic, struct ximpacket* x, int attrid,
		      int nest, char* buf, int len, int border)
{
	int ret;
	CHECK_NEST (nest, ICATTR_NONE);

	DPRINTF (" [%lx]setIC preeditattr\n", xic->imc->clientwindow);
	ret = ic_set_attributes (xic, x, buf, ICATTR_PREEDIT_ATTRIBUTE,
				 len, border);
	DPRINTF (" [%lx]setIC preeditattr done\n", xic->imc->clientwindow);
	return ret;
}
static int
setstatusattributes (struct ximic* xic, struct ximpacket* x, int attrid,
		     int nest, char* buf, int len, int border)
{
	int ret;
	CHECK_NEST (nest, ICATTR_NONE);

	DPRINTF (" [%lx]setIC statusattr\n", xic->imc->clientwindow);
	ret = ic_set_attributes (xic, x, buf, ICATTR_STATUS_ATTRIBUTE,
				 len, border);
	DPRINTF (" [%lx]setIC statusattr done\n", xic->imc->clientwindow);
	return ret;
}
static int
setarea (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	 char* buf, int len, int border)
{
	XRectangle rect;
	struct ps_attribute* patr;
	CHECK_LEN (len, 8);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	rect.x      = get16 (buf,     border);
	rect.y      = get16 (buf + 2, border);
	rect.width  = get16 (buf + 4, border);
	rect.height = get16 (buf + 6, border);

	if (! (patr->set_mask & ATTR_MASK_AREA)
	    || patr->area.x      != rect.x     
	    || patr->area.y      != rect.y     
	    || patr->area.width  != rect.width 
	    || patr->area.height != rect.height) {
		patr->changed_mask |= ATTR_MASK_AREA;
	}

	patr->area.x       = rect.x;
	patr->area.y       = rect.y;
	patr->area.width   = rect.width;
	patr->area.height  = rect.height;
	patr->set_mask     |= ATTR_MASK_AREA;

	DPRINTF ("  [%lx]setIC setarea x=%d y=%d width=%d height=%d\n",
		 xic->imc->clientwindow, patr->area.x, patr->area.y,
		 patr->area.width, patr->area.height);
	return 0;
}
static int
setareaneeded (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	XRectangle rect;
	struct ps_attribute* patr;
	CHECK_LEN (len, 8);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	rect.width  = get16 (buf + 4, border);
	rect.height = get16 (buf + 6, border);

	if (! (patr->set_mask & ATTR_MASK_AREA_NEEDED)
	    || patr->area_needed.width  != rect.width
	    || patr->area_needed.height != rect.height) {
		patr->changed_mask |= ATTR_MASK_AREA_NEEDED;
	}
	patr->area_needed.width  = rect.width;
	patr->area_needed.height = rect.height;
	patr->set_mask |= ATTR_MASK_AREA_NEEDED;

	DPRINTF ("  [%lx]setIC area_needed x=%d y=%d width=%d height=%d\n",
		 xic->imc->clientwindow,
		 patr->area_needed.x, patr->area_needed.y,
		 patr->area_needed.width, patr->area_needed.height);
	return 0;
}
static int
setspotlocation (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		 char* buf, int len, int border)
{
	XPoint s;
	struct ps_attribute* patr;
	CHECK_LEN (len, 4);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	s.x = get16 (buf,     border);
	s.y = get16 (buf + 2, border);

	if (! (patr->set_mask & ATTR_MASK_SPOT_LOCATION)
	    || patr->spot_location.x != s.x
	    || patr->spot_location.y != s.y) {
		patr->changed_mask |= ATTR_MASK_SPOT_LOCATION;
	}
	patr->spot_location.x = s.x;
	patr->spot_location.y = s.y;
	patr->set_mask |= ATTR_MASK_SPOT_LOCATION;

	DPRINTF ("  [%lx]setIC location x=%d y=%d\n",
		 xic->imc->clientwindow,
		 patr->spot_location.x, patr->spot_location.y);
	return 0;
}
static int
setforeground (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	Pixel f;
	struct ps_attribute* patr;
	CHECK_LEN (len, 4);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	f = get32 (buf, border);
	if (! (patr->set_mask & ATTR_MASK_FOREGROUND)
	    || patr->foreground != f) {
		patr->changed_mask |= ATTR_MASK_FOREGROUND;
	}
	patr->foreground = f;
	patr->set_mask |= ATTR_MASK_FOREGROUND;

	DPRINTF ("  [%lx]setIC foreground=%ld\n",
		 xic->imc->clientwindow, patr->foreground);
	return 0;
}
static int
setbackground (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	Pixel b;
	struct ps_attribute* patr;
	CHECK_LEN (len, 4);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	b = get32 (buf, border);

	if (! (patr->set_mask & ATTR_MASK_BACKGROUND)
	    || patr->background != b) {
		patr->changed_mask |= ATTR_MASK_BACKGROUND;
	}
	patr->background = b;
	patr->set_mask |= ATTR_MASK_BACKGROUND;

	DPRINTF ("  [%lx]setIC background=%ld\n",
		 xic->imc->clientwindow, patr->background);
	return 0;
}
static int
setfontset (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	    char* buf, int len, int border)
{
	struct ps_attribute* patr;
	char* string;
	int l;

	if (len < 2) SEND_BAD_LENGTH;

	l = get16 (buf, border);
	if (2 + l > len) SEND_BAD_LENGTH;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	string = malloc (sizeof (char) * (l + 1));
	if (string == NULL) {
		int ret;
		NOMEMORY ("can't allocate fontset string");
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADALLOC,
					      xic->imid, xic->icid,
					      "can't alloc", border);
		if (ret == -1) return -1;
		else           return 1;
	}
	memcpy (string, buf + 2, l);
	string[l] = '\0';

	if (patr->set_mask & ATTR_MASK_FONTSET) {
		if (strcmp (fontset_get_fontsetname (patr->fontset), string)
		    != 0) {
			fontset_release (patr->fontset);
			patr->fontset = fontset_get_byname (xic->imc->display,
							    string);
			patr->changed_mask |= ATTR_MASK_FONTSET;
		}
	} else {
		patr->fontset = fontset_get_byname (xic->imc->display, string);
		if (patr->fontset == NULL) {
			patr->fontset = fontset_get_byname (
				xic->imc->display, DEFAULT_FONTSETNAME);
		}
		patr->changed_mask |= ATTR_MASK_FONTSET;
	}

	patr->set_mask |= ATTR_MASK_FONTSET;
	DPRINTF ("  [%lx]setIC fonset=%s\n", xic->imc->clientwindow, string);
	free (string);

	return 0;
}
static int
setlinespace (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	      char* buf, int len, int border)
{
	int ls;
	struct ps_attribute* patr;
	CHECK_LEN (len, 4);

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	ls = get32 (buf, border);

	if (! (patr->set_mask & ATTR_MASK_LINE_SPACE)
	    ||	patr->line_space != ls) {
		patr->changed_mask |= ATTR_MASK_LINE_SPACE;
	}

	patr->line_space = ls;
	patr->set_mask |= ATTR_MASK_LINE_SPACE;

	DPRINTF ("  [%lx]setIC linespace=%d\n",
		 xic->imc->clientwindow, patr->line_space);
	return 0;
}

static void
fill_up_psattribute_default (struct ximic* xic, int nest, int mask);
static int
getinputstyle (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	int ret;
	CHECK_NEST (nest, ICATTR_NONE);

	if (xic->common_attr.set_mask & ATTR_MASK_CLIENT_WINDOW) {
		ret = 0;
		ret |= ximpacket_put16 (x, attrid, border);
		ret |= ximpacket_put16 (x, 4, border);	/* length */
		ret |= ximpacket_put32 (x, xic->common_attr.input_style,
					border);
		return ret;
	} else {
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADSTYLE,
					      xic->imid, xic->icid,
					      "style is not specified yet",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
}
static int
getclientwindow (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		 char* buf, int len, int border)
{
	int ret;
	CHECK_NEST (nest, ICATTR_NONE);

	if (xic->common_attr.set_mask & ATTR_MASK_CLIENT_WINDOW) {
		ret = 0;
		ret |= ximpacket_put16 (x, attrid, border);
		ret |= ximpacket_put16 (x, 4, border);	/* length */
		ret |= ximpacket_put32 (x, xic->common_attr.client_window,
					border);
		return ret;
	} else {
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADCLIENTWINDOW,
					      xic->imid, xic->icid,
					      "client window is not specified yet",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
}
static int
getfocuswindow (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		char* buf, int len, int border)
{
	int ret;
	CHECK_NEST (nest, ICATTR_NONE);

	if (! (xic->common_attr.set_mask & ATTR_MASK_FOCUS_WINDOW)) {
		if (xic->common_attr.set_mask & ATTR_MASK_CLIENT_WINDOW) {
			xic->common_attr.focus_window
				= xic->common_attr.client_window;
			xic->common_attr.set_mask |= ATTR_MASK_FOCUS_WINDOW;
		}
	}

	if (xic->common_attr.set_mask & ATTR_MASK_FOCUS_WINDOW) {
		ret = 0;
		ret |= ximpacket_put16 (x, attrid, border);
		ret |= ximpacket_put16 (x, 4, border);	/* length */
		ret |= ximpacket_put32 (x, xic->common_attr.focus_window,
					border);
		return ret;
	} else {
		ximpacket_init_buffer (x);
		ret = ximpacket_error_packet (x, XIM_ERROR_BADFOCUSWINDOW,
					      xic->imid, xic->icid,
					      "neither focus window nor client window are specified yet",
					      border);
		if (ret == -1) return -1;
		else           return  1;
	}
}
static int
getpreeditattributes (struct ximic* xic, struct ximpacket* x, int attrid,
		      int nest, char* buf, int len, int border)
{
	CHECK_NEST (nest, ICATTR_NONE);
	return ic_get_attributes (xic, x, buf, ICATTR_PREEDIT_ATTRIBUTE,
				  len, border);
}
static int
getstatusattributes (struct ximic* xic, struct ximpacket* x, int attrid,
		     int nest, char* buf, int len, int border)
{
	CHECK_NEST (nest, ICATTR_NONE);
	return ic_get_attributes (xic, x, buf, ICATTR_PREEDIT_ATTRIBUTE, len,
				  border);
}
static int
getarea (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	 char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_AREA)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_AREA);
	}
	
	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 8, border);
	ret |= ximpacket_put16 (x, patr->area.x, border);
	ret |= ximpacket_put16 (x, patr->area.y, border);
	ret |= ximpacket_put16 (x, patr->area.width, border);
	ret |= ximpacket_put16 (x, patr->area.height, border);
	return ret;
}
static int
getareaneeded (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_AREA_NEEDED)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_AREA_NEEDED);
	}

	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 8, border);
	ret |= ximpacket_put16 (x, patr->area_needed.x, border);
	ret |= ximpacket_put16 (x, patr->area_needed.y, border);
	ret |= ximpacket_put16 (x, patr->area_needed.width, border);
	ret |= ximpacket_put16 (x, patr->area_needed.height, border);
	return ret;
}
static int
getspotlocation (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
		 char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_SPOT_LOCATION)) {
		fill_up_psattribute_default (xic, nest,
					     ATTR_MASK_SPOT_LOCATION);
	}

	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 4, border);
	ret |= ximpacket_put16 (x, patr->spot_location.x, border);
	ret |= ximpacket_put16 (x, patr->spot_location.y, border);
	return ret;
}
static int
getforeground (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_FOREGROUND)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_FOREGROUND);
	}

	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 4, border);
	ret |= ximpacket_put32 (x, patr->foreground, border);
	return ret;
}
static int
getbackground (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	       char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_BACKGROUND)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_BACKGROUND);
	}

	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 4, border);
	ret |= ximpacket_put32 (x, patr->background, border);
	return ret;
}
static int
getfontset (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	    char* buf, int len, int border)
{
	struct ps_attribute* patr;
	char* string;
	int ret = 0;
	int l;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_FONTSET)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_FONTSET);
	}

	string = fontset_get_fontsetname (patr->fontset);
	l = strlen (string);
	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, l, border);
	ret |= ximpacket_putbytes (x, string, l, border);
	ret |= ximpacket_putpad (x, l);
	return ret;
}
static int
getlinespace (struct ximic* xic, struct ximpacket* x, int attrid, int nest,
	      char* buf, int len, int border)
{
	struct ps_attribute* patr;
	int ret = 0;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) SEND_BAD_NEST;

	if (! (patr->set_mask & ATTR_MASK_LINE_SPACE)) {
		fill_up_psattribute_default (xic, nest, ATTR_MASK_LINE_SPACE);
	}

	ret |= ximpacket_put16 (x, attrid, border);
	ret |= ximpacket_put16 (x, 4, border);
	ret |= ximpacket_put32 (x, patr->line_space, border);
	return ret;
}

static struct xicattributes
{
	const char*	name;
	int	type;
	int (*set_ic_value_proc) (struct ximic* xic, struct ximpacket* x,
				  int attrid, int nest, char* buf, int len,
				  int border);
	int (*get_ic_value_proc) (struct ximic* xic, struct ximpacket* x,
				  int attrid, int nest, char* buf, int len,
				  int border);
}xicattr[]={
	{XNInputStyle,        TYPE_LONG,       setinputstyle, getinputstyle},
	{XNClientWindow,      TYPE_WINDOW,     setclientwindow,
	 getclientwindow},
	{XNFocusWindow,       TYPE_WINDOW,     setfocuswindow, getfocuswindow},

	{XNPreeditAttributes, TYPE_NESTEDLIST, setpreeditattributes,
	 getpreeditattributes},
	{XNStatusAttributes,  TYPE_NESTEDLIST, setstatusattributes,
	 getstatusattributes},
	{XNArea,              TYPE_XRECTANGLE, setarea, getarea},
	{XNAreaNeeded,        TYPE_XRECTANGLE, setareaneeded, getareaneeded},
	{XNSpotLocation,      TYPE_XPOINT,     setspotlocation,
	 getspotlocation},
	{XNForeground,        TYPE_LONG,       setforeground, getforeground},
	{XNBackground,        TYPE_LONG,       setbackground, getbackground},
	{XNFontSet,           TYPE_XFONTSET,   setfontset, getfontset},
	{XNLineSpace,         TYPE_WORD,       setlinespace, getlinespace},

	{XNColormap,          TYPE_WORD},
	{XNStdColormap,       TYPE_WORD},
	{XNBackgroundPixmap,  TYPE_LONG},
	{XNCursor,            TYPE_WORD},

	{XNPreeditAttributes, TYPE_NESTEDLIST},
	{NULL},
};

int
ximic_ic_putattributes (struct ximpacket* x, int type)
{
	int ret = 0;
	int i;
	int len = 0;

	/* check len */
	for (i = 0; xicattr[i].name != NULL; i++) {
		len += 2 * 3;
		len += strlen (xicattr[i].name);
		len += PAD4 (strlen (xicattr[i].name) + 2);
	}

	ret |= ximpacket_put16 (x, len, type);
	ret |= ximpacket_put16 (x, 0,   type);	/* padding */
	for (i = 0; xicattr[i].name != NULL; i++) {
		ret |= ximpacket_put16 (x, i, type);
		ret |= ximpacket_put16 (x, xicattr[i].type, type);
		ret |= ximpacket_put16 (x, strlen (xicattr[i].name), type);
		ret |= ximpacket_putbytes (x, xicattr[i].name,
					   strlen (xicattr[i].name), type);
		ret |= ximpacket_putpad (x, strlen (xicattr[i].name) + 2);
	}

	return ret;
}
static int
ic_set_attributes (struct ximic* xic, struct ximpacket* x, char* data,
		   int nest, int len, int border)
{
	unsigned int offset = 0;

	while (len >= 4) {
		unsigned int id, l;
		id = get16 (data + offset,     border);
		l  = get16 (data + offset + 2, border);
		l  += PAD4 (l);
		len -= 4;
		offset += 4;

		if (l > len) {
			ximpacket_init_buffer (x);
			return ximp_bad_length (x, xic->imid, xic->icid,
						border);
		}

		if (id >= NUMBER_OF_ELEMENTS (xicattr)) {
			/* error */
			ximpacket_init_buffer (x);
			return ximp_bad_length (x, xic->imid, xic->icid,
						border);
		}

		if (xicattr[id].set_ic_value_proc) {
			int ret;
			ret = xicattr[id].set_ic_value_proc (xic, x, id,
							     nest,
							     data + offset,
							     l, border);
			if (ret == -1) return -1;
			else if (ret == 1) return 0;
		}
		len    -= l;
		offset += l;
	}

	return 0;
}
void
ximic_make_peattribute (struct ximic* xic, struct pe_attribute* merged)
{
	struct common_attribute* c = &xic->common_attr;
	struct ps_attribute* p = &xic->preedit_attr;
	struct ps_attribute* s = &xic->status_attr;

	merged->changed_mask = 0;

	if (c->changed_mask & ATTR_MASK_FOCUS_WINDOW) {
		merged->focus_window = c->focus_window;
		merged->changed_mask |= ATTR_MASK_FOCUS_WINDOW;
		merged->set_mask     |= ATTR_MASK_FOCUS_WINDOW;

	} else if (c->changed_mask & ATTR_MASK_CLIENT_WINDOW) {
		merged->focus_window = c->client_window;
		merged->changed_mask |= ATTR_MASK_FOCUS_WINDOW;
		merged->set_mask     |= ATTR_MASK_FOCUS_WINDOW;
	}

	if (p->changed_mask & ATTR_MASK_AREA) {
		merged->client_area.x      = p->area.x;
		merged->client_area.y      = p->area.y;
		merged->client_area.width  = p->area.width;
		merged->client_area.height = p->area.height;
		merged->changed_mask |= ATTR_MASK_AREA;
		merged->set_mask     |= ATTR_MASK_AREA;
	}

	if (p->changed_mask & ATTR_MASK_FOREGROUND) {
		merged->foreground = p->foreground;
		merged->changed_mask |= ATTR_MASK_FOREGROUND;
		merged->set_mask     |= ATTR_MASK_FOREGROUND;
	}

	if (p->changed_mask & ATTR_MASK_BACKGROUND) {
		merged->background = p->background;
		merged->changed_mask |= ATTR_MASK_BACKGROUND;
		merged->set_mask     |= ATTR_MASK_BACKGROUND;
	}

	if (p->changed_mask & ATTR_MASK_FONTSET) {
		merged->fontset = p->fontset;
		merged->changed_mask |= ATTR_MASK_FONTSET;
		merged->set_mask     |= ATTR_MASK_FONTSET;
	}

	if (p->changed_mask & ATTR_MASK_SPOT_LOCATION) {
		merged->spot_location.x = p->spot_location.x;
		merged->spot_location.y = p->spot_location.y;
		merged->changed_mask |= ATTR_MASK_SPOT_LOCATION;
		merged->set_mask     |= ATTR_MASK_SPOT_LOCATION;
	}

	if (s->changed_mask & ATTR_MASK_AREA) {
		merged->status_area.x      = s->area.x;
		merged->status_area.y      = s->area.y;
		merged->status_area.width  = s->area.width;
		merged->status_area.height = s->area.height;
		merged->changed_mask |= ATTR_MASK_STATUS_AREA;
		merged->set_mask     |= ATTR_MASK_STATUS_AREA;
	}
}
static void
compute_area_needed (struct ximic* xic)
{
	struct ps_attribute* attr;
	int font_height;
	int max_width, max_height;
	int width, height;

	attr = &xic->status_attr;
	
	fill_up_psattribute_default (xic, ICATTR_STATUS_ATTRIBUTE,
				     ATTR_MASK_LINE_SPACE);
	font_height = attr->line_space + 2;
	max_width = max_height = 0;
	if (attr->set_mask & ATTR_MASK_AREA_NEEDED) {
		max_width  = attr->area_needed.width;
		max_height = attr->area_needed.height;
	}
	/* guess */
	width = font_height * 40;

	height = font_height;
	if (max_height > 0 && height > max_height) height = max_height;
	if (max_width  > 0 && width  > max_width)  width  = max_width;
	
	attr->area_needed.x      = 0;
	attr->area_needed.y      = 0;
	attr->area_needed.width  = width;
	attr->area_needed.height = height;


	fill_up_psattribute_default (xic, ICATTR_PREEDIT_ATTRIBUTE,
				     ATTR_MASK_LINE_SPACE);
	attr = &xic->preedit_attr;
	font_height = attr->line_space + 2;
	max_width = max_height = 0;
	font_height = attr->line_space + 2;
	max_width = max_height = 0;
	if (attr->set_mask & ATTR_MASK_AREA_NEEDED) {
		max_width  = attr->area_needed.width;
		max_height = attr->area_needed.height;
	}
	/* guess */
	width = font_height * 40;

	height = font_height;
	if (max_height > 0 && height > max_height) height = max_height;
	if (max_width  > 0 && width  > max_width)  width  = max_width;
	
	attr->area_needed.x      = 0;
	attr->area_needed.y      = 0;
	attr->area_needed.width  = width;
	attr->area_needed.height = height;
}
static void
fill_up_psattribute_default (struct ximic* xic, int nest, int mask)
{
	struct ps_attribute* patr;

	patr = get_preedit_attribute (xic, nest);
	if (!patr) return;

	if (mask & ATTR_MASK_AREA) {
		compute_area_needed (xic);

		if (! (patr->set_mask & ATTR_MASK_AREA)) {
			if (xic->common_attr.input_style == OFF_THE_SPOT) {
				patr->area = patr->area_needed;
			} else {
				patr->area.x      = 0;
				patr->area.y      = 0;
				patr->area.width
					= xic->common_attr.focus_width;
				patr->area.height
					= xic->common_attr.focus_height;
			}
		}
	}

	if (mask & ATTR_MASK_LINE_SPACE) {
		XFontSetExtents* ex;

		if (! (patr->set_mask & ATTR_MASK_FONTSET)) {
			fill_up_psattribute_default (xic, nest,
						     ATTR_MASK_FONTSET);
		}

		ex = fontset_get_xfontsetextents (patr->fontset);
		patr->line_space =  (ex->max_logical_extent.y
				     + ex->max_logical_extent.height)
			+ ex->max_logical_extent.height;
		patr->set_mask |= ATTR_MASK_LINE_SPACE;
	}

	if (mask & ATTR_MASK_FONTSET) {
		patr->fontset = fontset_get_byname (xic->imc->display,
						    DEFAULT_FONTSETNAME);
		patr->set_mask |= ATTR_MASK_FONTSET;
	}
}

static int
validate_ximic_attr (struct ximic* xic, struct ximpacket* xip, int border)
{
	struct ps_attribute* attr;
	int mask;
	int ret = 0;

	
	if (! (xic->common_attr.set_mask & ATTR_MASK_INPUT_STYLE)) {
		ximpacket_init_buffer (xip);
		ximpacket_error_packet (xip, XIM_ERROR_BADSTYLE,
					xic->imid, xic->icid,
					"style is not specified", border);
		return -1;
	}

	if (xic->common_attr.set_mask & ATTR_MASK_CLIENT_WINDOW) {
		Window root;
		int s,x,y;
		unsigned int w,h,b,d;
		s = XGetGeometry (xic->imc->display,
				  xic->common_attr.client_window, &root,
				  &x, &y, &w, &h, &b, &d);
		if (s) {
			xic->common_attr.client_width  = w;
			xic->common_attr.client_height = h;
			ret = 1;
		} else {
			ximpacket_init_buffer (xip);
			ximpacket_error_packet (xip, XIM_ERROR_BADCLIENTWINDOW,
						xic->imid, xic->icid,
						"client window is invalid",
						border);
			return -1;
		}
	}
	if (xic->common_attr.set_mask & ATTR_MASK_FOCUS_WINDOW) {
		if (xic->common_attr.focus_window
		    == xic->common_attr.client_window) {
			xic->common_attr.focus_width
				= xic->common_attr.client_width;
			xic->common_attr.focus_height
				= xic->common_attr.client_height;
			ret = 1;
		} else {
			Window root;
			int s,x,y;
			unsigned int w,h,b,d;
			s = XGetGeometry (xic->imc->display,
					  xic->common_attr.client_window,
					  &root, &x, &y, &w, &h, &b, &d);
			if (s) {
				xic->common_attr.focus_width  = w;
				xic->common_attr.focus_height = h;
				ret = 1;
			} else {
				ximpacket_init_buffer (xip);
				ximpacket_error_packet (
					xip, XIM_ERROR_BADFOCUSWINDOW,
					xic->imid, xic->icid,
					"focus window is invalid", border);
			}
		}
	}
	if (ret == 0) {
		ximpacket_init_buffer (xip);
		ximpacket_error_packet (xip, XIM_ERROR_BADSOMETHING,
					xic->imid, xic->icid,
					"neither focus window nor client window are specified yet",
					border);
		return -1;
	}

	attr = &xic->status_attr;
	mask = attr->set_mask & attr->changed_mask;
	if (mask & ATTR_MASK_AREA) {
		if (attr->area.width == 0 || attr->area.height) {
			attr->set_mask &= ~ATTR_MASK_AREA;
		}
	}
	attr = &xic->preedit_attr;
	mask = attr->set_mask & attr->changed_mask;
	if (mask & ATTR_MASK_AREA) {
		if (attr->area.width == 0 || attr->area.height) {
			attr->set_mask &= ~ATTR_MASK_AREA;
		}
	}
	return 0;
}

int
ximic_ic_set_attributes (struct ximic* xic, struct ximpacket* x, char* data,
			 int len, int border)
{
	int ret;
	xic->common_attr.changed_mask = 0;
	xic->preedit_attr.changed_mask = 0;
	xic->status_attr.changed_mask = 0;

	ret = ic_set_attributes (xic, x, data, ICATTR_NONE, len, border);

	if (ret == 0) {
		ret = validate_ximic_attr (xic, x, border);
		if (ret == -1) return -1;

		/*
		 * at first time pew is not created yet.
		 */
		if (xic->pew) {
			struct pe_attribute* m =
				pewindow_get_peattribute (xic->pew);

			ximic_make_peattribute (xic, m);

			ret = pewindow_change_attribute (xic->pew);
			if (ret == -1) {
				ximpacket_init_buffer (x);
				ximpacket_error_packet (
					x, xic->imid, xic->icid,
					XIM_ERROR_BADSOMETHING,
					"can't set ic attributes", border);
				return -1;
			}
		}
	}

	return 0;
}
	
static int
ic_get_attributes (struct ximic* xic, struct ximpacket* x, char* data,
		   int nest, int len, int border)
{
	int start, end;
	int ret = 0;
	unsigned int offset = 0;

	start = ximpacket_position (x);
	ret   |= ximpacket_put16 (x, 0, border);
	ret   |= ximpacket_put16 (x, 0, border);	/* padding */
	if (ret != 0) return -1;

	while (len >= 2) {
		unsigned int id;

		id = get16 (data + offset, border);
		len -= 2;

		if (id >= NUMBER_OF_ELEMENTS (xicattr)) {
			/* error */
			ximpacket_init_buffer (x);
			return ximp_bad_length (x, xic->imid, xic->icid,
						border);
		}
		if (xicattr[id].get_ic_value_proc) {
			ret = xicattr[id].get_ic_value_proc (xic, x, id,
							     nest,
							     data + offset,
							     len, border);
			if (ret == -1) return -1;
			else if (ret == 1) return 0;
		}
	}

	end = ximpacket_position (x);
	ximpacket_reput16 (x, end - start, start, border);

	return 0;
}
int
ximic_ic_get_attributes (struct ximic* xic, struct ximpacket* x, char* data,
			 int len, int border)
{
	return ic_get_attributes (xic, x, data, ICATTR_NONE, len, border);
}

struct ximidlist
{
	int	id;
	SLIST_ENTRY(ximidlist)	entry;
};
static SLIST_HEAD(,ximidlist)	imidhead = SLIST_HEAD_INITIALIZER(0);
struct xicidlist
{
	int	id;
	SLIST_ENTRY(xicidlist)	entry;
};
static SLIST_HEAD(,xicidlist)	icidhead = SLIST_HEAD_INITIALIZER(0);

static int
ximimlist_allocate (struct ximim* x)
{
	int maxim = 0;
	struct ximidlist* im;
	struct ximidlist* l;

	im = malloc (sizeof (struct ximidlist));
	if (im == NULL) {
		NOMEMORY ("can't allocate ximid");
		return -1;
	}

	SLIST_FOREACH (l, &imidhead, entry) {
		if (maxim < l->id) maxim = l->id;
	}
	im->id = maxim + 1;
	SLIST_INSERT_HEAD (&imidhead, im, entry);
	x->imid = im->id;

	return 0;
}
static void
ximimlist_destroy (int imid)
{
	struct ximidlist* x = SLIST_FIRST (&imidhead);

	while (x) {
		struct ximidlist* next = SLIST_NEXT (x, entry);
		if (x->id == imid) {
			SLIST_REMOVE (&imidhead, x, ximidlist, entry);
			free (x);
		}
		x = next;
	}
}
static int
ximiclist_allocate (struct ximic* x)
{
	int maxic = 0;
	struct xicidlist* ic;
	struct xicidlist* i;

	ic = malloc (sizeof (struct xicidlist));
	if (ic == NULL) {
		NOMEMORY ("can't allocate xicid");
		return -1;
	}
	memset (ic, 0, sizeof (struct xicidlist));

	SLIST_FOREACH (i, &icidhead, entry) {
		if (maxic < i->id) maxic = i->id;
	}
	ic->id = maxic + 1;
	SLIST_INSERT_HEAD (&icidhead, ic, entry);
	x->icid = ic->id;

	return 0;
}
static void
ximiclist_destroy (int icid)
{
	struct xicidlist* x = SLIST_FIRST (&icidhead);

	while (x) {
		struct xicidlist* next = SLIST_NEXT (x, entry);
		if (x->id == icid) {
			SLIST_REMOVE (&icidhead, x, xicidlist, entry);
			free (x);
		}
		x = next;
	}
}


struct ximic*
ximim_allocate_ic (struct ximim* xim, struct imconnection* imc)
{
	int ret;
	struct ximic* xic;

	xic = malloc (sizeof (struct ximic));
	if (xic == NULL) {
		NOMEMORY ("can't allocate ximic");
		return NULL;
	}
	memset (xic, 0, sizeof (struct ximic));

	xic->imc = imc;
	
	ret = ximiclist_allocate (xic);
	if (ret == -1) {
		free (xic);
		return NULL;
	}
	xic->imid = xim->imid;	/* duplicate entry */

	SLIST_INSERT_HEAD (&xim->iclist, xic, entry);
	return xic;
}
void
ximim_destroy_ic (struct ximim* xim, struct ximic* xic)
{
	conversion_destroy (xic->conv);
	pewindow_destroy (xic->pew);
	ximiclist_destroy (xic->icid);

	SLIST_REMOVE (&xim->iclist, xic, ximic, entry);

	fontset_release (xic->preedit_attr.fontset);
	fontset_release (xic->status_attr.fontset);
	free (xic);
}
struct ximic*
ximim_search_ximic (struct ximim* xim, int icid)
{
	struct ximic* xic;
	SLIST_FOREACH (xic, &xim->iclist, entry) {
		if (xic->icid == icid) return xic;
	}
	return NULL;
}

struct ximim*
ximim_allocate (void)
{
	struct ximim* x;
	int ret;

	x = malloc (sizeof (struct ximim));
	if (x == NULL) {
		NOMEMORY ("can't allocate ximim");
		return NULL;
	}

	memset (x, 0, sizeof (struct ximim));
	SLIST_INIT (&x->iclist);

	ret = ximimlist_allocate (x);
	if (ret == -1) {
		free (x);
		return NULL;
	}
	return x;
}
void
ximim_destroy (struct ximim* x)
{
	struct ximic* xic;

	if (!x) return;

	ximimlist_destroy (x->imid);
	
	xic = SLIST_FIRST (&x->iclist);
	while (xic) {
		struct ximic* n = SLIST_NEXT (xic, entry);
		ximim_destroy_ic (x, xic);
		xic = n;
	}
	free (x);
}
