/* aimetrans.c -- transport facilities for AIME user interface.
   Copyright (C) 2002  TAKAI Kousuke <tak@kmc.gr.jp>.  */

/* $Id: aimetrans.c,v 1.2 2002/03/26 10:47:00 taka Exp $ */

/* FIXME!! non-"local" transport don't work! */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include <string.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <err.h>

#include "aimetrans.h"

struct aimetrans_ops
  {
    struct aimetrans *(*open) (void *);
    void (*close) (struct aimetrans *);
    int (*send) (struct aimetrans *, const char *, size_t);
    int (*receive) (struct aimetrans *, char *, size_t);
    int (*fd_input) (struct aimetrans *);
  };

#define CONTROL_LINE_SIZE	1024

#ifndef PATH_AIMESOCK
# define PATH_AIMESOCK	"/tmp/.aime"
#endif

#define ATF_FD_INPUT		(1 << 0)
#define ATF_XPROPERTY_INPUT	(1 << 1)

static char *
skip_word (const char *p, const char *const limit)
{
  for (; p < limit && *p != ' ' && *p != '\t'; p++)
    if (*p == '\\' && ++p >= limit)
      break;
  return (char *)p;
}

static const char *
get_unsigned_integer (const char *p, const char *const limit,
		      unsigned int *valuep)
{
  unsigned int value;

  if (p >= limit || (value = *p++ - '0') > 9)
    return NULL;
  while (p < limit && '0' <= *p && *p <= '9')
    if (value > UINT_MAX / 10
	|| (value == UINT_MAX / 10 && (*p - '0') > (UINT_MAX % 10)))
      return NULL;
    else
      value = value * 10 + (*p++ - '0');
  *valuep = value;
  return p;
}

static const char *
get_integer (const char *p, const char *const limit, int *valuep)
{
  bool negative = 0;

  if (p < limit && *p == '-')
    ++p, negative = 1;
  p = get_unsigned_integer (p, limit, (unsigned int *)valuep);
  if (p)
    if (negative ? (*valuep = - *(unsigned int *)valuep) >= 0 : *valuep < 0)
      p = NULL;
  return p;
}

static void
handle_command (struct aimetrans *const x, char *p, char *const limit)
{
  const char *cmd;
  int l;

#define SKIP_BLANKS()	while (p < limit && (*p == ' ' || *p == '\t')) p++
#define NO_ARGUMENTS()	do if (p < limit) goto no_arguments; while (0)
#define INT_ARGUMENT(Var) do {				\
    if (!(p = (char *) get_integer (p, limit, &(Var)))	\
	|| (p < limit && *p != ' ' && *p != '\t'))	\
      goto bad_integer;					\
  } while (0)
#define UINT_ARGUMENT(Var) do {					\
    if (!(p = (char *) get_unsigned_integer (p, limit, &(Var)))	\
	|| (p < limit && *p != ' ' && *p != '\t'))		\
      goto bad_integer;						\
  } while (0)

  SKIP_BLANKS ();
  printf ("R <- %.*s\n", (int) (limit - p), p);
  cmd = p;
  p = skip_word (cmd, limit);
  l = (const char *)p - cmd;
  SKIP_BLANKS ();
  switch (l)
    {
    case 2:
      if (!memcmp (cmd, "on", 2))
	{
	  NO_ARGUMENTS ();
	  aimetrans_cb_onoff (x, 1);
	  return;
	}
      break;
    case 3:
      if (!memcmp (cmd, "off", 3))
	{
	  NO_ARGUMENTS ();
	  aimetrans_cb_onoff (x, 0);
	  return;
	}
      break;
    case 4:
      if (!memcmp (cmd, "mode", 4))
	{
	  const char *modename = p;
	  int l;
	  enum aimetrans_mode mode;

	  p = skip_word (p, limit);
	  l = (const char *)p - modename;
	  SKIP_BLANKS ();
	  switch (l)
	    {
	    case 4:
	      if (!memcmp (modename, "hira", 4))
		mode = AMODE_HIRA;
	      else if (!memcmp (modename, "kata", 4))
		mode = AMODE_KATA;
	      else if (!memcmp (modename, "kana", 4))
		mode = AMODE_KANA;
	      else
		goto unknown_mode;
	      break;
	    case 5:
	      if (memcmp (modename, "ascii", 5))
		goto unknown_mode;
	      mode = AMODE_ASCII;
	      break;
	    case 9:
	      if (memcmp (modename, "zen_ascii", 9))
		goto unknown_mode;
	      mode = AMODE_ZEN_ASCII;
	      break;
	    default:
	    unknown_mode:
	      warnx ("unknown mode `%.*s'", l, modename);
	      return;
	    }
	  NO_ARGUMENTS ();
	  aimetrans_cb_mode (x, mode);
	  return;
	}
      break;
    case 9:
      if (!memcmp (cmd, "candidate", 9))
	{
	  int l;
	  int id;
	  const char *subcommand;
	  unsigned int n;

	  INT_ARGUMENT (id);
	  SKIP_BLANKS ();
	  subcommand = p;
	  p = skip_word (p, limit);
	  l = (const char *)p - subcommand;
	  SKIP_BLANKS ();
	  switch (l)
	    {
	    case 4:
	      if (!memcmp (subcommand, "cand", 4))
		goto do_candidate;
	      if (!memcmp (subcommand, "stop", 4))
		{
		  NO_ARGUMENTS ();
		  aimetrans_cb_candidate_stop (x, id);
		  return;
		}
	      break;
	    case 5:
	      if (!memcmp (subcommand, "start", 4))
		{
		  UINT_ARGUMENT (n);
		  SKIP_BLANKS ();
		  NO_ARGUMENTS ();
		  aimetrans_cb_candidate_start (x, id, n);
		  return;
		}
	      break;
	    case 6:
	      if (!memcmp (subcommand, "select", 6))
		{
		  UINT_ARGUMENT (n);
		  SKIP_BLANKS ();
		  NO_ARGUMENTS ();
		  aimetrans_cb_candidate_select (x, id, n);
		  return;
		}
	      if (!memcmp (subcommand, "window", 6))
		{
		  int wx, wy;

		  INT_ARGUMENT (wx);
		  SKIP_BLANKS ();
		  INT_ARGUMENT (wy);
		  SKIP_BLANKS ();
		  NO_ARGUMENTS ();
		  aimetrans_cb_candidate_coord (x, id, wx, wy);
		  return;
		}
	      break;
	    case 9:
	      if (!memcmp (subcommand, "candidate", 9))
		{
		  char *str;

		do_candidate:
		  UINT_ARGUMENT (n);
		  SKIP_BLANKS ();
		  str = alloca (limit - p + 1);
		  memcpy (str, p, limit - p);
		  str[limit - p] = '\0';
		  aimetrans_cb_candidate (x, id, n, str);
		  return;
		}
	      break;
	    }
	  warnx ("unknown candidate subcommand `%.*s'", l, subcommand);
	  return;
	}
      break;
    }
  warnx ("unknown control command `%.*s'", l, cmd);
  return;

 no_arguments:
  warnx ("excessive argument `%.*s' in command `%.*s'",
	 limit - p, p, l, cmd);
  return;

 bad_integer:
  warnx ("bad integer in command `%.*s'", l, cmd);
  return;
}

static struct aimetrans *local_open (void *);

static void
local_close (struct aimetrans *const x)
{
  close (x->fd);
}

static int
local_fd_input (struct aimetrans *const x)
{
  char *p, *nl, *bol, *end;
  ssize_t nr = read (x->fd, x->rbuf + x->rbuf_cur, x->rbuf_size - x->rbuf_cur);

  if (nr <= 0)
    {
      if (nr)
	{
	  fprintf (stderr, "AIMEtrans: read: %s\n", strerror (errno));
	  return -1;
	}
      fprintf (stderr, "AIMEtrans: server connection closed\n");
      return 1;
    }

  end = (p = (bol = x->rbuf) + x->rbuf_cur) + nr;
  while ((nl = memchr (p, '\n', end - p)) != NULL)
    {
      handle_command (x, bol, nl);
      p = bol = nl + 1;
    }
  x->rbuf_cur = end - bol;
  if (x->rbuf_cur)
    memmove (x->rbuf, bol, x->rbuf_cur);
  return 0;
}

static const struct aimetrans_ops local_ops = {
  .open		= local_open,
  .close	= local_close,
  /*  .send		= local_send,
      .receive	= local_receive, */
  .fd_input	= local_fd_input,
};

static struct aimetrans *
local_open (void *sockpath)
{
  int fd;
  size_t plen;
  struct sockaddr_un addr;
  struct aimetrans *const x = malloc (sizeof (struct aimetrans));

  if (!x)
    {
      errno = ENOMEM;
      return NULL;
    }

  x->rbuf_size = CONTROL_LINE_SIZE;
  if (!(x->rbuf = malloc (x->rbuf_size)))
    {
      free (x);
      errno = ENOMEM;
      return NULL;
    }
  x->rbuf_cur = 0;

  if (!sockpath)
    sockpath = PATH_AIMESOCK;

  plen = strlen (sockpath) + 1;
  if (plen > sizeof (addr.sun_path))
    {
#ifdef ENAMETOOLONG
      errno = ENAMETOOLONG;
#endif
      return NULL;
    }

  memcpy (addr.sun_path, sockpath, plen);
  addr.sun_family = AF_LOCAL;

  fd = socket (PF_LOCAL, SOCK_STREAM, 0);
  if (fd < 0)
    return NULL;
  if (connect (fd, (struct sockaddr *) &addr,
	       offsetof (struct sockaddr_un, sun_path) + plen))
    {
      int save = errno;

      close (fd);
      errno = save;
      return NULL;
    }

  x->fd = fd;
  x->ops = &local_ops;
  return x;
}

#include <X11/Xlib.h>

struct xtrans
  {
    struct aimetrans ax;
    Display *dpy;
  };

static struct aimetrans *x_open (void *);

static void
x_close (struct aimetrans *x)
{
}

static const struct aimetrans_ops x_ops = {
  .open		= x_open,
  .close	= x_close,
  /* .flags	= ATF_XPROPERTY_INPUT, */
};

static struct aimetrans *
x_open (void *vdpy)
{
  struct xtrans *const x = malloc (sizeof (struct xtrans));

  if (!x)
    {
#ifdef ENOMEM
      errno = ENOMEM;
#endif
      return NULL;
    }

  x->dpy = (Display *) vdpy;
  x->ax.fd = ConnectionNumber (x->dpy);
  x->ax.ops = &x_ops;
  return &x->ax;
}

struct aimetrans *
aimetrans_open (const char *type, void *peer)
{
  if (!strcmp (type, "local")
      || !strcmp (type, "unix")
      || !strcmp (type, "file"))
    return local_open (peer);
  if (!strcmp (type, "x"))
    return x_open (peer);
  errno = EINVAL;
  return NULL;
}

void
aimetrans_close (struct aimetrans *x)
{
  x->ops->close (x);
  free (x);
}

bool
aimetrans_fd_input_needed (const struct aimetrans *const x)
{
  return x->ops->fd_input != NULL;
}

int
aimetrans_process_fd_input (struct aimetrans *const x)
{
  if (x->ops->fd_input)
    return x->ops->fd_input (x);
  return -1;
}
