#ifndef ALTDBCONNECTION_H
#define ALTDBCONNECTION_H
/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altDBConnection.h 313 2007-09-03 10:43:12Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.
 
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *---------------------------------------------------------------------------*/
/**
 * \file    altDBConnection.h
 * \brief   DB Connection Class
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include <altDB/altDBResultSet.h>
#include <altDB/altSQL.h>
#include <altDB/altPreparedStatement.h>

#include <map>

/*----------------------------------------------------------------*
 * Type Definition
 *----------------------------------------------------------------*/
typedef std::map<altStr, altPreparedStatement *> altPreparedStatementMap;

/*----------------------------------------------------------------*
 * Class Definition
 *----------------------------------------------------------------*/
///
/// \brief  DB Connection
///
class altDBConnection : public altBase
{
public:
  ///
  /// \brief  Constructor
  ///
  altDBConnection();

  ///
  /// \brief  Destructor
  ///
  virtual ~altDBConnection();

  ///
  /// \brief  Connect
  ///
  /// \param  sHost     [I ] Host Name
  /// \param  nPort     [I ] Port Number
  /// \param  sDB       [I ] DB Name
  /// \param  sUser     [I ] User ID
  /// \param  sPass     [I ] Password
  /// \param  sCharset  [I ] Charcter Set
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  virtual alt_t Connect(const altStr & sHost, const altUInt nPort, const altStr & sDB, const altStr & sUser, const altStr & sPass, const altStr & sCharset, const altBool bAutoCommit = false) = 0;

  ///
  /// \brief  Disconnect
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
  virtual alt_t Disconnect() = 0;

  ///
  /// \brief  Execute SQL
  ///
  /// \param  oSQL    [I ] SQL
  /// \param  oResult [ O] Result Set
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_SQL     SQL Error
  ///
	virtual alt_t ExecuteSQL(const altSQL & oSQL, altDBResultSet & oResult) = 0;

  ///
  /// \brief  Execute SQL
  ///
  /// \param  oSQL          [I ] SQL
  /// \param  pAffectedRow  [ O] Affected Row
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_SQL     SQL Error
  ///
	virtual alt_t ExecuteSQL(const altSQL & oSQL, altULongLong * pAffectedRow = NULL) = 0;

  ///
  /// \brief  Get Prepared Statement
  ///
  /// \param  oSQL        [I ] SQL
  /// \param  pStatement  [ O] Prepared Statement
  ///
  /// \return ALT_S_SUCCESS Success
  /// \return ALT_E_SQL     SQL Error
  ///
  virtual alt_t GetPreparedStatement(const altSQL & oSQL, altPreparedStatement * & pStatement) = 0;

  ///
  /// \brief  Execute SQL
  ///
  /// \param  oStmt       [IO] Prepared Statement
  /// \param  oResultSet  [ O] ResultSet
  ///
  /// \return ALT_S_SUCCESS Success
  /// \return ALT_E_SQL     SQL Error
  ///
  virtual alt_t ExecuteSQL(altPreparedStatement & oStmt, altDBResultSet & oResultSet) = 0;

  ///
  /// \brief  Execute SQL
  ///
  /// \param  oStmt         [IO] Prepared Statement
  /// \param  pAffectedRow  [ O] ResultSet
  ///
  /// \return ALT_S_SUCCESS Success
  /// \return ALT_E_SQL     SQL Error
  ///
  virtual alt_t ExecuteSQL(altPreparedStatement & oStmt, altULongLong * pAffectedRow = NULL) = 0;

  ///
  /// \brief  Commit
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
	virtual alt_t Commit() = 0;

  ///
  /// \brief  Rollback
  ///
  /// \return ALT_S_SUCCESS success
  /// \return ALT_E_ERROR   error
  ///
	virtual alt_t Rollback() = 0;

  ///
  /// \brief  Set Explain Log
  ///
  /// \param  bEnable [I ] Enable or not
  /// \param  sLogDir [I ] Explain Log Directory
  ///
  /// \return ALT_S_SUCCESS   success
  ///
  alt_t SetExplainLog(altBool bEnable, const altStr & sLogDir);

  ///
  /// \brief  Clear Prepared Statement Cache
  ///
  /// \return ALT_S_SUCCESS Success
  ///
  alt_t ClearPreparedStatementCache();

protected:
  altBool                 m_bExplainLogEnable;      ///< Explain Log Enable
  altStr                  m_bExplainLogDir;         ///< Explain LOg Directory
  altPreparedStatementMap m_oPreparedStatementMap;  ///< Prepared Statement Cache
};

#endif  //ALTDBCONNECTION_H
