/*--------------------------------------------------------------------------*

   Alternative Llibrary

  $Id: altSyncThread.cpp 553 2007-11-10 03:15:25Z nekosys $

  Copyright (C) 2007 NEKO SYSTEM
 
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.
 
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *---------------------------------------------------------------------------*/
/**
 * \file    altSyncThread.cpp
 * \brief   thread sync
 * \date    2007
 * \author  NEKO SYSTEM
 */
/*----------------------------------------------------------------*
 * Include
 *----------------------------------------------------------------*/
#include "altSyncThread.h"

#ifdef ALT_LINUX
  #include <sys/time.h>
  #include <errno.h>
#endif

/*----------------------------------------------------------------*
 * Function Implements
 *----------------------------------------------------------------*/
///
/// \brief  Constructor
///
LIBALT_API altSyncThread::altSyncThread() :
m_bLock (false)
{
#ifdef ALT_WIN
  InitializeCriticalSection (& m_oCriticalSection);
  m_oEvent = CreateEvent (NULL, false, true, NULL);
#else
  pthread_mutex_init (& m_oMutex, NULL);
  pthread_cond_init (& m_oCond, NULL);
#endif
}

///
/// \brief  Destructor
///
LIBALT_API altSyncThread::~altSyncThread()
{
#ifdef ALT_WIN
  CloseHandle (m_oEvent);
  DeleteCriticalSection (& m_oCriticalSection);
#else
  altInt nRet = pthread_cond_destroy (& m_oCond);
  if (nRet == EBUSY) {
    SetSignal();
    pthread_cond_destroy (& m_oCond);
  }
  pthread_mutex_destroy (& m_oMutex);
#endif
}

///
/// \brief  Begin Lock
///
LIBALT_API void altSyncThread::BeginLock() const
{
#ifdef ALT_WIN
  EnterCriticalSection (& m_oCriticalSection);
#endif
#ifdef ALT_LINUX
  pthread_mutex_lock (& m_oMutex);
#endif
  m_bLock = true;
}

///
/// \brief  End Lock
///
LIBALT_API void altSyncThread::EndLock() const
{
  if (m_bLock) {
    m_bLock = false;
#ifdef ALT_WIN
    LeaveCriticalSection (& m_oCriticalSection);
#endif
#ifdef ALT_LINUX
    pthread_mutex_unlock (& m_oMutex);
#endif
  }
}

///
/// \brief  Wait Signal
///
/// \param  dwTimeout   [I ] timeout(msec) (default value is no timeout)
///
void altSyncThread::WaitSignal(DWORD dwTimeout) const
{
#ifdef ALT_WIN
  WaitForSingleObject (m_oEvent, dwTimeout);
#else
  if (dwTimeout == INFINITE) {
    pthread_mutex_lock (& m_oMutex);
    pthread_cond_wait (& m_oCond, & m_oMutex);
    pthread_mutex_unlock (& m_oMutex);
  }
  else {
    struct timeval  oNow;
    struct timespec oTimeout;

    gettimeofday (& oNow, NULL);
    
    time_t  nSec = oNow.tv_sec + (dwTimeout / 1000);
    altLong nNanoSec = (oNow.tv_usec * 1000) + ((dwTimeout % 1000) * 1000000);

    if (nNanoSec >= 1000000000) {
      nNanoSec -= 1000000000;
      nSec += 1;
    }

    oTimeout.tv_sec = nSec;
    oTimeout.tv_nsec = nNanoSec;

    pthread_mutex_lock (& m_oMutex);
    pthread_cond_timedwait (& m_oCond, & m_oMutex, & oTimeout);
    pthread_mutex_unlock (& m_oMutex);
  }
#endif
}

///
/// \brief	Set Signal
///
void altSyncThread::SetSignal() const
{
#ifdef ALT_WIN
  SetEvent (m_oEvent);
#else
  pthread_cond_signal (& m_oCond);
#endif
}
