package tk.eclipse.plugin.visualjsf;

import java.io.ByteArrayInputStream;
import java.util.EventObject;
import java.util.HashSet;
import java.util.Set;

import jp.aonir.fuzzyxml.FuzzyXMLDocument;
import jp.aonir.fuzzyxml.FuzzyXMLElement;
import jp.aonir.fuzzyxml.FuzzyXMLNode;
import jp.aonir.fuzzyxml.FuzzyXMLParser;
import jp.aonir.fuzzyxml.FuzzyXMLText;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteDrawer;
import org.eclipse.gef.palette.PaletteGroup;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.SelectionToolEntry;
import org.eclipse.gef.palette.ToolEntry;
import org.eclipse.gef.ui.actions.DeleteAction;
import org.eclipse.gef.ui.actions.RedoRetargetAction;
import org.eclipse.gef.ui.actions.UndoRetargetAction;
import org.eclipse.gef.ui.parts.GraphicalEditorWithPalette;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;

import tk.eclipse.plugin.htmleditor.HTMLPlugin;
import tk.eclipse.plugin.htmleditor.HTMLProjectParams;
import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.jsf.JSFProject;
import tk.eclipse.plugin.jsf.ManagedBean;
import tk.eclipse.plugin.jsf.editors.FacesConfigOperation;
import tk.eclipse.plugin.jsf.wizards.ManagedBeanWizard;
import tk.eclipse.plugin.visualjsf.editparts.AbstractEditPart;
import tk.eclipse.plugin.visualjsf.editparts.JSFEditPartFactory;
import tk.eclipse.plugin.visualjsf.loader.VisualJSFLoader;
import tk.eclipse.plugin.visualjsf.models.CommandButtonModel;
import tk.eclipse.plugin.visualjsf.models.CommandLinkModel;
import tk.eclipse.plugin.visualjsf.models.DataTableModel;
import tk.eclipse.plugin.visualjsf.models.InputSecretModel;
import tk.eclipse.plugin.visualjsf.models.InputTextModel;
import tk.eclipse.plugin.visualjsf.models.InputTextareaModel;
import tk.eclipse.plugin.visualjsf.models.OutputTextModel;
import tk.eclipse.plugin.visualjsf.models.RootModel;
import tk.eclipse.plugin.visualjsf.models.SelectBooleanCheckboxModel;
import tk.eclipse.plugin.visualjsf.models.SelectManyCheckboxModel;
import tk.eclipse.plugin.visualjsf.models.SelectManyListboxModel;
import tk.eclipse.plugin.visualjsf.models.SelectOneMenuModel;
import tk.eclipse.plugin.visualjsf.models.SelectOneRadioModel;

/**
 * @author Naoki Takezoe
 */
public class VisualJSFEditor extends GraphicalEditorWithPalette {
	
	private RootModel root;
	private boolean savePreviouslyNeeded;
	private Set componentNames = new HashSet();
	
	public static final String MENU_GROUP_EDIT = "edit";
	public static final String MENU_GROUP_BEAN = "backingBean";
	public static final String MENU_GROUP_COMPONENT = "component";
	public static final String MENU_GROUP_ADDITIONS = "additions";
	
	
	public VisualJSFEditor(){
		super();
		setEditDomain(new DefaultEditDomain(this));
		getActionRegistry().registerAction(new UndoRetargetAction());
		getActionRegistry().registerAction(new RedoRetargetAction());
	}
	
	protected PaletteRoot getPaletteRoot() {
		PaletteRoot root = new PaletteRoot();
		
		PaletteGroup tools = new PaletteGroup(VisualJSFPlugin.getResourceString("palette.tool"));
		// selection tool
		ToolEntry tool = new SelectionToolEntry();
		tools.add(tool);
		root.setDefaultEntry(tool);
		// marquee tool
		tool = new MarqueeToolEntry();
		tools.add(tool);
		
		// creation model tools
		PaletteDrawer html = new PaletteDrawer(VisualJSFPlugin.getResourceString("palette.component"));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.outputText"),
				"h:outputText", OutputTextModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_LABEL)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.inputText"),
				"h:inputText", InputTextModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_TEXT)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.inputTextarea"),
				"h:inputTextarea", InputTextareaModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_TEXTAREA)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.inputSecret"),
				"h:inputSecret", InputSecretModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_PASS)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.commandButton"),
				"h:commandButton", CommandButtonModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_BUTTON)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.commandLink"),
				"h:commandLink", CommandLinkModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_LINK)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.selectBooleanCheckbox"),
				"h:selectBooleanCheckbox", SelectBooleanCheckboxModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_CHECK)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.selectManyCheckbox"),
				"h:selectManyCheckbox", SelectManyCheckboxModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_CHECK)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.selectOneRadio"),
				"h:selectOneRadio", SelectOneRadioModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_RADIO)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.selectOneMenu"),
				"h:selectOneMenu", SelectOneMenuModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_SELECT)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.selectManyListbox"),
				"h:selectManyListbox", SelectManyListboxModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_LIST)));
		
		html.add(createEntry(VisualJSFPlugin.getResourceString("component.dataTable"),
				"h:dataTable", DataTableModel.class, 
				HTMLPlugin.getDefault().getImageRegistry().getDescriptor(HTMLPlugin.ICON_TABLE)));
		
		// adds groups to the root.
		root.add(tools);
		root.add(html);
		
		return root;
	}
	
	private CreationToolEntry createEntry(String label, String tooltip, Class clazz, ImageDescriptor image){
		CreationToolEntry entry = new CreationToolEntry(
				label, tooltip, new VisualJSFModelFactory(clazz, componentNames), image, image);
		return entry;
	}
	
	private String getBackingBeanName(){
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try {
				IFile file = ((IFileEditorInput)input).getFile();
				
				HTMLProjectParams params = new HTMLProjectParams(file.getProject());
				String root = params.getRoot();
				if(root.startsWith("/")){
					root = root.substring(1);
				}
				String path = file.getProjectRelativePath().toString();
				path = path.substring(root.length());
				return VisualJSFUtil.getBackingBeanName(path);
			} catch(Exception ex){
				VisualJSFPlugin.logException(ex);
			}
		}
		return null;
	}
	
	private void checkBackingBean(){
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try {
				IFile file = ((IFileEditorInput)input).getFile();
				IJavaProject javaProject = JavaCore.create(file.getProject());
				JSFProject jsfProject = new JSFProject(javaProject);
				
				String beanName = getBackingBeanName();
				IType beanType = null;
				
				ManagedBean bean = jsfProject.getManagedBean(beanName);
				if(bean!=null){
					beanType = javaProject.findType(bean.getClassName());
				}
				if(beanType==null || !beanType.exists()){
					beanType = null;
					beanName = null;
				}
//				if(beanType==null || beanName==null){
//					// TODO debug
//					System.out.println("Backing Bean '" + beanName + "' doesn't exist!");
//				}
				this.root.setBackingBean(beanType, beanName);
				
			} catch(Exception ex){
				VisualJSFPlugin.logException(ex);
			}
		}
	}

	/**
	 * Returns an editor action.
	 */
	public IAction getAction(Object key){
		return getActionRegistry().getAction(key);
	}
	
	protected void configureGraphicalViewer() {
		super.configureGraphicalViewer();
		GraphicalViewer viewer = getGraphicalViewer();
		viewer.setEditPartFactory(new JSFEditPartFactory());
	}

	protected void initializeGraphicalViewer() {
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try {
				IFile file = ((IFileEditorInput)input).getFile();
				String source = new String(HTMLUtil.readStream(file.getContents()), file.getCharset());
				this.root = VisualJSFLoader.loadJSP(source, componentNames);
//				this.root.setComponentNames(componentNames);
				checkBackingBean();
			} catch(Exception ex){
				VisualJSFPlugin.logException(ex);
				// TODO error
			}
		} else {
			// TODO error
		}
		
		GraphicalViewer viewer = getGraphicalViewer();
		ScalableRootEditPart rootEditPart = new ScalableRootEditPart();
		viewer.setRootEditPart(rootEditPart);
		viewer.setContents(this.root);
		
		final DeleteAction deleteAction = new DeleteAction((IWorkbenchPart) this);
		deleteAction.setSelectionProvider(getGraphicalViewer());
		getActionRegistry().registerAction(deleteAction);
		getGraphicalViewer().addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				deleteAction.update();
			}
		});
		
		MenuManager menuMgr = new MenuManager();
//		menuMgr.add(getAction(OPEN_SELECTION));
		menuMgr.add(new Separator(MENU_GROUP_EDIT));
		menuMgr.add(getAction(ActionFactory.UNDO.getId()));
		menuMgr.add(getAction(ActionFactory.REDO.getId()));
		menuMgr.add(getAction(ActionFactory.DELETE.getId()));
		menuMgr.add(new Separator(MENU_GROUP_BEAN));
		menuMgr.add(new Separator(MENU_GROUP_COMPONENT));
		menuMgr.add(new Separator(MENU_GROUP_ADDITIONS));
//		menuMgr.add(new Separator("zoom"));
//		menuMgr.add(getActionRegistry().getAction(GEFActionConstants.ZOOM_IN));
//		menuMgr.add(getActionRegistry().getAction(GEFActionConstants.ZOOM_OUT));
//		menuMgr.add(new Separator("print"));
//		menuMgr.add(new SaveAsImageAction(viewer));
//		menuMgr.add(printAction);
		viewer.setContextMenu(menuMgr);
		
		menuMgr.addMenuListener(new IMenuListener(){
			public void menuAboutToShow(IMenuManager manager) {
				// update actions for the backing-bean
				manager.remove(CreateBackingBeanAction.ID);
				manager.remove(OpenBackingBeanAction.ID);
				if(root.getBeanType()!=null){
					manager.appendToGroup(MENU_GROUP_BEAN, new OpenBackingBeanAction());
				} else {
					manager.appendToGroup(MENU_GROUP_BEAN, new CreateBackingBeanAction());
				}
				// update actions for the selected component
				IContributionItem[] items = manager.getItems();
				boolean deleteFlag = true;
				for(int i=0;i<items.length;i++){
					if(items[i].isSeparator()){
						deleteFlag = items[i].getId().equals(MENU_GROUP_COMPONENT);
					} else if(deleteFlag){
						manager.remove(items[i]);
					}
				}
				IStructuredSelection sel = (IStructuredSelection)getGraphicalViewer().getSelection();
				if(sel.getFirstElement()!=null){
					Object obj = sel.getFirstElement();
					if(obj instanceof AbstractEditPart){
						IAction[] actions = ((AbstractEditPart)obj).getMenuActions();
						if(actions!=null){
							for(int i=0;i<actions.length;i++){
								manager.appendToGroup(MENU_GROUP_COMPONENT, actions[i]);
							}
						}
					}
				}
			}
		});
	}
	
	public void init(IEditorSite site, IEditorInput input) throws PartInitException {
		super.init(site, input);
		setPartName(getEditorInput().getName());
	}
	
    protected void setInput(IEditorInput input) {
    	super.setInput(input);
    	if(this.root != null){
    		checkBackingBean();
    	}
    }
	
	
	public void doSave(IProgressMonitor monitor) {
		IEditorInput input = getEditorInput();
		if(input instanceof IFileEditorInput){
			try {
				IFile file = ((IFileEditorInput)input).getFile();
				file.setContents(new ByteArrayInputStream(this.root.toHTML().getBytes()),
						false, true, monitor);
			} catch(Exception ex){
				VisualJSFPlugin.logException(ex);
			}
		} else {
			// TODO Can't save to the file.
		}
		
		setPartName(getEditorInput().getName());
		getCommandStack().markSaveLocation();
	}

	public void doSaveAs() {
		doSave(new NullProgressMonitor());
	}
	
	public boolean isSaveOnCloseNeeded() {
		return getCommandStack().isDirty();
	}
	
	public boolean isDirty() {
		return isSaveOnCloseNeeded();
	}

	public boolean isSaveAsAllowed() {
		return true;
	}
	
	private void setSavePreviouslyNeeded(boolean value) {
		savePreviouslyNeeded = value;
	}
	
	private boolean savePreviouslyNeeded() {
		return savePreviouslyNeeded;
	}
	
	public void commandStackChanged(EventObject event) {
		if (isDirty()) {
			if (!savePreviouslyNeeded()) {
				setSavePreviouslyNeeded(true);
				firePropertyChange(IEditorPart.PROP_DIRTY);
			}
		} else {
			setSavePreviouslyNeeded(false);
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
		super.commandStackChanged(event);
	}
	
	private class CreateBackingBeanAction extends Action {
		
		public static final String ID = "registerBackingBean";
		
		public CreateBackingBeanAction(){
			super(VisualJSFPlugin.getResourceString("action.createBackingBean"),
					VisualJSFPlugin.getImageDescriptor("icons/bean.gif"));
			setId(ID);
		}
		
		public void run(){
			IFileEditorInput input = (IFileEditorInput)getEditorInput();
			IFile file = input.getFile();
			
			ManagedBeanWizard wizard = new ManagedBeanWizard();
			wizard.init(PlatformUI.getWorkbench(),new StructuredSelection(file.getProject()));
			
			// TODO It must specify the web application relative path instead of IFile#getName().
			wizard.setClassName(VisualJSFPlugin.BEAN_PACKAGE + "." + 
					VisualJSFUtil.getBackingBeanClassName(file.getName()));
			
			WizardDialog dialog = new WizardDialog(
					getGraphicalViewer().getControl().getShell(), wizard);
			
			if(dialog.open()==WizardDialog.OK){
				JSFProject project = new JSFProject(JavaCore.create(file.getProject()));
				IFile[] configFiles = project.getFacesConfigXML();
				if(configFiles.length > 0){
					try {
						IType type = (IType)wizard.getCreatedElement();
						registerManagedBean(configFiles[0], type.getFullyQualifiedName(), getBackingBeanName());
					} catch(Exception ex){
						VisualJSFPlugin.logException(ex);
					}
					checkBackingBean();
				}
			}
		}
		
		/**
		 * This methods was copied from {@link FacesConfigOperation#addManagedBean()}.
		 */
		private void registerManagedBean(IFile config, String className, String beanName) throws Exception {
			
			FuzzyXMLDocument doc = new FuzzyXMLParser().parse(config.getContents());
			FuzzyXMLElement root = (FuzzyXMLElement)doc.getDocumentElement().getChildren()[0];
			
			FuzzyXMLNode[] children = root.getChildren();
			FuzzyXMLNode before = null;
			boolean flag = false;
			for(int i=0;i<children.length;i++){
				if(children[i] instanceof FuzzyXMLElement){
					if(flag==false && ((FuzzyXMLElement)children[i]).getName().equals("managed-bean")){
						flag = true;
					}
					if(flag==true && !((FuzzyXMLElement)children[i]).getName().equals("managed-bean")){
						before = children[i];
						break;
					}
				}
			}
			
			FuzzyXMLElement bean  = doc.createElement("managed-bean");
			if(before==null){
				root.appendChild(doc.createText("\t"));
				root.appendChild(bean);
			} else {
				FuzzyXMLText text = doc.createText("\t");
				root.insertBefore(text,before);
				root.insertAfter(bean,text);
			}
			
			bean.appendChild(doc.createText("\n\t\t"));
			FuzzyXMLElement name  = doc.createElement("managed-bean-name");
			bean.appendChild(name);
			name.appendChild(doc.createText(beanName));
			bean.appendChild(doc.createText("\n\t\t"));
			
			FuzzyXMLElement clazz = doc.createElement("managed-bean-class");
			bean.appendChild(clazz);
			clazz.appendChild(doc.createText(className));
			bean.appendChild(doc.createText("\n\t\t"));
			
			FuzzyXMLElement scope = doc.createElement("managed-bean-scope");
			bean.appendChild(scope);
			scope.appendChild(doc.createText("request"));
			bean.appendChild(doc.createText("\n\t"));
			
			if(before==null){
				root.appendChild(doc.createText("\n"));
			} else {
				root.insertBefore(doc.createText("\n"),before);
			}
			
			// Generates faces-config.xml
			StringBuffer sb = new StringBuffer();
			sb.append("<?xml version=\"1.0\"?>\n");
			sb.append("<!DOCTYPE faces-config PUBLIC ");
			sb.append("\"-//Sun Microsystems, Inc.//DTD JavaServer Faces Config 1.1//EN\" ");
			sb.append("\"http://java.sun.com/dtd/web-facesconfig_1_1.dtd\">\n");
			sb.append(root.toXMLString());
			
			ByteArrayInputStream in = new ByteArrayInputStream(sb.toString().getBytes(config.getCharset()));
			config.setContents(in, true, true, new NullProgressMonitor());
		}
	}
	
	private class OpenBackingBeanAction extends Action {
		
		public static final String ID = "openBackingBean";
		
		public OpenBackingBeanAction(){
			super(VisualJSFPlugin.getResourceString("action.openBackingBean"), 
					VisualJSFPlugin.getImageDescriptor("icons/bean.gif"));
			setId(ID);
		}
		
		public void run(){
			IType type = root.getBeanType();
			VisualJSFUtil.openJavaElement(type);
		}
	}

}
