package tk.eclipse.plugin.visualjsf.models;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.eclipse.jdt.core.IType;
import org.eclipse.ui.views.properties.TextPropertyDescriptor;

import tk.eclipse.plugin.htmleditor.HTMLUtil;

/**
 * The root model.
 * 
 * @author Naoki Takezoe
 */
public class RootModel extends AbstractModel implements ContainerModel {
	
	public static final String P_CHILDREN = "_children";
	public static final String P_TITLE = "_title";
	
	private String title = "";
	private ArrayList children = new ArrayList();
	private IType beanType;
	private String beanName;
	private Set componentNames;
	
	public RootModel(Set componentNames){
		addModelProperty(P_TITLE, new ModelProperty(this, "title", String.class, 
				new TextPropertyDescriptor(P_TITLE, "title")));
		this.componentNames = componentNames;
	}
	
	public void setBackingBean(IType type, String name){
		this.beanType = type;
		this.beanName = name;
	}
	
	public IType getBeanType(){
		return this.beanType;
	}
	
	public String getBeanName(){
		return this.beanName;
	}
	
	/**
	 * Add the child model.
	 * 
	 * @param child the child model
	 */
	public void addChild(AbstractJSFModel child){
		children.add(child);
		child.added(this.beanType, this.beanName);
		componentNames.add(child.getName());
		child.setRoot(this);
		firePropertyChange(P_CHILDREN,null,null);
	}
	
	/**
	 * Returns all child models as the List.
	 * 
	 * @return all child models
	 */
	public List getChildren(){
		return children;
	}
	
	/**
	 * Removes the child model.
	 * 
	 * @param obj the child model
	 */
	public void removeChild(AbstractJSFModel obj){
		children.remove(obj);
		obj.removed(this.beanType, this.beanName);
		componentNames.remove(obj.getName());
		firePropertyChange(P_CHILDREN, null, null);
	}
	
	/**
	 * Removes all child models.
	 */
	public void removeAll(){
		for(int i=0;i<children.size();i++){
			AbstractJSFModel child = (AbstractJSFModel)children.get(i);
			child.removed(this.beanType, this.beanName);
			componentNames.remove(child.getName());
		}
		children.clear();
		firePropertyChange(P_CHILDREN, null, null);
	}
	
	/**
	 * Returns the page title.
	 * 
	 * @return the page title
	 */
	public String getTitle() {
		return title;
	}
	
	/**
	 * Sets the page title.
	 * 
	 * @param title the page title
	 */
	public void setTitle(String title) {
		this.title = title;
		firePropertyChange(P_TITLE, null, title);
	}
	
	public String toHTML(){
		StringBuffer sb = new StringBuffer();
		sb.append("<%@ page contentType=\"text/html; charset=Windows-31J\" %>\n");
		sb.append("<%@ taglib uri=\"http://java.sun.com/jsf/html\" prefix=\"h\" %>\n");
		sb.append("<%@ taglib uri=\"http://java.sun.com/jsf/core\" prefix=\"f\" %>\n");
		sb.append("<html>\n");
		sb.append("\t<head>\n");
		sb.append("\t\t<meta http-equiv=\"Content-Type\" content=\"text/html; charset=Windows-31J\"/>\n");
		sb.append("\t\t<title>").append(HTMLUtil.escapeHTML(getTitle())).append("</title>\n");
		sb.append("\t</head>\n");
		sb.append("\t<body>\n");
		sb.append("\t\t<f:view>\n");
		sb.append("\t\t\t<h:form>\n");
		for(int i=0;i<children.size();i++){
			AbstractJSFModel model = (AbstractJSFModel)children.get(i);
			sb.append("\t\t\t\t");
			sb.append(model.toHTML());
			sb.append("\n");
		}
		sb.append("\t\t\t</h:form>\n");
		sb.append("\t\t</f:view>\n");
		sb.append("\t</body>\n");
		sb.append("</html>\n");
		return sb.toString();
	}
}
