package net.java.amateras.db.util;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.java.amateras.db.view.dialect.IDialect;
import net.java.amateras.db.visual.model.ColumnModel;
import net.java.amateras.db.visual.model.ForeignKeyMapping;
import net.java.amateras.db.visual.model.ForeignKeyModel;
import net.java.amateras.db.visual.model.RootModel;
import net.java.amateras.db.visual.model.TableModel;

public class DatabaseInfo {
	
	private String uri = "";
	private String user = "";
	private String password = "";
	private String catalog = "";
	private String schema = "";
	private Driver driver = null;
	private boolean enableView = false;
	private String productName = null;
	
	final public String POSTGRESQL = "PostgreSQL";
	final public String MYSQL = "MySQL";
	final public String HSQLDB = "HSQL Database Engine";
	final public String DERBY = "Apache Derby";

	public DatabaseInfo(Class driverClass) throws InstantiationException, IllegalAccessException {
		driver = (Driver) driverClass.newInstance();
	}

	public void setURI(String uri) {
		this.uri = uri;
	}

	public void setCatalog(String catalog) {
		this.catalog = catalog;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public void setSchema(String schema) {
		this.schema = schema;
	}

	public void setUser(String user) {
		this.user = user;
	}

	public void enableView(boolean flag) {
		enableView = flag;
	}
	
	public TableModel getTableInfo(String tableName, IDialect dialect) throws SQLException {
		TableModel table = new TableModel();
		table.setTableName(tableName);
		Connection con = null;
		ArrayList list = new ArrayList();
		
		try {
			con = connect();
			DatabaseMetaData meta = con.getMetaData();
			ResultSet columns = meta.getColumns(catalog,schema,tableName,"%");
			while(columns.next()){
				ColumnModel column = new ColumnModel();
				column.setColumnName(columns.getString("COLUMN_NAME"));
				column.setColumnType(dialect.getColumnType(columns.getInt("DATA_TYPE")));
				column.setSize(columns.getInt("COLUMN_SIZE"));
				column.setNotNull(columns.getString("IS_NULLABLE").equals("NO"));
				list.add(column);
			}
			columns.close();
			
			ResultSet keys = meta.getPrimaryKeys(catalog, schema, tableName);
			while(keys.next()){
				String columnName = keys.getString("COLUMN_NAME");
				for(int i=0;i<list.size();i++){
					ColumnModel column = (ColumnModel)list.get(i);
					if(column.getColumnName().equals(columnName)){
						column.setPrimaryKey(true);
					}
				}
			}
			keys.close();
			
			table.setColumns((ColumnModel[])list.toArray(new ColumnModel[list.size()]));
			
		} finally {
			if(con != null){
				con.close();
			}
		}		
		
		return table;
	}
	
	public void setForeignKeys(RootModel root) throws SQLException {
		Connection con = null;
		try {
			con = connect();
			DatabaseMetaData meta = con.getMetaData();
			for(int i=0;i<root.getChildren().size();i++){
				TableModel table = (TableModel)root.getChildren().get(i);
				ResultSet rs = meta.getImportedKeys(catalog, schema, table.getTableName());
				HashMap map = new HashMap();
				while(rs.next()){
					String pkTable  = rs.getString("PKTABLE_NAME");
					String pkColumn = rs.getString("PKCOLUMN_NAME");
					String fkTable  = rs.getString("FKTABLE_NAME");
					String fkColumn = rs.getString("FKCOLUMN_NAME");
					String keyName  = rs.getString("FK_NAME");
					
					if(root.getTable(pkTable)!=null && root.getTable(fkTable)!=null){
						if(map.get(keyName)==null){
							HashMap entry = new HashMap();
							entry.put("fkTable", fkTable);
							entry.put("pkTable", pkTable);
							entry.put("mappings", new ArrayList());
							map.put(keyName, entry);
						}
						List mappings = (List)((Map)map.get(keyName)).get("mappings");
						ForeignKeyMapping mapping = new ForeignKeyMapping();
						mapping.setRefer(root.getTable(fkTable).getColumn(fkColumn));
						mapping.setTarget(root.getTable(pkTable).getColumn(pkColumn));
						mappings.add(mapping);
					}
				}
				rs.close();
				
				Iterator ite = map.entrySet().iterator();
				while(ite.hasNext()){
					Map.Entry entry = (Map.Entry)ite.next();
					Map entryMap = (Map)entry.getValue();
					List mappings = (List)entryMap.get("mappings");
					
					ForeignKeyModel fkeyModel = new ForeignKeyModel();
					fkeyModel.setForeignKeyName((String)entry.getKey());
					fkeyModel.setMapping((ForeignKeyMapping[])mappings.toArray(new ForeignKeyMapping[mappings.size()]));
					
					fkeyModel.setSource(root.getTable((String)entryMap.get("fkTable")));
					fkeyModel.setTarget(root.getTable((String)entryMap.get("pkTable")));
					fkeyModel.attachSource();
					fkeyModel.attachTarget();
				}
			}
		} finally {
			if(con != null){
				con.close();
			}
		}		
	}
	
	private Connection connect() throws SQLException {
		Properties p = new Properties();
		p.setProperty("user", user);
		p.setProperty("password", password);
		return driver.connect(uri, p);
	}
	
	public ArrayList loadTables() throws SQLException {
		ArrayList list = new ArrayList();
		Connection con = null;
		
		try {
			con = connect();
			DatabaseMetaData meta = con.getMetaData();
			productName = meta.getDatabaseProductName();
			if (isMSSQL()) {
				if (catalog.length() == 0) {
					catalog = "%";
				}
			}

			catalog = (catalog.length() == 0) ? null : catalog;
			schema  = (schema.length()  == 0) ? null : schema;
			ResultSet tables = meta.getTables(catalog, schema, "%",
					isOracle() ? new String[] { "TABLE", "VIEW", "SYNONYM" } : null);
			
			while (tables.next()) {
				String t = tables.getString("TABLE_TYPE");
				if ("TABLE".equals(t) || ("VIEW".equals(t) && enableView) || (isOracle() && "SYNONYM".equals(t))) {
					list.add(tables.getString("table_name"));
				}
			}

			tables.close();

			if (driver.getClass().getName().equals("org.hsqldb.jdbcDriver") && uri.indexOf("jdbc:hsqldb:hsql://")!=0) {
				Statement stmt = null;
				try {
					stmt = con.createStatement();
//					System.out.println("SHUTDOWN");
//					System.out.println(uri);
					stmt.executeUpdate("SHUTDOWN;");
				} finally {
					if(stmt!=null)
						stmt.close();
				}
			}
		} finally {
			if (con != null){
				con.close();
			}
		}
		return list;

	}	
	public String getProductName() {
		return productName;
	}

	public boolean isPostgreSQL() {
		return POSTGRESQL.equals(productName);
	}

	public boolean isMySQL() {
		return MYSQL.equals(productName);
	}

	public boolean isHSQLDB() {
		return HSQLDB.equals(productName);
	}

	public boolean isDerby() {
		return DERBY.equals(productName);
	}

	public boolean isMSSQL() {
		if (productName.toLowerCase().indexOf("microsoft") != -1) {
			return true;
		} else {
			return false;
		}
	}

	public boolean isOracle() {
		if (productName.toLowerCase().indexOf("oracl") != -1) {
			return true;
		} else {
			return false;
		}
	}

	public void writeResultColumns(ResultSet cols) throws SQLException {
		ResultSetMetaData m = cols.getMetaData();
		int c = m.getColumnCount();

		while (cols.next()) {
			for (int k = 1; k < (c + 1); k++) {
				System.out.println(m.getColumnName(k) + ":" + cols.getString(k));
			}
		}
	}
}
