package net.java.amateras.db.visual.editor;

import java.util.ArrayList;
import java.util.EventObject;
import java.util.List;

import net.java.amateras.db.DBPlugin;
import net.java.amateras.db.view.dialect.DialectProvider;
import net.java.amateras.db.visual.action.AbstractDBAction;
import net.java.amateras.db.visual.action.GenerateAction;
import net.java.amateras.db.visual.editpart.DBEditPartFactory;
import net.java.amateras.db.visual.editpart.IDoubleClickSupport;
import net.java.amateras.db.visual.generate.GeneratorProvider;
import net.java.amateras.db.visual.generate.IGenerater;
import net.java.amateras.db.visual.model.ForeignKeyModel;
import net.java.amateras.db.visual.model.RootModel;
import net.java.amateras.db.visual.model.TableModel;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.gef.DefaultEditDomain;
import org.eclipse.gef.GraphicalViewer;
import org.eclipse.gef.editparts.ScalableRootEditPart;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.gef.palette.ConnectionCreationToolEntry;
import org.eclipse.gef.palette.CreationToolEntry;
import org.eclipse.gef.palette.MarqueeToolEntry;
import org.eclipse.gef.palette.PaletteDrawer;
import org.eclipse.gef.palette.PaletteEntry;
import org.eclipse.gef.palette.PaletteGroup;
import org.eclipse.gef.palette.PaletteRoot;
import org.eclipse.gef.palette.SelectionToolEntry;
import org.eclipse.gef.requests.SimpleFactory;
import org.eclipse.gef.ui.actions.DeleteAction;
import org.eclipse.gef.ui.actions.DeleteRetargetAction;
import org.eclipse.gef.ui.actions.GEFActionConstants;
import org.eclipse.gef.ui.actions.RedoRetargetAction;
import org.eclipse.gef.ui.actions.UndoRetargetAction;
import org.eclipse.gef.ui.actions.ZoomInAction;
import org.eclipse.gef.ui.actions.ZoomOutAction;
import org.eclipse.gef.ui.parts.GraphicalEditorWithPalette;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.actions.ActionFactory;

public class VisualDBEditor extends GraphicalEditorWithPalette {
	
	private boolean savePreviouslyNeeded = false;
	private List dbActions = new ArrayList();
	
	public VisualDBEditor(){
		super();
		setEditDomain(new DefaultEditDomain(this));
		getActionRegistry().registerAction(new UndoRetargetAction());
		getActionRegistry().registerAction(new RedoRetargetAction());
		getActionRegistry().registerAction(new DeleteRetargetAction());
	}
	
	protected PaletteRoot getPaletteRoot() {
		PaletteRoot root = new PaletteRoot();
		
		PaletteGroup tools = new PaletteGroup(DBPlugin.getDefault().getResourceString("palette.tools"));
		tools.add(new SelectionToolEntry());
		tools.add(new MarqueeToolEntry());
		
		PaletteDrawer drawer = new PaletteDrawer(DBPlugin.getDefault().getResourceString("palette.db"));
		drawer.add(createEntityEntry(
				DBPlugin.getDefault().getResourceString("palette.db.table"),TableModel.class,"icons/table.gif"));
		drawer.add(createConnectionEntry(
				DBPlugin.getDefault().getResourceString("palette.db.reference"),ForeignKeyModel.class,"icons/reference.gif"));
		
		root.add(tools);
		root.add(drawer);
		
		return root;
	}
	
	protected void setInput(IEditorInput input) {
		super.setInput(input);
		setPartName(input.getName());
	}

	protected void initializeGraphicalViewer() {
		GraphicalViewer viewer = getGraphicalViewer();
		
		ScalableRootEditPart rootEditPart = new ScalableRootEditPart();
		viewer.setEditPartFactory(new DBEditPartFactory());
		viewer.setRootEditPart(rootEditPart);
		
	    // ZoomManager
	    ZoomManager manager = rootEditPart.getZoomManager();
	    
	    // zoom level
	    double[] zoomLevels = new double[] {
	      0.25,0.5,0.75,1.0,1.5,2.0,2.5,3.0,4.0,5.0,10.0,20.0
	    };
	    manager.setZoomLevels(zoomLevels);
	    
	    // zoom level contribution
	    ArrayList zoomContributions = new ArrayList();
	    zoomContributions.add(ZoomManager.FIT_ALL);
	    zoomContributions.add(ZoomManager.FIT_HEIGHT);
	    zoomContributions.add(ZoomManager.FIT_WIDTH);
	    manager.setZoomLevelContributions(zoomContributions);
	    
	    getActionRegistry().registerAction(new ZoomInAction(manager));
	    getActionRegistry().registerAction(new ZoomOutAction(manager));
	    
		IFile file = ((IFileEditorInput)getEditorInput()).getFile();
		RootModel root = null;
		// desirialize
		try {
			root = VisualDBSerializer.deserialize(file.getContents());
		} catch(Exception ex){
			DBPlugin.logException(ex);
			root = new RootModel();
			root.setDialectName(DialectProvider.HSQLDB);
		}
		viewer.setContents(root);
		
		final DeleteAction deleteAction = new DeleteAction((IWorkbenchPart) this);
		deleteAction.setSelectionProvider(getGraphicalViewer());
		getActionRegistry().registerAction(deleteAction);
		getGraphicalViewer().addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				deleteAction.update();
			}
		});
		
//		// actions
//		ImportFromJDBCAction importAction = new ImportFromJDBCAction(viewer);
//		dbActions.add(importAction);
		
		// creates context menu
		MenuManager menuMgr = new MenuManager();
		menuMgr.add(getActionRegistry().getAction(ActionFactory.UNDO.getId()));
		menuMgr.add(getActionRegistry().getAction(ActionFactory.REDO.getId()));
		menuMgr.add(getActionRegistry().getAction(ActionFactory.DELETE.getId()));
		menuMgr.add(new Separator());
		menuMgr.add(getActionRegistry().getAction(GEFActionConstants.ZOOM_IN));
		menuMgr.add(getActionRegistry().getAction(GEFActionConstants.ZOOM_OUT));
		menuMgr.add(new Separator());
		
//		menuMgr.add(importAction);
		
		MenuManager generate = new MenuManager("Generate");
		IGenerater[] generaters = GeneratorProvider.getGeneraters();
		for(int i=0;i<generaters.length;i++){
			generate.add(new GenerateAction(generaters[i], viewer, this));
		}
		menuMgr.add(generate);
		
		viewer.setContextMenu(menuMgr);
		
		viewer.getControl().addMouseListener(new MouseAdapter(){
			public void mouseDoubleClick(MouseEvent e){
				IStructuredSelection selection = (IStructuredSelection)getGraphicalViewer().getSelection();
				Object obj = selection.getFirstElement();
				if(obj!=null && obj instanceof IDoubleClickSupport){
					((IDoubleClickSupport)obj).doubleClicked();
				}
			}
		});
	}

	public void doSave(IProgressMonitor monitor) {
		try {
			IFile file = ((IFileEditorInput)getEditorInput()).getFile();
			file.setContents(VisualDBSerializer.serialize((RootModel)getGraphicalViewer().getContents().getModel()),
					true, true, monitor);
		} catch(Exception ex){
			DBPlugin.logException(ex);
		}
		getCommandStack().markSaveLocation();
	}

	public void doSaveAs() {
		doSave(new NullProgressMonitor());
	}

	public boolean isSaveAsAllowed() {
		return true;
	}
	
	public void commandStackChanged(EventObject event) {
		if (isDirty()) {
			if (!savePreviouslyNeeded()) {
				setSavePreviouslyNeeded(true);
				firePropertyChange(IEditorPart.PROP_DIRTY);
			}
		} else {
			setSavePreviouslyNeeded(false);
			firePropertyChange(IEditorPart.PROP_DIRTY);
		}
		super.commandStackChanged(event);
	}

	private void setSavePreviouslyNeeded(boolean value) {
		this.savePreviouslyNeeded = value;
	}
	
	private boolean savePreviouslyNeeded() {
		return this.savePreviouslyNeeded;
	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection selection) {
		super.selectionChanged(part,selection);
		for(int i=0;i<dbActions.size();i++){
			AbstractDBAction action = (AbstractDBAction)dbActions.get(i);
			action.update((IStructuredSelection)selection);
		}
	}

	/**
	 * Creates <code>PaletteEntry</code> for the connection.
	 * 
	 * @param itemName the display name
	 * @param clazz the model class
	 * @param icon the icon path
	 * @return created <code>PaletteEntry</code>
	 */
	private PaletteEntry createConnectionEntry(String itemName,Class clazz,String icon){
		ImageDescriptor image = null;
		if(icon!=null){
			image = DBPlugin.getImageDescriptor(icon);
		}
		ConnectionCreationToolEntry entry = new ConnectionCreationToolEntry(
				itemName, itemName, new SimpleFactory(clazz), image, image);
		return entry;
	}
	
	/**
	 * Creates <code>PaletteEntry</code> for the entity.
	 * 
	 * @param itemName the display name
	 * @param clazz the model class
	 * @param icon the icon path
	 * @return created <code>PaletteEntry</code>
	 */
	private PaletteEntry createEntityEntry(String itemName,Class clazz,String icon){
		ImageDescriptor image = null;
		if(icon!=null){
			image = DBPlugin.getImageDescriptor(icon);
		}
		CreationToolEntry entry = new CreationToolEntry(
				itemName, itemName, new SimpleFactory(clazz), image, image);
		
		return entry;
	}

}
