package tk.eclipse.plugin.struts.wizards;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;

import tk.eclipse.plugin.htmleditor.HTMLProjectParams;
import tk.eclipse.plugin.htmleditor.HTMLUtil;
import tk.eclipse.plugin.htmleditor.IOUtil;
import tk.eclipse.plugin.struts.StrutsConfigModel2XML;
import tk.eclipse.plugin.struts.StrutsPlugin;
import tk.eclipse.plugin.struts.Util;
import tk.eclipse.plugin.struts.editors.models.ControllerModel;
import tk.eclipse.plugin.struts.editors.models.MessageResourcesModel;
import tk.eclipse.plugin.struts.editors.models.PluginModel;
import tk.eclipse.plugin.struts.editors.models.RootModel;
import tk.eclipse.plugin.struts.properties.Properties;

/**
 * <p>
 *   A wizard to add Struts libraries and the nature to the project.
 * </p>
 * <ul>
 *   <li>Copy Struts JAR files to WEB-INF/lib, and add them to the project classpath</li>
 *   <li>Copy Struts TLD files to WEB-INF</li>
 *   <li>Generate default struts-config.xml template to WEB-INF</li>
 *   <li>Generate web.xml which was configured as Struts application to WEB-INF</li>
 * </ul>
 * <p>
 *   If WEB-INF and WEB-INF/lib don't exist, they are created.
 *   And if same files already exist, they are overwrited.
 *   Of course, if same JARs were already added to the classpath, 
 *   they aren't added to the classpath.
 * </p>
 * <ul>
 *   <li>TODO Cancel processing</li>
 * </ul>
 * @author Naoki Takezoe
 */
public class StrutsWizard extends Wizard implements INewWizard {
	
	private StrutsWizardPage page1;
	private StrutsWizardPage2 page2;
	private IStructuredSelection selection;
	private ResourceBundle resource = StrutsPlugin.getDefault().getResourceBundle();
	
	/** JARs of Struts */
	public static final String[] JARS = {
			"/struts-1.2/commons-beanutils.jar",
            "/struts-1.2/commons-collections.jar",
			"/struts-1.2/commons-digester.jar",
			"/struts-1.2/commons-fileupload.jar",
//			"/struts-1.2/commons-lang.jar", removed in Struts 1.2
			"/struts-1.2/commons-logging.jar",
			"/struts-1.2/commons-validator.jar",
			"/struts-1.2/jakarta-oro.jar",
//			"/struts-1.2/struts-legacy.jar", removed in Struts 1.2
			"/struts-1.2/struts.jar"};
	
	/** TLDs of Struts taglibs */
	public static final String[] TLDS = {
			"/struts-1.2/struts-bean.tld",
            "/struts-1.2/struts-html.tld",
			"/struts-1.2/struts-logic.tld",
			"/struts-1.2/struts-nested.tld",
//			"/struts-1.2/struts-template.tld", removed in Struts 1.2
			"/struts-1.2/struts-tiles.tld"
	};
	
	/** MessageResources.properties */
	public static final String MESSAGE = "/struts-1.2/MessageResources.properties";
	
	/** Configuration files for validator */
	public static final String[] VALIDATOR_FILES = {
			"/struts-1.2/validation.xml",
            "/struts-1.2/validator-rules.xml",
	};
	
	/** Configuration files for tiles */
	public static final String TILES = "/struts-1.2/tiles-defs.xml";
	
	public StrutsWizard(){
		super();
		setNeedsProgressMonitor(true);
		setWindowTitle(resource.getString("wizard.strutsSupport.title"));
	}
	
	public boolean performFinish() {
		
		final String projectName = page1.getWebAppRoot();
		final boolean addJARs = page1.getAddStrutsLibraries();
		final boolean addTLDs = page1.getAddTLDFiles();
		final boolean createStrutsConfigXML = page1.getCreateStrutsConfigXML();
		final boolean createWebXML = page1.getCreateWebXML();
		final boolean createMessage = page1.getCreateMessageResources();
		final boolean useValidator = page2.getUseValidator();
		final boolean useTiles = page2.getUseTiles();
		final String mapping = page1.getMapping();
		
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException {
				try {
					doFinish(projectName,
							addJARs,
							addTLDs,
							createStrutsConfigXML,
							createWebXML,
							createMessage,
							useValidator,
							useTiles,
							mapping,
							monitor);
				} catch (Exception e) {
					throw new InvocationTargetException(e);
				} finally {
					monitor.done();
				}
			}
		};
		try {
			getContainer().run(true, false, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			Throwable realException = e.getTargetException();
			Util.openErrorDialog(realException);
			return false;
		}
		return true;
	}
	
	/**
	 * This is a main processing of this wizard.
	 * 
	 * @param projectName           Path from project root.
	 * @param addJARs               copy JAR files or not
	 * @param addTLDs               copy TLD files or not
	 * @param createStrutsConfigXML generate struts-config.xml or not
	 * @param createWebXML          generate web.xml or not
	 * @param createMessage         generate MessageResources.properties or not
	 * @param useValidator          use Validator or not
	 * @param useTiles              use Tiles or not
	 * @param mapping               servlet-mapping
	 * @param monitor               IProgressMonitor
	 * @throws CoreException
	 */
	private void doFinish(String path,
			boolean addJARs,
			boolean addTLDs,
			boolean createStrutsConfigXML,
			boolean createWebXML,
			boolean createMessage,
			boolean useValidator,
			boolean useTiles,
			String mapping,
			IProgressMonitor monitor) throws Exception {
		
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource resource = (IResource)root.findMember(new Path(path));
		if (!resource.exists() || !(resource instanceof IContainer)) {
			Util.throwCoreException(Util.createMessage(
					this.resource.getString("error.notexists"),
					new String[]{path}));
		}
		
		// add nature
		addNature(resource.getProject(),path);
		
		// calculate number of tasks
		int totalTask = 0;
		if(addJARs){
			totalTask = totalTask + JARS.length;
		}
		if(addTLDs){
			totalTask = totalTask + TLDS.length;
		}
		if(createStrutsConfigXML){
			totalTask = totalTask + 1;
		}
		if(createWebXML){
			totalTask = totalTask + 1;
		}
		if(createMessage){
			totalTask = totalTask + 1;
		}
		if(useValidator){
			totalTask = totalTask + VALIDATOR_FILES.length;;
		}
		if(useTiles){
			totalTask = totalTask + 1;
		}
		
		// Is it OK?
		monitor.beginTask(StrutsPlugin.getResourceString("wizard.strutsSupport.title"),totalTask);
		
		IContainer container = (IContainer) resource;
		// Creates WEB-INF directory if it doesn't exist
		File webinf = container.getFolder(new Path("/WEB-INF")).getLocation().makeAbsolute().toFile();
		if(!webinf.exists()){
			webinf.mkdir();
		}
		// Creates WEB-INF/lib directory if it doesn't exist
		File libdir = container.getFolder(new Path("/WEB-INF/lib")).getLocation().makeAbsolute().toFile();
		if(!libdir.exists()){
			libdir.mkdir();
		}
		// copy JARs to /WEB=INF and add them to the classpath
		if(addJARs){
			// At first, copy JAR files
			for(int i=0;i<JARS.length;i++){
				copyFile(monitor,container,JARS[i],"/WEB-INF/lib");
				monitor.worked(1);
			}
			// refresh
			container.refreshLocal(IResource.DEPTH_INFINITE,monitor);
			// add JARs to the classpath
			if(!HTMLUtil.isDynamicWebProject(container.getProject())){
				try {
					IJavaProject project = JavaCore.create(container.getProject());
					
					IClasspathEntry[] rawClasspath = project.getRawClasspath();
					ArrayList list = new ArrayList();
					list.addAll(Arrays.asList(rawClasspath));
					for(int i=0;i<JARS.length;i++){
						String fileName = new File(JARS[i]).getName();
						IClasspathEntry classPath = JavaCore.newLibraryEntry(new Path(path+"/WEB-INF/lib/"+fileName),null,null);
						if(!list.contains(classPath)){
							list.add(classPath);
						}
					}
					IClasspathEntry[] newClasspath = (IClasspathEntry[])list.toArray(new IClasspathEntry[list.size()]);
					project.setRawClasspath(newClasspath,monitor);
				} catch(Exception ex){
					// TODO In the WTP J2EEProject, WEB-INF/lib/*.jar are added to the classpath automatically. So exception is caused here.
				}
			}
		}
		// copy TLDs to /WEB-INF
		if(addTLDs){
			for(int i=0;i<TLDS.length;i++){
				copyFile(monitor,container,TLDS[i],"/WEB-INF");
				monitor.worked(1);
			}
		}
		// generate struts-config.xml to /WEB-INF
		if(createStrutsConfigXML){
//			monitor.beginTask(
//					StrutsPlugin.getResourceString("wizard.progress.createStrutsConfigXML"),1);
			try {
				IFile file = container.getFile(new Path("/WEB-INF/struts-config.xml"));
				InputStream stream = createStrutsConfigStream(createMessage,useValidator,useTiles);
				if (file.exists()) {
					file.setContents(stream, true, true, monitor);
				} else {
					file.create(stream, true, monitor);
				}
				stream.close();
			} catch (IOException e) {
			}
			monitor.worked(1);
		}
		// generate web.xml to /WEB-INF
		if(createWebXML){
//			monitor.beginTask(
//					StrutsPlugin.getResourceString("wizard.progress.createWebXML"),1);
			URL url = StrutsPlugin.getDefault().getBundle().getEntry("/struts-1.2/web.xml");
			try {
				String source = new String(IOUtil.readStream(url.openStream()));
				source = source.replaceAll("<url-pattern>\\*\\.do</url-pattern>", "<url-pattern>" + mapping + "</url-pattern>");
				IFile file = container.getFile(new Path("/WEB-INF/web.xml"));
				InputStream stream = new ByteArrayInputStream(source.getBytes());
				if (file.exists()) {
					file.setContents(stream, true, true, monitor);
				} else {
					file.create(stream, true, monitor);
				}
				stream.close();
				
			} catch(Exception ex){
			}
			monitor.worked(1);
		}
		
		// copy configuration files of Validator
		if(useValidator){
			for(int i=0;i<VALIDATOR_FILES.length;i++){
				copyFile(monitor,container,VALIDATOR_FILES[i],"/WEB-INF");
				monitor.worked(1);
			}
		}
		
		// copy configuration files of Tiles
		if(useTiles){
			copyFile(monitor,container,TILES,"/WEB-INF");
			monitor.worked(1);
		}
		
		// copy MessageResources.properties onto the source directory found first
		if(createMessage){
//			monitor.beginTask(
//					StrutsPlugin.getResourceString("wizard.progress.createMessage"),1);
			IJavaProject project = JavaCore.create(container.getProject());
			IPackageFragmentRoot[] roots = project.getPackageFragmentRoots();
			for(int i=0;i<roots.length;i++){
				IResource classpath = roots[i].getResource();
				if(classpath!=null && (classpath instanceof IFolder || classpath instanceof IProject)){
					copyFile(monitor,project.getProject(),MESSAGE,
							classpath.getProjectRelativePath().toString());
					break;
				}
			}
			monitor.worked(1);
		}
		
		container.refreshLocal(IResource.DEPTH_INFINITE,monitor);
	}
	
	/** Adds StrutsProjectNature to the project */
	private void addNature(IProject project,String path) throws Exception {
		// save configuration
		HTMLProjectParams params = new HTMLProjectParams();
		params.setRoot("/" + new Path(path).removeFirstSegments(1).toString());
		params.save(project);
//		StrutsProjectParams params = new StrutsProjectParams();
//		params.setWebAppRoot("/" + new Path(path).removeFirstSegments(1).toString());
//		params.save(JavaCore.create(project));
		
		IProjectDescription description = project.getDescription();
		String[] natures = description.getNatureIds();
		// skip if already added
		for(int i=0;i<natures.length;i++){
			if(natures[i].equals(StrutsPlugin.STRUTS_PROJECT_NATURE)){
				return;
			}
		}
		String[] newNatures = new String[natures.length + 1];
		System.arraycopy(natures, 0, newNatures, 0, natures.length);
		newNatures[natures.length] = StrutsPlugin.STRUTS_PROJECT_NATURE;
		description.setNatureIds(newNatures);
		project.setDescription(description, null);
	}
	
	/** Copy a file.  */
	private synchronized void copyFile(IProgressMonitor monitor,IContainer container,
			                           String entry,String target){
		try {
			URL url = StrutsPlugin.getDefault().getBundle().getEntry(entry);
			String fileName = new File(entry).getName();
			
			File file = container.getFile(new Path(target+"/"+fileName)).getLocation().makeAbsolute().toFile();
			InputStream in = url.openStream();
			OutputStream out = new FileOutputStream(file);
			byte[] buf = new byte[1024 * 8];
			int length = 0;
			while((length=in.read(buf))!=-1){
				out.write(buf,0,length);
			}
			out.close();
			in.close();
			
		} catch(Exception ex){
			System.out.println(entry);
			ex.printStackTrace();
			Util.openErrorDialog(ex);
		}
	}
	
	/** Generates default struts-config.xml. */
	private InputStream createStrutsConfigStream(boolean createMessage,
			boolean useValidator,boolean useTiles){
		
		StringBuffer sb = new StringBuffer();
		
		RootModel root = new RootModel();
		root.setCharset("UTF-8");
		root.setDtdName("struts-config");
		root.setDtdPublicId("-//Apache Software Foundation//DTD Struts Configuration 1.2//EN");
		root.setDtdSystemId("http://struts.apache.org/dtds/struts-config_1_2.dtd");
		
		//message-resources
		if(createMessage){
			MessageResourcesModel message = new MessageResourcesModel();
			message.setParameter("MessageResources");
			root.addChild(message);
		}
		
		// tiles
		if(useTiles){
			PluginModel tilesPlugin = new PluginModel();
			tilesPlugin.setClassName("org.apache.struts.tiles.TilesPlugin");
			Properties tilesProps = new Properties();
			tilesProps.addProperty("definitions-config","/WEB-INF/tiles-defs.xml",null);
			tilesProps.addProperty("moduleAware","true",null);
			tilesPlugin.setProperties(tilesProps);
			root.addChild(tilesPlugin);
			
			ControllerModel controller = new ControllerModel();
			controller.setProcessorClass("org.apache.struts.tiles.TilesRequestProcessor");
			root.addChild(controller);
		}
		
		// validator
		if(useValidator){
			PluginModel validatorPlugin = new PluginModel();
			validatorPlugin.setClassName("org.apache.struts.validator.ValidatorPlugIn");
			Properties validatorProps = new Properties();
			validatorProps.addProperty("pathnames","/WEB-INF/validator-rules.xml,/WEB-INF/validation.xml",null);
			validatorPlugin.setProperties(validatorProps);
			root.addChild(validatorPlugin);
		}
		
		sb.append(StrutsConfigModel2XML.createStrutsConfig(root, ".do"));
		return new ByteArrayInputStream(sb.toString().getBytes());
	}
	
	/** Initilizes this wizard. */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		this.selection = selection;
	}
	
	/** Adds WizardPages to this wizard. */
	public void addPages() {
		page1 = new StrutsWizardPage(selection);
		page2 = new StrutsWizardPage2(selection);
		addPage(page1);
		addPage(page2);
	}
}
