package net.sf.amateras.air.builder;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.internal.events.ILifecycleListener;
import org.eclipse.core.internal.events.LifecycleEvent;
import org.eclipse.core.internal.resources.Workspace;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.ui.texteditor.MarkerUtilities;

/**
 * 
 * 
 * @author hideko ogawa
 */
@SuppressWarnings("restriction")
public abstract class AbstractAIRBuilder extends IncrementalProjectBuilder {

	public static Map<IProject, FcshShell> fcshShells = new HashMap<IProject, FcshShell>();

	private Pattern messagePattern;
	private Pattern errorPattern;
	private Pattern warnPattern;
	private Pattern failPattern;
	private Pattern errorPattern2;
	private Pattern warnPattern2;
	private Pattern failPattern2;
	private boolean isAddInfoMarker;

	private ILifecycleListener lifecycleListener;

	private int workd;

	public AbstractAIRBuilder() {
		IPreferenceStore store = AIRPlugin.getDefault().getPreferenceStore();
		getCompileErrorPattern();
		store.addPropertyChangeListener(new IPropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				getCompileErrorPattern();
			}
		});
	}

	/**
	 * build
	 * @param kind
	 * @param args
	 * @param moniter
	 */
	@Override
	protected IProject[] build(int kind, @SuppressWarnings("unchecked")
	Map args, IProgressMonitor monitor) throws CoreException {
		workd = 0;
		monitor.beginTask("compile", 3);
		if (lifecycleListener == null) {
			addLifecycleListener();
		}
		if (kind == FULL_BUILD) {
			fullBuild(monitor);

		} else {
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				fullBuild(monitor);

			} else {
				incrementalBuild(delta, monitor);
			}
		}
		return null;
	}

	/**
	 * full build
	 * @param monitor
	 * @throws CoreException
	 */
	public abstract void fullBuild(final IProgressMonitor monitor) throws CoreException;

	/**
	 * incremental Build.
	 * @param delta
	 * @param monitor
	 * @param mainFile
	 * @throws CoreException
	 */
	public abstract void incrementalBuild(IResourceDelta delta, final IProgressMonitor monitor) throws CoreException;

	/**
	 * get error pattern from PreferenceStore.
	 */
	private void getCompileErrorPattern() {
		IPreferenceStore store = AIRPlugin.getDefault().getPreferenceStore();
		messagePattern = getPattern(store.getString(AIRPlugin.PREF_COMPILE_MESSAGE_PATTERN));
		failPattern = getPattern(store.getString(AIRPlugin.PREF_COMPILE_FAIL_PATTERN));
		errorPattern = getPattern(store.getString(AIRPlugin.PREF_COMPILE_ERROR_PATTERN));
		warnPattern = getPattern(store.getString(AIRPlugin.PREF_COMPILE_WARNING_PATTERN));

		failPattern2 = getPattern(store.getString(AIRPlugin.PREF_COMPILE_FAIL_PATTERN_2));
		errorPattern2 = getPattern(store.getString(AIRPlugin.PREF_COMPILE_ERROR_PATTERN_2));
		warnPattern2 = getPattern(store.getString(AIRPlugin.PREF_COMPILE_WARNING_PATTERN_2));
		isAddInfoMarker = store.getBoolean(AIRPlugin.PREF_IS_ADD_MARKER_COMPILE_INFO);
	}

	private Pattern getPattern(String value) {
		if (value == null || value.length() == 0) {
			return null;
		}
		return Pattern.compile(value);
	}

	/**
	 * add Error and Warning marker.
	 * @param result
	 * @param project
	 * @param markerId
	 * @throws CoreException
	 */
	protected void addMarker(String result, IProject project, String markerId) throws CoreException {
		boolean isErrorPattern = false;
		isErrorPattern = addErrorMarker(errorPattern, result, project, markerId);

		if (!isErrorPattern) {
			isErrorPattern = addErrorMarker(errorPattern2, result, project, markerId);
		}

		if (!isErrorPattern) {
			isErrorPattern = addFailureMarker(failPattern, result, project, markerId);
		}

		if (!isErrorPattern) {
			isErrorPattern = addFailureMarker(failPattern2, result, project, markerId);
		}

		boolean isWarningPattern = addWarningMarker(warnPattern, result, project, markerId);

		if (!isWarningPattern) {
			addWarningMarker(warnPattern2, result, project, markerId);
		}

		if (isAddInfoMarker) {
			Map<String, Object> attributes = new HashMap<String, Object>();
			attributes.put(IMarker.PRIORITY, Integer.valueOf(IMarker.PRIORITY_LOW));
			attributes.put(IMarker.SEVERITY, Integer.valueOf(IMarker.SEVERITY_INFO));
			attributes.put(IMarker.LINE_NUMBER, Integer.valueOf(0));
			attributes.put(IMarker.MESSAGE, result);
			MarkerUtilities.createMarker(getProject(), attributes, markerId);
		}

	}

	private boolean addErrorMarker(Pattern pattern, String result, IProject project, String markerId)
			throws CoreException {
		if (pattern != null) {
			boolean isExists = false;
			Matcher matcher = pattern.matcher(result);
			while (matcher.find()) {
				if (matcher.groupCount() >= 3) {
					String fileName = new File(matcher.group(1)).getName();
					addMarker(project.getFile(fileName), matcher.group(1), matcher.group(2), matcher.group(3),
							IMarker.SEVERITY_ERROR, markerId);
					isExists = true;
				}
			}
			return isExists;
		}
		return false;
	}

	private boolean addFailureMarker(Pattern pattern, String result, IProject project, String markerId)
			throws CoreException {
		if (pattern != null) {
			Matcher matcher = pattern.matcher(result);
			boolean isExists = false;
			while (matcher.find()) {
				Map<String, Object> attributes = new HashMap<String, Object>();
				attributes.put(IMarker.PRIORITY, Integer.valueOf(IMarker.PRIORITY_HIGH));
				attributes.put(IMarker.SEVERITY, Integer.valueOf(IMarker.SEVERITY_ERROR));
				attributes.put(IMarker.LINE_NUMBER, Integer.valueOf(0));
				attributes.put(IMarker.MESSAGE, matcher.group(1));
				MarkerUtilities.createMarker(getProject(), attributes, markerId);
				isExists = true;
			}
			return isExists;
		}
		return false;
	}

	private boolean addWarningMarker(Pattern pattern, String result, IProject project, String markerId)
			throws CoreException {
		if (pattern != null) {
			Matcher matcher = pattern.matcher(result);
			boolean isExists = false;
			while (matcher.find()) {
				if (matcher.groupCount() >= 3) {
					String fileName = new File(matcher.group(1)).getName();
					addMarker(project.getFile(fileName), matcher.group(1), matcher.group(2), matcher.group(3),
							IMarker.SEVERITY_WARNING, markerId);
					isExists = true;
				}
			}
			return isExists;
		}
		return false;
	}

	/**
	 * create FcshSell class.
	 * @param project
	 * @param monitor
	 * @return
	 * @throws IOException
	 */
	protected FcshShell createFcshShell(IProject project, IProgressMonitor monitor) throws IOException {
		FcshShell fcsh = fcshShells.get(project);
		monitor.worked(workd++);
		if (fcsh == null || fcsh.getProcess() == null) {
			fcsh = new FcshShell();
			clearCompiler(project);
			fcshShells.put(project, fcsh);
			fcsh.startShell(new File(project.getLocation().toString()), monitor);
			monitor.worked(workd++);
		}
		fcsh.setWorkdIndex(workd);
		return fcsh;
	}

	protected abstract void clearCompiler(IProject project);

	/**
	 * add error marker.
	 * @param compileFile
	 * @param fileName
	 * @param line
	 * @param message
	 * @param level
	 * @param markerId
	 * @throws CoreException
	 */
	private void addMarker(IFile compileFile, String fileName, String line, String message, int level, String markerId)
			throws CoreException {

		fileName = fileName.substring(compileFile.getParent().getLocation().toString().length() + 1);
		IFile errorFile = compileFile.getParent().getFile(new Path(fileName));

		IMarker marker = errorFile.createMarker(markerId);
		Map<String, Object> map = new HashMap<String, Object>();
		map.put(IMarker.SEVERITY, new Integer(level));
		map.put(IMarker.MESSAGE, message);
		map.put(IMarker.LINE_NUMBER, new Integer(line));
		marker.setAttributes(map);
	}

	/**
	 * when project delete or close, shutdown compiler.
	 */
	@SuppressWarnings("restriction")
	private void addLifecycleListener() {
		if (getProject().getWorkspace() instanceof Workspace) {
			ILifecycleListener listener = new ILifecycleListener() {
				public void handleEvent(LifecycleEvent event) {
					if ((event.kind == LifecycleEvent.PRE_PROJECT_DELETE || event.kind == LifecycleEvent.PRE_PROJECT_CLOSE)
							&& event.resource == getProject()) {
						AirBuilderManager.shutdownCompiler(getProject());
					}
				}
			};
			((Workspace) getProject().getWorkspace()).addLifecycleListener(listener);
		}
	}

	/**
	 * get Compile id match pattern.
	 * @return
	 */
	protected Pattern getMessagePattern() {
		return messagePattern;
	}

}
