package net.sf.amateras.air.builder;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.amateras.air.AIRPlugin;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.ui.internal.ide.IDEWorkbenchMessages;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * 
 * @author hideko ogawa
 */
@SuppressWarnings("restriction")
public class AirBuilderManager extends AbstractAIRBuilder {
	public static final String BUILDER_ID = "net.sf.amateras.air.MXMLBuilder";
	public static final String MARKER_ID = "net.sf.amateras.air.MXMLProblemMarker";
	public static final String MARKER_ID_AS3 = "net.sf.amateras.air.AS3ProblemMarker";

	public static Map<IProject, MXMLCompiler> mxmlCompilers = new HashMap<IProject, MXMLCompiler>();
	public static Map<IProject, CompcCompiler> compcCompilers = new HashMap<IProject, CompcCompiler>();

	@Override
	public void fullBuild(final IProgressMonitor monitor) throws CoreException {
		ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(getProject()), AIRPlugin.PLUGIN_ID);
		List<CompileProperty> compiles = CompileProperty.load(store);
		getProject().deleteMarkers(MARKER_ID, true, IResource.DEPTH_INFINITE);
		getProject().deleteMarkers(MARKER_ID_AS3, true, IResource.DEPTH_INFINITE);
		compile(compiles, monitor);
	}

	@Override
	public void incrementalBuild(IResourceDelta delta, IProgressMonitor monitor) throws CoreException {

		List<CompileProperty> compiles = parseDelta(delta, getProject());
		compile(compiles, monitor);
	}

	/**
	 * compile directory.
	 * @param compiles
	 * @param monitor
	 * @throws CoreException
	 */
	private void compile(List<CompileProperty> compiles, IProgressMonitor monitor) throws CoreException {

		for (CompileProperty prop : compiles) {
			// get all ASfiles
			StringBuilder allAsFiles = new StringBuilder();
			if (prop.getCommandType() == CompileProperty.AS3_TYPE) {
				IResource folder = null;
				if (prop.getDirectory().length() == 0) {
					folder = getProject();
				} else {
					folder = getProject().getFolder(prop.getDirectory());
				}

				List<String> allASList = new ArrayList<String>();
				getAllActionScriptFile(allASList, folder);

				for (String fileName : allASList) {
					fileName = fileName.substring(0, fileName.length() - 3);
					fileName = fileName.replaceFirst(prop.getDirectory() + "/", "");
					allAsFiles.append(" ").append(fileName);
				}
			}

			//library
			ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(getProject()), AIRPlugin.PLUGIN_ID);
			String libraries = store.getString(AIRPlugin.PREF_FLEX_LIBRARIES);

			String command = prop.getCommand();
			command = command + " "
					+ getReplace(prop.getOption(), allAsFiles.toString(), libraries, prop.getDirectory());

			try {
				if (prop.getCommandType() == CompileProperty.MXML_TYPE) {
					getProject().deleteMarkers(MARKER_ID, true, IResource.DEPTH_INFINITE);
					compileMXML(command, getProject(), monitor);

				} else if (prop.getCommandType() == CompileProperty.AS3_TYPE) {
					getProject().deleteMarkers(MARKER_ID_AS3, true, IResource.DEPTH_INFINITE);
					compileAS3(command, getProject(), monitor);

				} else if (prop.getCommandType() == CompileProperty.FLEX3_TYPE) {
					getProject().deleteMarkers(MARKER_ID, true, IResource.DEPTH_INFINITE);
					compileMXML(command, getProject(), monitor);
				}

			} catch (IOException ex) {
				AIRPlugin.logException(ex);
				throw new CoreException(new Status(IStatus.ERROR, AIRPlugin.getDefault().getBundle().getSymbolicName(),
						0, ex.getMessage(), ex));
			}
		}
	}

	private void getAllActionScriptFile(final List<String> fileList, IResource folder) throws CoreException {
		folder.accept(new IResourceVisitor() {
			public boolean visit(IResource resource) {
				if (resource instanceof IFile && resource.getName().endsWith(".as")) {
					String path = resource.getProjectRelativePath().toString();
					fileList.add(path);
				}
				return true;
			}
		});
	}

	private List<CompileProperty> parseDelta(IResourceDelta delta, IProject project) throws CoreException {
		ScopedPreferenceStore store = new ScopedPreferenceStore(new ProjectScope(project), AIRPlugin.PLUGIN_ID);
		final List<CompileProperty> compiles = CompileProperty.load(store);
		final List<CompileProperty> targetCompiles = new ArrayList<CompileProperty>();

		delta.accept(new IResourceDeltaVisitor() {
			public boolean visit(IResourceDelta delta) {
				IResource resource = delta.getResource();

				if (resource != null && resource instanceof IFile) {
					if (delta.getKind() == IResourceDelta.ADDED || delta.getKind() == IResourceDelta.CHANGED
							|| delta.getKind() == IResourceDelta.COPIED_FROM) {

						String path = resource.getProjectRelativePath().toString();

						if (path.endsWith(".as") || path.endsWith(".mxml") || path.endsWith(".swc")) {

							//search myCompileProperty
							CompileProperty myProp = null;
							CompileProperty rootProp = null;
							for (CompileProperty prop : compiles) {
								String dir = prop.getDirectory().trim();
								if (dir.length() == 0) {
									rootProp = prop;
									continue;
								} else if (path.startsWith(prop.getDirectory().trim())) {
									targetCompiles.add(prop);
									System.out.println("add build to:" + prop.getDirectory());
									compiles.remove(myProp);
									rootProp = null;
									break;
								}
							}
							if (rootProp != null) {
								targetCompiles.add(rootProp);
								compiles.remove(rootProp);
							}
						}
					}
				}
				return true;
			}
		});
		return targetCompiles;

	}

	public void compileAS3(String command, IProject project, IProgressMonitor monitor) throws CoreException,
			IOException {
		int workd = 2;
		monitor.worked(workd++);

		FcshShell fcsh = createFcshShell(project, monitor);

		CompcCompiler compiler = compcCompilers.get(project);
		if (compiler == null) {
			compiler = new CompcCompiler(project, getMessagePattern());
			compcCompilers.put(project, compiler);
		} else {
			compiler.setMessagePattern(getMessagePattern());
		}

		monitor.worked(workd++);
		String result = compiler.requestCompile(command, fcsh, monitor);

		monitor.worked(workd++);
		addMarker(result, project, AirBuilderManager.MARKER_ID_AS3);
	}

	public void compileMXML(String command, IProject project, IProgressMonitor monitor) throws CoreException,
			IOException {
		int workd = 2;
		monitor.worked(workd++);

		FcshShell fcsh = createFcshShell(project, monitor);

		MXMLCompiler compiler = mxmlCompilers.get(project);
		if (compiler == null) {
			compiler = new MXMLCompiler(project, getMessagePattern());
			mxmlCompilers.put(project, compiler);
		} else {
			compiler.setMessagePattern(getMessagePattern());
		}

		monitor.worked(workd++);
		String result = compiler.requestCompile(command, fcsh, monitor);

		monitor.worked(workd++);
		addMarker(result, project, MARKER_ID);
	}

	@Override
	protected void clearCompiler(IProject project) {
		MXMLCompiler compiler = mxmlCompilers.remove(project);
		if (compiler != null) {
			compiler.dispose();
		}

		CompcCompiler comp2 = compcCompilers.remove(project);
		if (comp2 != null) {
			comp2.dispose();
		}
	}

	/**
	 * Shutdown project Compiler and FcshShell.
	 * @param project
	 */
	public static void shutdownCompiler(IProject project) {
		mxmlCompilers.remove(project);
		compcCompilers.remove(project);
		FcshShell fcsh = fcshShells.get(project);
		if (fcsh != null) {
			fcsh.stopShell();
			fcshShells.remove(project);
		}
	}

	public static void doClean(final IProject project) {
		WorkspaceJob cleanJob = new WorkspaceJob(IDEWorkbenchMessages.CleanDialog_taskName) {
			@Override
			public boolean belongsTo(Object family) {
				return ResourcesPlugin.FAMILY_MANUAL_BUILD.equals(family);
			}

			@Override
			public IStatus runInWorkspace(IProgressMonitor monitor) throws CoreException {
				project.build(IncrementalProjectBuilder.CLEAN_BUILD, new SubProgressMonitor(monitor, 1));
				return Status.OK_STATUS;
			}
		};
		cleanJob.setRule(ResourcesPlugin.getWorkspace().getRuleFactory().buildRule());
		cleanJob.setUser(true);
		cleanJob.schedule();

	}

	/**
	 * Shutdown all active Compilers and FcshShells.
	 */
	public static void shutdownCompilers() {
		mxmlCompilers.clear();
		compcCompilers.clear();
		Object[] fcshs = fcshShells.values().toArray();
		for (int i = 0; i < fcshs.length; i++) {
			((FcshShell) fcshs[i]).stopShell();
		}
		fcshShells.clear();
	}

	private String getReplace(String value, String asFiles, String libraryPath, String directory) {
		try {
			value = value.replaceAll("\\$\\{PROJECT_NAME\\}", getProject().getName());
			if (libraryPath.length() > 0) {
				libraryPath = libraryPath.replaceAll("\t", ",");
				value = value.replaceAll("\\$\\{LIBRARY_PATH\\}", libraryPath);
			} else {
				value = value.replaceAll("\\$\\{LIBRARY_PATH\\}", "");
				if (value.indexOf("-library-path+= ") >= 0) {
					value = value.replaceAll("-library-path\\+\\= ", "");
				}
				if (value.endsWith("-library-path+=")) {
					value = value.replaceAll("-library-path\\+\\=", "");
				}
			}
			if (directory.length() == 0) {
				directory = ".";
			}
			directory = directory + "/";
			value = value.replaceAll("\\$\\{AS_FILES\\}", asFiles);
			value = value.replaceAll("\\$\\{DIRECTORY\\}", directory);

			return value;
		} catch (Exception ex) {
			ex.printStackTrace();
			return value;
		}
	}
}
