package net.sf.amateras.air.debug;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IWatchpoint;

/**
 * A watchpoint.
 */
public class AirWatchpoint extends AirLineBreakPoint implements IWatchpoint {

	// 'read' or 'write' depending on what caused the last suspend for this watchpoint
	private String fLastSuspendType;

	// marker attributes
	public static final String ACCESS = "ACCESS";
	public static final String MODIFICATION = "MODIFICATION";
	public static final String FUNCTION_NAME = "FUNCTION_NAME";
	public static final String VAR_NAME = "VAR_NAME";

	public AirWatchpoint() {
	}

	public AirWatchpoint(final IResource resource, final int lineNumber, final String functionName,
			final String varName, final boolean access, final boolean modification) throws CoreException {
		IWorkspaceRunnable runnable = new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				IMarker marker = resource.createMarker(AirLineBreakPoint.WATCH_POINT);
				setMarker(marker);
				setEnabled(true);
				ensureMarker().setAttribute(IMarker.LINE_NUMBER, lineNumber);
				ensureMarker().setAttribute(IBreakpoint.ID, getModelIdentifier());
				setAccess(access);
				setModification(modification);
				setVariable(functionName, varName);
				marker.setAttribute(IMarker.MESSAGE, "Watchpoint: " + resource.getName() + " [line: " + lineNumber
						+ "]");
			}
		};
		run(getMarkerRule(resource), runnable);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#isAccess()
	 */
	public boolean isAccess() {
		return getMarker().getAttribute(ACCESS, true);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#setAccess(boolean)
	 */
	public void setAccess(boolean access) throws CoreException {
		setAttribute(ACCESS, access);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#isModification()
	 */
	public boolean isModification() {
		return getMarker().getAttribute(MODIFICATION, true);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#setModification(boolean)
	 */
	public void setModification(boolean modification) throws CoreException {
		setAttribute(MODIFICATION, modification);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#supportsAccess()
	 */
	public boolean supportsAccess() {
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.debug.core.model.IWatchpoint#supportsModification()
	 */
	public boolean supportsModification() {
		return true;
	}

	/**
	 * Sets the variable and function names the watchpoint is set on.
	 * 
	 * @param functionName function name
	 * @param variableName variable name
	 * @throws CoreException if an exception occurrs setting marker attribtues
	 */
	protected void setVariable(String functionName, String variableName) throws CoreException {
		setAttribute(VAR_NAME, variableName);
		setAttribute(FUNCTION_NAME, functionName);
	}

	/**
	 * Returns the name of the variable this watchpoint is set on.
	 * 
	 * @return the name of the variable this watchpoint is set on
	 */
	public String getVariableName() {
		return getMarker().getAttribute(VAR_NAME, (String) null);
	}

	/**
	 * Returns the name of the function the variable associted with this watchpoint is defined in.
	 * 
	 * @return the name of the function the variable associted with this watchpoint is defined in
	 */
	public String getFunctionName() {
		return getMarker().getAttribute(FUNCTION_NAME, (String) null);
	}

	/**
	 * Sets the type of event that causes the last suspend event.
	 * 
	 * @param description one of 'read' or 'write'
	 */
	public void setSuspendType(String description) {
		fLastSuspendType = description;
	}

	/**
	 * Returns the type of event that caused the last suspend.
	 * 
	 * @return 'read', 'write', or <code>null</code> if undefined
	 */
	public String getSuspendType() {
		return fLastSuspendType;
	}

	protected void createRequest(AirDebugTarget target) {
		int flag = 0;
		if (isAccess()) {
			flag = flag | 1;
		}
		if (isModification()) {
			flag = flag | 2;
		}
		//target.sendRequest("watch " + getFunctionName() + "::" + getVariableName() + " " + flag);
	}

	protected void clearRequest(AirDebugTarget target) {
		//target.sendRequest("watch " + getFunctionName() + "::" + getVariableName() + " " + 0);
	}

	public void handleEvent(String event) {
		if (event.startsWith("suspended watch")) {
			handleHit(event);
		}
	}

	/**
	 * Determines if this breakpoint was hit and notifies the thread.
	 * 
	 * @param event breakpoint event
	 */
	private void handleHit(String event) {
		String[] strings = event.split(" ");
		if (strings.length == 4) {
			String fv = strings[3];
			int j = fv.indexOf("::");
			if (j > 0) {
				String fcn = fv.substring(0, j);
				String var = fv.substring(j + 2);
				if (getVariableName().equals(var) && getFunctionName().equals(fcn)) {
					setSuspendType(strings[2]);
					//notifyThread();
				}
			}
		}
	}
}
