package net.sf.amateras.air.debug.thread;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import net.sf.amateras.air.debug.AirDebugElement;
import net.sf.amateras.air.debug.AirDebugTarget;
import net.sf.amateras.air.debug.debugger.command.InfoStackFrames;
import net.sf.amateras.air.debug.debugger.command.StepCommand;
import net.sf.amateras.air.debug.debugger.command.StepOverCommand;
import net.sf.amateras.air.debug.debugger.command.StepReturnCommand;
import net.sf.amateras.air.debug.event.IAirEventListener;
import net.sf.amateras.air.debug.matcher.FdbConsoleTextMatcher;
import net.sf.amateras.air.debug.matcher.StackframeMatchResult;

import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;
import org.eclipse.debug.core.model.IWatchExpressionResult;

/**
 * A Air thread. 
 */
public class AirThread extends AirDebugElement implements IThread, IAirEventListener {

	private IBreakpoint fBreakpoint;
	private boolean fStepping = false;
	private String fErrorEvent;

	private List<IVariable> variables = new ArrayList<IVariable>();

	private String stopSourceName;
	private int stopLineNo;

	private LinkedHashMap<String, AirStackFrame> otherStackFrame = new java.util.LinkedHashMap<String, AirStackFrame>();
	private AirStackFrame stackFrame;

	/**
	 * Constructs a new thread for the given target
	 * 
	 * @param target VM
	 */
	public AirThread(AirDebugTarget target) {
		super(target);
		//DebugOptions.DEBUG_COMMANDS = true;
		//addAirVariable("c");
	}

	public int getPriority() {
		return 0;
	}

	public String getName() {
		return "Air thread";
	}

	public IBreakpoint[] getBreakpoints() {
		if (fBreakpoint == null) {
			return new IBreakpoint[0];
		}
		return new IBreakpoint[] { fBreakpoint };
	}

	// *******************************
	// stack frames
	// *******************************

	private void createStackFrame() {
		addDebuggerCommand(new InfoStackFrames());
		if (stackFrame == null || !(stackFrame.getSourceName().equals(stopSourceName))) {
			stackFrame = otherStackFrame.get(stopSourceName);
			if (stackFrame == null) {
				if (stopSourceName != null) {
					stackFrame = new AirStackFrame(this, stopSourceName, stopLineNo);
				}
			} else {
				stackFrame.setLineNumber(stopLineNo);
				otherStackFrame.remove(stopSourceName);
			}
		} else {
			stackFrame.setLineNumber(stopLineNo);
		}
	}

	public IStackFrame[] getStackFrames() {
		IStackFrame[] newStack = new IStackFrame[otherStackFrame.size() + 1];
		newStack[0] = stackFrame;
		int index = 1;
		for (String key : otherStackFrame.keySet()) {
			AirStackFrame airStackFrame = otherStackFrame.get(key);
			newStack[index] = airStackFrame;
			index++;
		}
		return newStack;
	}

	public boolean hasStackFrames() {
		return isSuspended();
	}

	public IStackFrame getTopStackFrame() {
		return stackFrame;
	}

	// *******************************
	// suspend
	// *******************************
	public void suspendedBy(IBreakpoint breakpoint) {
		fBreakpoint = breakpoint;
		suspended(DebugEvent.BREAKPOINT);
	}

	public boolean canSuspend() {
		return getDebugTarget().canSuspend();
	}

	public boolean isSuspended() {
		return getAirDebugTarget().isSuspended();
	}

	public void suspend() {
		getAirDebugTarget().suspend();
	}

	private void suspended(int detail) {
		fireSuspendEvent(detail);
	}

	// *******************************
	// resume
	// *******************************
	public boolean canResume() {
		return getDebugTarget().canResume();
	}

	public void resume() throws DebugException {
		getDebugTarget().resume();
	}

	private void resumed(int detail) {
		setError(null);
		//		synchronized (variables) {
		//			variables.clear();
		//		}
		fireResumeEvent(detail);
	}

	// *******************************
	// step into
	// *******************************
	public boolean canStepInto() {
		return isSuspended();
	}

	public void stepInto() {
		addDebuggerCommand(new StepCommand());
		setStepping(true);
	}

	// *******************************
	// step over
	// *******************************
	public boolean canStepOver() {
		return isSuspended();
	}

	public void stepOver() {
		addDebuggerCommand(new StepOverCommand());
	}

	private void setStepping(boolean stepping) {
		fStepping = stepping;
	}

	// *******************************
	// step return
	// *******************************
	public boolean canStepReturn() {
		return false;
	}

	public void stepReturn() {
		addDebuggerCommand(new StepReturnCommand());
	}

	public boolean isStepping() {
		return fStepping;
	}

	// *******************************
	// teminate
	// *******************************
	public boolean canTerminate() {
		return getDebugTarget().canTerminate();
	}

	public boolean isTerminated() {
		return getDebugTarget().isTerminated();
	}

	public void terminate() throws DebugException {
		getDebugTarget().terminate();
	}

	// *******************************
	// error
	// *******************************
	private void setError(String event) {
		fErrorEvent = event;
	}

	public Object getError() {
		return fErrorEvent;
	}

	// **************************************
	// variables
	// **************************************

	protected void addAirVariable(String name) {
		AirVariable a = new AirVariable(getDebugTarget(), name);
		addVariable(a);
	}

	protected void addVariable(IVariable variable) {
		variables.add(variable);
	}

	protected IVariable[] getVariables() {
		return variables.toArray(new IVariable[variables.size()]);
	}

	// **************************************
	// AirEventListener
	// **************************************

	public void suspended(String event, String file, int lineNo, int debugType) {
		if (file != null) {
			this.stopSourceName = file;
		}
		this.stopLineNo = lineNo;
		createStackFrame();
		suspended(debugType);
		//check Expression
		int p = event.indexOf("\n");
		if (p >= 0) {
			String text = event.substring(p + 1);
			List<IWatchExpressionResult> results = FdbConsoleTextMatcher.expressionMatch(
					(AirDebugTarget) getDebugTarget(), text);

			for (IWatchExpressionResult r : results) {
				getAirDebugTarget().addExpressionResult(r.getExpressionText(), r);
			}
			fireChangeEvent(DebugEvent.STATE);
		}
	}

	public void resumed(String event, int debugType) {
		resumed(debugType);
	}

	public void waitingAddBreakpoint(String event) {
		// Nothing todo.
		// see AirDebugTarget.
	}

	public void waitingConnectPlayer(String event) {
		// Nothing todo.
		// see AirDebugTarget.
	}

	public void waitingContinue(String event) {
		// Nothing todo.
		// see AirDebugTarget.
	}

	public void waitingStartPlayer(String event) {
		// Nothing todo.
		// see AirDebugTarget.
	}

	public void playerTerminated() {

	}

	public void resultOfStackFrames(List<StackframeMatchResult> results) {
		otherStackFrame.clear();
		for (StackframeMatchResult result : results) {
			if (stackFrame != null && result.sourceName.equals(stackFrame.getSourceName())) {
				stackFrame.setFdbNo(result.id);
				stackFrame.setObjectString(result.objectString);
				stackFrame.setMethod(result.methodString);
				continue;
			}
			AirStackFrame stack = new AirStackFrame(this, result.sourceName, result.lineNo, result.id,
					result.objectString, result.methodString);
			otherStackFrame.put(result.sourceName, stack);
		}
	}

}
