package net.sf.amateras.air.mxml.models;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.util.ColorUtil;

/**
 * An abstract class for GUI component models.
 * 
 * @author Naoki Takezoe
 */
public abstract class AbstractComponentModel extends AbstractModel implements IComponentModel {

	protected static final String[] LIST_IME_MODE = { "", "ALPHANUMERIC_FULL", "ALPHANUMERIC_HALF", "CHINESE",
			"JAPANESE_HIRAGANA", "JAPANESE_KATAKANA_FULL", "JAPANESE_KATAKANA_HALF", "KOREAN" };

	protected static final String[] LIST_BORDER_STYLE = { "none", "solid", "inset", "outset" };

	protected static final String[] LIST_FONT_STYLE = { "normal", "italic" };

	protected static final String[] LIST_FONT_WEIGHT = { "normal", "bold" };

	protected static final String[] LIST_LEFT_RIGHT_CENTER = { "left", "right", "center" };

	protected static final String[] LIST_FONT_ALIAS_TYPE = { "advanced", "normal" };

	protected static final String[] LIST_GRID_FIT_TYPE = { "pixel", "none", "subpixel" };

	protected static final String[] LIST_TEXT_DECORATION = { "none", "underline" };

	protected static final String[] LIST_LABEL_PLACEMENT = { "right", "left", "bottom", "top" };

	protected static final String[] LIST_BUTTON_ALIGNMENT = { "left", "right", "auto" };

	protected static final String[] LIST_TITLE_ALIGNMENT = { "left", "center", "auto" };

	protected static final String[] LIST_SYSTEM_CHROME = { "none", "standard" };

	protected static final String[] LIST_WINDOW_TYPE = { "lightweight", "normal", "utility" };

	public static final String ALPHA = "alpha";
	public static final String ENABLED = "enabled";
	public static final String ERROR_STRING = "errorString";
	public static final String VISIBLE = "visible";
	public static final String CONSTRAINT = "constraint";
	public static final String STYLE_NAME = "styleName";
	public static final String TOOL_TIP = "toolTip";
	public static final String DOUBLE_CLICK_ENABLED = "doubleClickEnabled";
	public static final String FOCUS_ENABLED = "focusEnabled";
	public static final String MOUSE_FOCUS_ENABLED = "mouseFocusEnabled";

	public static final String ERROR_COLOR = "errorColor";
	public static final String THEME_COLOR = "themeColor";

	public static final String MAX_HEIGHT = "maxHeight";
	public static final String MAX_WIDTH = "maxWidth";
	public static final String MIN_HEIGHT = "minHeight";
	public static final String MIN_WIDTH = "minWidth";

	public static final String CLICK = "click";
	public static final String DOUBLE_CLICK = "doubleClick";
	public static final String INVALID = "invalid";
	public static final String VALID = "valid";
	public static final String HIDE = "hide";
	public static final String SHOW = "show";
	public static final String INITIALIZE = "initialize";

	protected FlexRectangle constraint;
	private IContainerModel root;

	private ArrayList<IComponentModel> children = new ArrayList<IComponentModel>();

	public AbstractComponentModel() {

	}

	@Override
	protected void installModelProperty() {
		super.installModelProperty();
		addDoubleModelProperty(ALPHA, CATEGORY_COMMON_PROPERTY, null);
		addBooleanModelProperty(ENABLED, CATEGORY_COMMON_PROPERTY, true);
		addStringModelProperty(ERROR_STRING, CATEGORY_COMMON_PROPERTY);
		addBooleanModelProperty(VISIBLE, CATEGORY_COMMON_PROPERTY, true);
		addStringModelProperty(STYLE_NAME, CATEGORY_COMMON_PROPERTY);
		addStringModelProperty(TOOL_TIP, CATEGORY_COMMON_PROPERTY);

		addBooleanModelProperty(DOUBLE_CLICK_ENABLED, CATEGORY_COMMON_PROPERTY, false);
		addBooleanModelProperty(FOCUS_ENABLED, CATEGORY_COMMON_PROPERTY, true);
		addBooleanModelProperty(MOUSE_FOCUS_ENABLED, CATEGORY_COMMON_PROPERTY, true);

		addNumberModelProperty(MAX_HEIGHT, CATEGORY_COMMON_PROPERTY);
		addNumberModelProperty(MAX_WIDTH, CATEGORY_COMMON_PROPERTY);
		addNumberModelProperty(MIN_HEIGHT, CATEGORY_COMMON_PROPERTY);
		addNumberModelProperty(MIN_HEIGHT, CATEGORY_COMMON_PROPERTY);

		addColorModelProperty(ERROR_COLOR, CATEGORY_COMMON_STYLE, ColorUtil.toRGB("#FF0000"));
		addColorModelProperty(THEME_COLOR, CATEGORY_COMMON_STYLE);

		addStringModelProperty(CLICK, CATEGORY_COMMON_EVENT);
		addStringModelProperty(DOUBLE_CLICK, CATEGORY_COMMON_EVENT);
		addStringModelProperty(INVALID, CATEGORY_COMMON_EVENT);
		addStringModelProperty(VALID, CATEGORY_COMMON_EVENT);
		addStringModelProperty(SHOW, CATEGORY_COMMON_EVENT);
		addStringModelProperty(VALID, CATEGORY_COMMON_EVENT);

		addWidthHeightModelProperty(X, CATEGORY_LAYOUT, "");
		addWidthHeightModelProperty(Y, CATEGORY_LAYOUT, "");
		addWidthHeightModelProperty(WIDTH, CATEGORY_LAYOUT, "");
		addWidthHeightModelProperty(HEIGHT, CATEGORY_LAYOUT, "");
	}

	public void fireChildPropertyChange() {
		firePropertyChange(AbstractContainerModel.KEY_CHILDREN, null, null);
	}

	public void moveDown() {
		IContainerModel model = getRoot();
		int index = model.getChildIndex(this);
		model.getChildren().remove(this);
		model.getChildren().add(index + 1, this);
		model.fireChildPropertyChange();
	}

	public boolean isMoveDownEnabled() {
		IContainerModel model = getRoot();
		int index = model.getChildIndex(this) + 1;
		return (index < model.getChildren().size()) ? true : false;
	}

	public void moveUp() {
		IContainerModel model = getRoot();
		int index = model.getChildIndex(this);
		model.getChildren().remove(this);
		model.getChildren().add(index - 1, this);
		model.fireChildPropertyChange();
	}

	public boolean isMoveUpEnabled() {
		IContainerModel model = getRoot();
		int index = model.getChildIndex(this);
		return index == 0 ? false : true;
	}

	public void setRoot(IContainerModel root) {
		this.root = root;
	}

	public IContainerModel getRoot() {
		return root;
	}

	public FlexRectangle getConstraint() {
		return constraint;
	}

	public void setConstraint(FlexRectangle constraint) {
		this.constraint = constraint;
		setPropertyValue("_" + X, constraint.getStrX(), false);
		setPropertyValue("_" + Y, constraint.getStrY(), false);
		setPropertyValue("_" + WIDTH, constraint.getStrWidth(), false);
		setPropertyValue("_" + HEIGHT, constraint.getStrHeight(), false);
		firePropertyChange("_" + CONSTRAINT, null, constraint);
	}

	protected abstract String getComponentName();

	@Override
	public String toMXML() {
		StringBuilder sb = new StringBuilder();
		sb.append("<").append(AIRPlugin.getDefault().getNameSpacePrefix()).append(getComponentName());
		sb.append(getAttributesXML());

		//sb.append(getLocationAttributes());
		sb.append(getAdditionalAttributesXML());

		String childNodeXML = getChildAllNodeXML();
		if (childNodeXML != null && childNodeXML.length() != 0) {
			sb.append(">").append("\n");
			sb.append(childNodeXML);
			sb.append("</").append(AIRPlugin.getDefault().getNameSpacePrefix()).append(getComponentName()).append(">");
		} else {
			sb.append("/>");
		}
		return sb.toString();
	}

	protected String getChildAllNodeXML() {
		return getAdditionalChildElementsXML() + getChildrenXML();
	}

	@Override
	public String toString() {
		String str = AIRPlugin.getDefault().getNameSpacePrefix() + getComponentName();

		String id = getAttributeToString(ID);
		if (id.length() > 0) {
			str += " (id=" + id + ")";
		}
		return str;
	}

	public int getChildIndex(IComponentModel child) {
		int index = -1;
		for (IComponentModel model : children) {
			index++;
			if (model == child) {
				return index;
			}
		}
		return index;
	}

	public void addChild(IComponentModel child) {
		addChild(getChildren().size(), child);
	}

	public void addChild(int index, IComponentModel child) {
		children.add(index, child);
		fireChildPropertyChange();
	}

	public List<IComponentModel> getChildren() {
		return children;
	}

	public void removeChild(IComponentModel child) {
		boolean isRemoved = children.remove(child);
		if (isRemoved) {
			fireChildPropertyChange();
		} else {
			System.err.println("not remove child. parent=" + this + ", child=" + child);
		}
	}

	public void removeAll() {
		Iterator<IComponentModel> ite = children.iterator();
		while (ite.hasNext()) {
			Object object = ite.next();
			if (object instanceof AbstractContainerModel) {
				((AbstractContainerModel) object).removeAll();
			}
		}
		children.clear();
		fireChildPropertyChange();
	}

	// to XML
	// ----------------------------------------
	protected String getChildrenXML() {
		StringBuilder sb = new StringBuilder();
		if (children.size() != 0) {
			Iterator<IComponentModel> ite = children.iterator();
			while (ite.hasNext()) {
				sb.append(((AbstractModel) ite.next()).toMXML());
				sb.append("\n");
			}
		}
		return sb.toString();
	}

	/**
	 * Whether it is possible to add it as a child component or not?
	 * If it is AbstractComponentModel and not IDecidedParentComonent, it is OK.
	 * @param model
	 * @return
	 */
	public boolean canChildComponent(Object model) {
		return false;
	}

	public void addScript(String script) {

	}

	/**
	 * return to flex class pakage name.
	 * 
	 * @return
	 */
	public abstract String getFlexModelPackageName();

	/**
	 * return flex class name.
	 * @return
	 */
	public String getFlexClassName() {
		return getFlexModelPackageName() + "." + getComponentName();
	}
}
