package net.sf.amateras.air.propertyPages;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.builder.CompileProperty;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * create Build Setting dialog
 * @author ogawahideko
 *
 */
public class CreateBuildSettingDialog extends TitleAreaDialog {
	private CompileProperty compile;
	private Text command;

	private Button mxmlType;
	private Button as3Type;
	private Button flex3Type;
	private Button chkExtention;
	private Text options;

	private IProject project;
	private ScopedPreferenceStore store;

	private List<CheckEntry> needText = new ArrayList<CheckEntry>();

	private Map<Button, Text> optionMap = new HashMap<Button, Text>();

	public CreateBuildSettingDialog(Shell parentShell, IProject project, CompileProperty compile) {
		super(parentShell);
		this.project = project;
		this.compile = compile;
		store = new ScopedPreferenceStore(new ProjectScope(project), AIRPlugin.PLUGIN_ID);
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		setTitle(AIRPlugin.getResourceString("BUILD_SETTING") + " - " + compile.getDirectory());

		Composite area = (Composite) super.createDialogArea(parent);
		Composite composite = new Composite(area, SWT.NULL);
		composite.setLayout(new GridLayout(1, false));
		composite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		Group groupType = new Group(composite, SWT.NULL);
		groupType.setLayout(new GridLayout(3, false));
		groupType.setText(AIRPlugin.getResourceString("BUILD_TYPE"));
		groupType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		mxmlType = new Button(groupType, SWT.RADIO);
		mxmlType.setText("AIRFlex");
		mxmlType.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setDefaultSetting(CompileProperty.MXML_TYPE);
			}
		});
		as3Type = new Button(groupType, SWT.RADIO);
		as3Type.setText("ActionScript");
		as3Type.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setDefaultSetting(CompileProperty.AS3_TYPE);
			}
		});

		flex3Type = new Button(groupType, SWT.RADIO);
		flex3Type.setText("Flex3");
		flex3Type.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				setDefaultSetting(CompileProperty.FLEX3_TYPE);
			}
		});

		chkExtention = new Button(composite, SWT.CHECK);
		chkExtention.setText(AIRPlugin.getResourceString("ENABLE_BUILD_EXTENSION"));
		chkExtention.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				refreshExtensionState();
			}
		});
		Group group = new Group(composite, SWT.NULL);
		group.setLayout(new GridLayout(2, false));
		group.setLayoutData(new GridData(GridData.FILL_BOTH));

		command = createText("Command", group, true);
		options = createText("Options", group, false);

		addOption("-output", group);
		addOption("-library-path+=", group);
		addOption("-source-path", group);
		addOption("-include-classes", group);
		addOption("-locale=", group);
		addOption("-allow-source-path-overlap=", group);
		addOption("-debug=", group);

		return parent;
	}

	private void addOption(String key, Composite parent) {
		Button check = new Button(parent, SWT.CHECK);
		check.setText(key);
		Text text = new Text(parent, SWT.BORDER);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		optionMap.put(check, text);

	}

	private void refreshExtensionState() {
		if (chkExtention.getSelection()) {
			command.setEnabled(true);
			options.setEnabled(true);
			for (Button key : optionMap.keySet()) {
				key.setEnabled(true);
				optionMap.get(key).setEnabled(true);
			}

		} else {
			command.setEnabled(false);
			options.setEnabled(false);
			for (Button key : optionMap.keySet()) {
				key.setEnabled(false);
				optionMap.get(key).setEnabled(false);
			}
		}
	}

	@Override
	protected Control createContents(Composite parent) {
		Control control = super.createContents(parent);
		if (compile.getCommandType() == CompileProperty.MXML_TYPE) {
			mxmlType.setSelection(true);
		} else if (compile.getCommandType() == CompileProperty.AS3_TYPE) {
			as3Type.setSelection(true);
		} else if (compile.getCommandType() == CompileProperty.FLEX3_TYPE) {
			flex3Type.setSelection(true);
		}

		if (compile.isExtention()) {
			chkExtention.setSelection(true);
			setTypeContents(compile.getCommandType(), compile.getCommand(), compile.getOption());
		} else {
			chkExtention.setSelection(false);
			setDefaultSetting(compile.getCommandType());
		}

		setASCompVisible(compile.getCommandType());

		doValidate();
		return control;
	}

	private void setTypeContents(int type, String commandValue, String optionValue) {
		command.setText(commandValue);
		options.setText("");

		List<String> opt = parseOption(optionValue);
		for (Button key : optionMap.keySet()) {
			key.setSelection(false);
			optionMap.get(key).setText("");
		}

		for (int i = 0; i < opt.size(); i++) {
			String option = opt.get(i);
			System.out.println(i + ":" + option);
			boolean isSelected = false;
			for (Button key : optionMap.keySet()) {
				if (key.getText().endsWith("=")) {
					if (option.startsWith(key.getText())) {
						key.setSelection(true);
						optionMap.get(key).setText(replaceString(option.substring(key.getText().length())));
						isSelected = true;
						break;
					}
				} else {
					if (option.startsWith(key.getText() + " ") || option.startsWith(key.getText() + "=")
							|| option.startsWith(key.getText() + "+=")) {
						key.setSelection(true);
						String value = option.substring(key.getText().length());
						if (value.startsWith(" ")) {
							optionMap.get(key).setText(replaceString(value.substring(1)));
						} else {
							optionMap.get(key).setText(replaceString(value));
						}

						isSelected = true;
						break;
					}

				}
			}

			if (!isSelected) {
				if (options.getText().length() > 0) {
					options.setText(options.getText() + " " + replaceString(option));
				} else {
					options.setText(replaceString(option));
				}
			}
		}

		refreshExtensionState();

	}

	protected static List<String> parseOption(String value) {
		List<String> list = new ArrayList<String>();
		StringBuilder sb = new StringBuilder();
		char[] chars = value.toCharArray();

		for (int i = 0; i < chars.length; i++) {
			if (i >= chars.length - 1) {
				sb.append(chars[i]);
				list.add(sb.toString());
				sb = new StringBuilder();

			} else if (chars[i] == '-') {
				if (chars[i + 1] == '-') {
					sb.append(chars[i]);
					sb.append(chars[i + 1]);
					if (sb.length() != 0) {
						list.add(sb.toString());
						sb = new StringBuilder();
					}
					i++;
					continue;
				} else {
					sb.append(chars[i]);
					continue;
				}

			} else if (chars[i] == ' ') {
				if (chars[i + 1] == '-' || chars[i + 1] == '+') {
					if (chars[i + 1] == '-' && i < chars.length - 2 && chars[i + 2] == '-') {
						sb.append(chars[i]);
						sb.append(chars[i + 1]);
						sb.append(chars[i + 2]);
						if (sb.length() != 0) {
							list.add(sb.toString());
							sb = new StringBuilder();
						}
						i++;
						i++;
						continue;
					} else if (sb.length() != 0) {
						list.add(sb.toString());
						sb = new StringBuilder();
					}
					sb.append(chars[i + 1]);
					i++;
					continue;
				} else {
					sb.append(chars[i]);
					continue;
				}

			} else {
				sb.append(chars[i]);
			}
		}
		return list;
	}

	static final void main(String[] args) {
		List<String> list = parseOption("+configname=air flextest.mxml "
				+ "-output flextest.swf -library-path+=${LIBRARY_PATH}, 2 " + "-souce-path aaa bbb ccc -- -nextOption");
		for (String l : list) {
			System.out.println(l);
		}
	}

	private void setDefaultSetting(int type) {
		if (type == CompileProperty.MXML_TYPE) {
			setTypeContents(compile.getCommandType(), store.getDefaultString(AIRPlugin.PREF_AIR_COMPILE_COMMAND), store
					.getDefaultString(AIRPlugin.PREF_AIR_COMPILE_OPTION));
		} else if (type == CompileProperty.AS3_TYPE) {
			setTypeContents(compile.getCommandType(), store.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_COMMAND), store
					.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_OPTION));
		} else if (type == CompileProperty.FLEX3_TYPE) {
			setTypeContents(compile.getCommandType(), store.getDefaultString(AIRPlugin.PREF_FLEX3_COMPILE_COMMAND),
					store.getDefaultString(AIRPlugin.PREF_FLEX3_COMPILE_OPTION));

		}
		setASCompVisible(type);
	}

	private void setASCompVisible(int type) {
		//		if (type == CompileProperty.MXML_TYPE) {
		////			chkSourcePath.setVisible(false);
		////			txtSourcePath.setVisible(false);
		//			chkIncludeClasses.setVisible(false);
		//			txtIncludeClasses.setVisible(false);
		//		} else {
		////			chkSourcePath.setVisible(true);
		////			txtSourcePath.setVisible(true);
		//			chkIncludeClasses.setVisible(true);
		//			txtIncludeClasses.setVisible(true);
		//		}
	}

	@Override
	protected void okPressed() {
		if (mxmlType.getSelection()) {
			compile.setCommandType(CompileProperty.MXML_TYPE);
		} else if (as3Type.getSelection()) {
			compile.setCommandType(CompileProperty.AS3_TYPE);
		} else if (flex3Type.getSelection()) {
			compile.setCommandType(CompileProperty.FLEX3_TYPE);
		}

		compile.setExtention(chkExtention.getSelection());
		if (chkExtention.getSelection()) {
			compile.setCommand(command.getText());
			StringBuilder sb = new StringBuilder();

			if (options.getText().length() > 0) {
				sb.append(options.getText()).append(" ");
			}

			for (Button key : optionMap.keySet()) {
				if (key.getSelection()) {
					sb.append(key.getText());
					if (!key.getText().endsWith("=") && !optionMap.get(key).getText().startsWith("=")
							&& !optionMap.get(key).getText().startsWith("+=")) {
						sb.append(" ");
					}
					sb.append(optionMap.get(key).getText()).append(" ");
				}
			}
			compile.setOption(sb.toString().trim());
		} else {
			if (compile.getCommandType() == CompileProperty.MXML_TYPE) {
				compile.setCommand(store.getDefaultString(AIRPlugin.PREF_AIR_COMPILE_COMMAND));
				compile.setOption(store.getDefaultString(AIRPlugin.PREF_AIR_COMPILE_OPTION));
			} else if (compile.getCommandType() == CompileProperty.AS3_TYPE) {
				compile.setCommand(store.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_COMMAND));
				compile.setOption(store.getDefaultString(AIRPlugin.PREF_AS3_COMPILE_OPTION));
			} else if (compile.getCommandType() == CompileProperty.FLEX3_TYPE) {
				compile.setCommand(store.getDefaultString(AIRPlugin.PREF_FLEX3_COMPILE_COMMAND));
				compile.setOption(store.getDefaultString(AIRPlugin.PREF_FLEX3_COMPILE_OPTION));
			}
		}
		super.okPressed();
	}

	private Text createText(String label, Composite parent, boolean isNeed) {
		return createText(label, parent, isNeed, SWT.BORDER);
	}

	private Text createText(String label, Composite parent, boolean isNeed, int textType) {
		UIUtil.createLabel(label, parent);
		Text text = new Text(parent, textType);
		text.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		if (isNeed) {
			needText.add(new CheckEntry(text, label));
			text.addModifyListener(new ModifyListener() {
				public void modifyText(ModifyEvent e) {
					doValidate();
				}
			});
		}
		return text;
	}

	private void doValidate() {
		for (CheckEntry check : needText) {
			if (check.getText().getText().length() == 0) {
				setErrorMessage(getMessage("IS_NEED", check.getMessage()));
				getButton(IDialogConstants.OK_ID).setEnabled(false);
				return;
			}
		}
		setErrorMessage(null);
		getButton(IDialogConstants.OK_ID).setEnabled(true);
	}

	private String getMessage(String key, Object... args) {
		return AIRPlugin.getResourceString(key, args);
	}

	class CheckEntry {
		private Text text;
		private String message;

		CheckEntry(Text text, String message) {
			this.text = text;
			this.message = message;
		}

		public Text getText() {
			return text;
		}

		public void setText(Text text) {
			this.text = text;
		}

		public String getMessage() {
			return message;
		}

		public void setMessage(String message) {
			this.message = message;
		}
	}

	private String replaceString(String value) {
		try {
			value = value.replaceAll("\\$\\{PROJECT_NAME\\}", project.getName());
			
			String directory = compile.getDirectory();
			if (directory.length() != 0) {
				directory = directory + "/";
			}
			value = value.replaceAll("\\$\\{DIRECTORY\\}", directory);
			//			value = value.replaceAll("\\$\\{LIBRARY_PATH\\}", store.getString(
			//					AIRPlugin.PREF_FLEX_LIBRARIES).replaceAll("\t", ","));
			return value;
		} catch (Exception ex) {
			ex.printStackTrace();
			return value;
		}
	}
}
