package net.sf.amateras.air.wizards;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.builder.AirBuilderManager;
import net.sf.amateras.air.builder.CompileProperty;
import net.sf.amateras.air.util.UIUtil;

import org.eclipse.core.internal.resources.File;
import org.eclipse.core.internal.resources.Folder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.debug.internal.ui.DebugUIPlugin;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.internal.WorkbenchPlugin;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.preferences.ScopedPreferenceStore;

/**
 * Choose resources to export as the AIR package.
 * 
 * @author Naoki Takezoe
 */
@SuppressWarnings("restriction")
public class ExportAIRWizardPage extends WizardPage {

	private IProject project;
	private CheckboxTreeViewer treeViewer;
	private Text packageName;
	private Text storeType;
	private Text keystore;
	private Text storepass;
	private String descriptor;

	private ScopedPreferenceStore store;

	protected ExportAIRWizardPage(IProject project) {
		super("ExportAIRWizardPage");
		setTitle(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_TITLE"));
		setDescription(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_DESCRIPTION"));
		this.project = project;
		store = new ScopedPreferenceStore(new ProjectScope(this.project), AIRPlugin.PLUGIN_ID);
		descriptor = store.getString(AIRPlugin.PREF_DESCRIPTOR);
	}

	public void createControl(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout(1, false));

		treeViewer = new CheckboxTreeViewer(composite);
		treeViewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));
		treeViewer.setContentProvider(new WorkbenchContentProvider());
		treeViewer.setLabelProvider(new WorkbenchLabelProvider());
		treeViewer.setInput(project);
		treeViewer.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent arg0) {
				doValidate();
			}
		});
		Widget o = treeViewer.getControl();
		String[] listArray = null;
		String list = store.getString(AIRPlugin.PREF_EXPORT_PACK_LIST);
		if (list != null && list.length() != 0) {
			listArray = list.split(";");
		}
		checkDescription(o, listArray);

		// package name
		Composite exportPanel = new Composite(composite, SWT.NULL);
		exportPanel.setLayout(new GridLayout(2, false));
		exportPanel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_NAME"), exportPanel);
		packageName = new Text(exportPanel, SWT.BORDER);
		packageName.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		String name = store.getString(AIRPlugin.PREF_EXPORT_PACKAGE_NAME);
		if (name == null || name.length() == 0) {
			name = project.getName() + ".air";
		}
		packageName.setText(name);
		packageName.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		// store type
		Group group = new Group(composite, SWT.NULL);
		group.setText(AIRPlugin.getResourceString("CERTIFICATE"));
		group.setLayout(new GridLayout(2, false));
		group.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_STORE_TYPE"), group);
		storeType = new Text(group, SWT.BORDER);
		storeType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		String work = store.getString(AIRPlugin.PREF_EXPORT_STORE_TYPE);
		if (work == null || work.length() == 0) {
			work = "pkcs12";
		}
		storeType.setText(work);
		storeType.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		//KeyStore
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_KEY_STORE"), group);
		Composite panel = new Composite(group, SWT.NULL);
		panel.setLayout(new GridLayout(2, false));
		panel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		keystore = new Text(panel, SWT.BORDER);
		keystore.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		work = store.getString(AIRPlugin.PREF_EXPORT_KEY_STORE);
		if (work == null || work.length() == 0) {
			work = "";
		}
		keystore.setText(work);
		keystore.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});
		UIUtil.createBrowseFileButton(keystore, panel);

		//StorePass
		UIUtil.createLabel(AIRPlugin.getResourceString("EXPORT_AIR_PACKAGE_STOREPASS"), group);
		storepass = new Text(group, SWT.BORDER | SWT.PASSWORD);
		storepass.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		work = store.getString(AIRPlugin.PREF_EXPORT_STORE_PASS);
		if (work == null || work.length() == 0) {
			work = "";
		}
		storepass.setText(work);
		storepass.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				doValidate();
			}
		});

		// create Strore
		UIUtil.createLabel("", group);
		Button button = new Button(group, SWT.PUSH);
		button.setText(AIRPlugin.getResourceString("CREATE_CERTIFICATE"));
		button.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent evt) {
				CreateCertificateDialog dialog = new CreateCertificateDialog(getShell(), project, keystore, storepass);
				dialog.open();
				dialog = null;
			}
		});

		setControl(composite);
		doValidate();
	}

	private void checkDescription(Widget o, String[] listArray) {
		Item[] items = getChildren(o);
		for (int i = 0; i < items.length; i++) {
			Item item = items[i];
			if (item instanceof TreeItem) {
				setDefaultChecked((TreeItem) item, listArray);
			}
			checkDescription(item, listArray);
		}
	}

	private Item[] getChildren(Widget o) {
		if (o instanceof TreeItem) {
			return ((TreeItem) o).getItems();
		} else if (o instanceof Tree) {
			return ((Tree) o).getItems();
		} else {
			return null;
		}
	}

	private void setDefaultChecked(TreeItem item, String[] listArray) {
		Object data = item.getData();
		if (data != null) {
			String fileName = null;
			if (data instanceof File) {
				fileName = ((File) data).getName();
			} else if (data instanceof Folder) {
				fileName = ((Folder) data).getName();
			}

			if (fileName.equals(descriptor)) {
				item.setChecked(true);

			} else if (listArray != null) {
				for (String check : listArray) {
					if (fileName.equals(check)) {
						item.setChecked(true);
						return;
					}
				}

			} else if (data instanceof File && (fileName.endsWith(".html") || fileName.endsWith(".swf"))) {
				item.setChecked(true);
			}
		}
	}

	private void doValidate() {
		if (treeViewer.getCheckedElements().length == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_RESOURCE"));
			setPageComplete(false);
			return;
		}
		if (packageName.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_NAME"));
			setPageComplete(false);
			return;
		}
		if (storeType.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_STORE_TYPE"));
			setPageComplete(false);
			return;
		}
		if (keystore.getText().length() == 0) {
			setErrorMessage(AIRPlugin.getResourceString("EXPORT_AIR_CHECK_STORE"));
			setPageComplete(false);
			return;
		}
		setErrorMessage(null);
		setPageComplete(true);
	}

	/**
	 * export prepare
	 */
	private void prepare() {
		ScopedPreferenceStore scopedPreferenceStore = new ScopedPreferenceStore(new ProjectScope(project),
				AIRPlugin.PLUGIN_ID);
		List<CompileProperty> compiles = CompileProperty.load(scopedPreferenceStore);
		boolean isChanged = false;
		for (CompileProperty prop : compiles) {
			String option = prop.getOption();
			if (option.indexOf("-debug=true") >= 0) {
				isChanged = true;
				break;
			}
		}

		if (isChanged && confirmRemoveDebugOption()) {
			for (CompileProperty prop : compiles) {
				String option = prop.getOption();
				option = option.replaceAll("-debug=true", "");
				prop.setOption(option);
			}

			CompileProperty.save(scopedPreferenceStore, compiles);
			WorkbenchPlugin.getDefault().getDecoratorManager().updateForEnablementChange();
			AirBuilderManager.doClean(project);
		}
	}

	private boolean confirmRemoveDebugOption() {
		return MessageDialog.openConfirm(DebugUIPlugin.getShell(), "CONFIRM", AIRPlugin
				.getResourceString("REMOVE_DEBUG_OPTION"));
	}

	public boolean finish() {
		prepare();
		String prefDescriptor = store.getString(AIRPlugin.PREF_DESCRIPTOR);

		store.setValue(AIRPlugin.PREF_EXPORT_STORE_TYPE, storeType.getText());
		store.setValue(AIRPlugin.PREF_EXPORT_KEY_STORE, keystore.getText());
		store.setValue(AIRPlugin.PREF_EXPORT_STORE_PASS, storepass.getText());
		store.setValue(AIRPlugin.PREF_EXPORT_PACKAGE_NAME, packageName.getText());

		Object[] checkedElements = treeViewer.getCheckedElements();

		List<String> options = new ArrayList<String>();
		options.add("-package");
		options.add("-storetype");
		options.add(storeType.getText());
		options.add("-keystore");
		options.add(keystore.getText());
		options.add("-storepass");
		options.add(storepass.getText());
		options.add(packageName.getText());

		List<IResource> resources = new ArrayList<IResource>();
		for (int i = 0; i < checkedElements.length; i++) {
			if (checkedElements[i] instanceof IResource) {
				if (((IResource) checkedElements[i]).getName().endsWith(prefDescriptor)) {
					resources.add(0, (IResource) checkedElements[i]);
				} else {
					resources.add((IResource) checkedElements[i]);
				}
			}
		}

		StringBuilder sb = new StringBuilder();
		for (IResource resource : resources) {
			sb.append(resource.getProjectRelativePath().toOSString()).append(";");
		}
		store.setValue(AIRPlugin.PREF_EXPORT_PACK_LIST, sb.toString());
		try {
			store.save();
		} catch (IOException e) {
			e.printStackTrace();
		}

		ADTRunner runner = new ADTRunner();
		return runner.run(project, options, resources);

	}
}
