package net.sf.amateras.air.debug.ruler;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.sf.amateras.air.AIRPlugin;
import net.sf.amateras.air.debug.AirLineBreakPoint;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.IVerticalRulerInfo;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.ui.texteditor.IUpdate;
import org.eclipse.ui.texteditor.MarkerUtilities;

public abstract class AbstractBreakpointRulerAction extends Action implements IUpdate {

	protected IVerticalRulerInfo fInfo;
	protected ITextEditor fTextEditor;
	private IBreakpoint fBreakpoint;

	protected IBreakpoint getBreakpoint() {
		return fBreakpoint;
	}

	protected void setBreakpoint(IBreakpoint breakpoint) {
		fBreakpoint = breakpoint;
	}

	protected ITextEditor getTextEditor() {
		return fTextEditor;
	}

	protected void setTextEditor(ITextEditor textEditor) {
		fTextEditor = textEditor;
	}

	protected IVerticalRulerInfo getInfo() {
		return fInfo;
	}

	protected void setInfo(IVerticalRulerInfo info) {
		fInfo = info;
	}

	protected IDocument getDocument() {
		IDocumentProvider provider = fTextEditor.getDocumentProvider();
		return provider.getDocument(fTextEditor.getEditorInput());
	}

	protected IResource getResourceForDebugMarkers() {
		return getResourceForDebugMarkers(fTextEditor);
	}

	public static Position getMarkerPosition(IDocument document, IMarker marker) {
		int start = MarkerUtilities.getCharStart(marker);
		int end = MarkerUtilities.getCharEnd(marker);

		if (start > end) {
			end = start + end;
			start = end - start;
			end = end - start;
		}

		if (start == -1 && end == -1) {
			// marker line number is 1-based
			int line = MarkerUtilities.getLineNumber(marker);
			if (line > 0 && document != null) {
				try {
					start = document.getLineOffset(line - 1);
					end = start;
				} catch (BadLocationException x) {
					AIRPlugin.logException(x);
				}
			}
		}

		if (start > -1 && end > -1) {
			return new Position(start, end - start);
		}

		return null;
	}

	public static IResource getResourceForDebugMarkers(ITextEditor textEditor) {
		IEditorInput input = textEditor.getEditorInput();
		IResource resource = (IResource) input.getAdapter(IFile.class);
		if (resource == null) {
			resource = (IResource) input.getAdapter(IResource.class);
		}
		if (resource == null) {
			resource = ResourcesPlugin.getWorkspace().getRoot();
		}
		return resource;
	}

	public static boolean includesRulerLine(Position position, IDocument document, IVerticalRulerInfo info) {
		if (position != null && info != null && document != null) {
			try {
				int markerLine = document.getLineOfOffset(position.getOffset());
				int line = info.getLineOfLastMouseButtonActivity();
				if (line == markerLine) {
					return true;
				}
			} catch (BadLocationException x) {
				AIRPlugin.logException(x);
			}
		}
		return false;
	}

	public static List<IMarker> getMarkersFromEditorResource(IResource resource, IDocument document,
			IEditorInput editorInput, IVerticalRulerInfo info, boolean onlyIncludeLastLineActivity) {

		if (onlyIncludeLastLineActivity) {
			Assert.isNotNull(info);
		}

		List<IMarker> breakpoints = new ArrayList<IMarker>();

		try {
			List<IMarker> markers = new ArrayList<IMarker>();
			boolean isExternalFile = false;

			markers.addAll(Arrays.asList(resource.findMarkers(AirLineBreakPoint.BREAK_POINT_MARKER, true,
					IResource.DEPTH_INFINITE)));

			if (!(resource instanceof IFile)) {
				//it was created from an external file
				isExternalFile = true;
			}

			IBreakpointManager breakpointManager = DebugPlugin.getDefault().getBreakpointManager();
			for (IMarker marker : markers) {
				if (marker == null) {
					continue;
				}
				IBreakpoint breakpoint = breakpointManager.getBreakpoint(marker);
				if (breakpoint != null && breakpointManager.isRegistered(breakpoint)) {
					Position pos = getMarkerPosition(document, marker);

					if (!isExternalFile) {
						if (!onlyIncludeLastLineActivity) {
							breakpoints.add(marker);
						} else if (includesRulerLine(pos, document, info)) {
							breakpoints.add(marker);
						}
					} else {

						//if(isInSameExternalEditor(marker, editorInput)){
						if (!onlyIncludeLastLineActivity) {
							breakpoints.add(marker);
						} else if (includesRulerLine(pos, document, info)) {
							breakpoints.add(marker);
						}
						//}
					}
				}
			}
		} catch (CoreException x) {
			AIRPlugin.logException(x);
		}
		return breakpoints;
	}

}
