/*
 * Copyright (c) 2004, SHIRAMOTO Takeyuki
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Debugger.cc,v 1.8 2004/05/17 07:31:52 siramoto Exp $
 */

#include<cassert>
#include<cstdlib>
#include<iostream>
#include<iomanip>
#include<sstream>
#include<string>
#include<vector>

using namespace std;

#include"Debugger.h"
#include"Input_position.h"
#include"global.h"

Debugger::Debugger(bool mode)
:debug_mode(mode),
 debug_level(0),
 stopped(false)
{}

void
Debugger::in(const Input_position& position, const string& description)
{
	in_noninteractive(position, description);
	if (debug_mode) {
		if (step_mode || debug_stack.size() <= debug_level + 1)
			inquire();
	}
}

void
Debugger::in_noninteractive(const Input_position& position,
			    const string& description)
{
	debug_element e(position, description, false, false, false);
	debug_stack.push_back(e);
}

void
Debugger::out(const string& result)
{
	if (debug_mode) {
		if (debug_stack.size() <= debug_level + 1) {
			cerr << debug_stack.back().description <<
				"\nresult -->\n" <<
				result << endl;
		}
	}
	out();
}

istream*
Debugger::out(istream* result)
{
	if (debug_mode) {
		if (debug_stack.size() <= debug_level + 1) {
			cerr << debug_stack.back().description << endl;
			char ch;
			string s;
			while (result->get(ch)) {
				cerr << ch;
				s += ch;
			}
			delete result;
			result = new istringstream(s);
		}
	}
	out();
	return result;
}

void
Debugger::out()
{
	assert(! debug_stack.empty());
	debug_element d = debug_stack.back();
	debug_stack.pop_back();
	assert(! d.is_expanded);
	assert(! d.is_sub);
}

void
Debugger::put_msg(const string& msg) const
{
	if (debug_mode)
		if (step_mode || debug_stack.size() <= debug_level)
			cerr << msg << endl;
}

void
Debugger::in_expanded(const Input_position& position, const string& description)
{
	debug_element d(position, description, true, false, false);
	debug_stack.push_back(d);
	if (debug_mode) {
		if (step_mode || debug_stack.size() <= debug_level + 1)
			inquire();
	}
}

void
Debugger::out_expanded()
{
	assert(! debug_stack.empty());
	vector<debug_element>::reverse_iterator p;
	for (p = debug_stack.rbegin(); p != debug_stack.rend(); p++) {
		if (p->is_expanded && !p->is_released) {
			p->is_released = true;
			break;
		}
	}
	assert(p != debug_stack.rend());
}

void
Debugger::in_sub(const Input_position& position,
		 const string& description)
{
	debug_element d(position, description, true, false, true);
	debug_stack.push_back(d);
	result_output_enable_stack.push_back(result_output_enable);
	result_output_enable = false;
	if (debug_mode) {
		if (step_mode || debug_stack.size() <= debug_level + 1)
			inquire();
	}
}

void
Debugger::clean_released()
{
	for (;;) {
		if (debug_stack.empty() ||
		    ! debug_stack.back().is_expanded ||
		    ! debug_stack.back().is_released)
		{
			break;
		}
		debug_element d = debug_stack.back();
		debug_stack.pop_back();
		assert(d.is_expanded);
		assert(d.is_released);
		delete d.position.is;
		if (d.is_sub) {
			assert(! result_output_enable_stack.empty());
			result_output_enable =
				result_output_enable_stack.back();
			result_output_enable_stack.pop_back();
		}
	}
}

void
Debugger::inquire_put_file(int level)
{
	debug_element &de = debug_stack[level];
	cerr << "=== " << de.position.name << " ===" << endl;

	streampos current_pos = de.position.is->tellg();
	istream::iostate current_state = de.position.is->rdstate();
	de.position.is->clear();
	de.position.is->seekg(0, istream::beg);
	if (de.position.is->fail()) {
		if (de.position.is->eof())
			cerr << "empty stream" << endl;
		else
			cerr << "unseekable stream" << endl;
		de.position.is->clear();
		return;
	}

	char ch;
	int line_no = 1;
	cerr << setw(4) << line_no << ":";
	while (de.position.is->get(ch)) {
		cerr << ch;
		if (ch == '\n') {
			line_no ++;
			cerr << setw(4) << line_no << ":";
		}
	}
	de.position.is->clear();
	de.position.is->seekg(current_pos);
	de.position.is->setstate(current_state);
}

void
Debugger::inquire_put_position(int level)
{
	debug_element &de = debug_stack[level];
	cerr << "\nat " << de.description << endl;
	cerr << "in " << de.position.name <<
	     ":" << de.position.row << "," << de.position.column << endl;

	streampos current_pos = de.position.is->tellg();
	istream::iostate current_state = de.position.is->rdstate();
	de.position.is->clear();
	de.position.is->seekg(de.position.pos);
	if (de.position.is->fail()) {
		if (de.position.is->eof())
			cerr << "empty stream" << endl;
		else
			cerr << "unseekable stream" << endl;
		de.position.is->clear();
		return;
	}
	for (int i = 1; i <= 3; i++) {
		char s[80];
		de.position.is->getline(s, sizeof s);
		cerr << s << endl;
	}
	de.position.is->clear();
	de.position.is->seekg(current_pos);
	de.position.is->setstate(current_state);
}



void
Debugger::inquire()
{
	string cmd;
	int level = debug_stack.size() - 1;
	for (;;) {
		assert(level >= 0);
		assert(level < debug_stack.size());
		inquire_put_position(level);
		cerr << "debug> ";
		if (cin.eof())
			break;
		cin >> cmd;
		if (cmd == "s") {
			if (stopped) {
				cerr << "program is not running" << endl;
			} else {
				step_mode = true;
				result_output_enable = true;
				break;
			}
		} else if (cmd == "n") {
			if (stopped) {
				cerr << "program is not running" << endl;
			} else {
				step_mode = false;
				result_output_enable = true;
				break;
			}
		} else if (cmd == "N") {
			if (stopped) {
				cerr << "program is not running" << endl;
			} else {
				step_mode = false;
				result_output_enable = false;
				break;
			}
		} else if (cmd == "u") {
			if (level > 0)
				level --;
			else
				cerr << "in the top level!" << endl;
		} else if (cmd == "d") {
			if (level < debug_stack.size() - 1)
				level ++;
			else
				cerr << "in the bottom level!" << endl;
		} else if (cmd == "U") {
			level = 0;
		} else if (cmd == "D") {
			level = debug_stack.size() - 1;
		} else if (cmd == "f") {
			inquire_put_file(level);
		} else {
			cerr << "command not found" << endl;
		}
	}
	debug_level = level;
}

void
Debugger::put(const string& result) const
{
	if (debug_mode) {
		if (result_output_enable || step_mode ||
		    debug_stack.size() <= debug_level) 
		{
			cerr << result;
		}
	}
}

string
Debugger::stack_trace() const
{
	ostringstream s;
	for (vector<debug_element>::const_reverse_iterator p =
		debug_stack.rbegin();
	     p != debug_stack.rend();
	     p ++)
	{
		s << "at " << p->description << " in " <<
		  p->position.name << ":" <<
		  p->position.row << "," <<
		  p->position.column << endl;
	}
	return s.str();
}

void
Debugger::error()
{
	stopped = true;
	cerr << stack_trace() << endl;
	if (debug_mode) 
		inquire();
	exit(EXIT_FAILURE);
}


