/*
 * Copyright (c) 2004, SHIRAMOTO Takeyuki
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: main.cc,v 1.8 2004/05/17 07:31:52 siramoto Exp $
 */

#include<algorithm>
#include<fstream>
#include<cassert>
#include<cctype>
#include<cstdlib>
#include<iostream>
#include<sstream>
#include<stack>
#include<vector>

extern "C" {
#include<unistd.h>
};


using namespace std;

#include"Debugger.h"
#include"Input_position.h"
#include"Macro_definition.h"
#include"Macro_input.h"
#include"Macro_output.h"
#include"Macro_output_file.h"
#include"Macro_output_memory.h"
#include"Macro_set.h"
#include"Token.h"
#include"define.h"
#include"global.h"
#include"primitive.h"


//
// functions
//
string get_macro_param(Macro_input* input);
istream* macro(Macro_input* input, const string& name,
	       Input_position macro_position);
void main_loop(Macro_input* input, Macro_output* output);
static istream* primitive_define(const vector<string>& param_list);
int main(int argc, char *argv[]);

//
// global variables
//
Special_char	special_char;
string		current_module(":user");
Debugger*	debugger;
Macro_set	macro_set;



//
// function: get_macro_param
//
string
get_macro_param(Macro_input* input)
{
	string result;
	Input_position position = input->position();
	Token token = input->get_token();
	if (token.is_string()) {
		result = token.str();
	} else if (token.is_macro()) {
		debugger->in(position, "macro " + token.str());
		istream* expanded = macro(input, token.str(), position);
		expanded = debugger->out(expanded);

		position.name = "expanded by" + token.str();
		position.is = expanded;
		position.row = position.column = 1;
		position.pos = expanded->tellg();
		debugger->in_sub(position, "expanded by " + token.str());
		Macro_input mi;
		mi.push(expanded, "expanded by " + token.str());
		Macro_output_memory mo;
		main_loop(&mi, &mo);
		result = mo.str();
		// debugger->out();
	} else {
		cerr << "bad parameter" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	// skip spaces after parameter
	input->skip_spaces();
	return result;
}

//
// function: macro
//
istream*
macro(Macro_input* input, const string& name, Input_position macro_position)
{
	Macro_definition m = macro_set.get_definition(name);
	int param_num = m.parameter_num();
	vector<string> param_list;
	input->skip_spaces();
	for (int i = 1; i <= param_num; i++) {
		ostringstream s; s << "parameter " << i;
		debugger->in(input->position(), s.str());
		string param = get_macro_param(input);
		debugger->out(param);
		param_list.push_back(param);
	}

	debugger->in_noninteractive(macro_position, "expand " + name);
	istream* result = m.call(param_list);
	debugger->out();

	return result;
}



//
// function: main_loop
//
void
main_loop(Macro_input* input, Macro_output* output)
{
	for (;;) {
		Token token;
		Input_position position;

		// normal string
		for (;;) {
			debugger->clean_released();
			position = input->position();
			token = input->get_token();
			if (! token.is_string())
				break;
			output->out(token.str());
			debugger->put(token.str());
		}
		if (token.is_end())
			break;
		assert(token.is_macro());

		// macro
		debugger->in(position, "macro " + token.str());
		istream* expanded = macro(input, token.str(), position);
		expanded = debugger->out(expanded);

		position.name = "expanded by" + token.str();
		position.is = expanded;
		position.row = position.column = 1;
		position.pos = position.is->tellg();
		debugger->in_expanded(position, "expanded by " + token.str());
		input->push(expanded, "expanded by " + token.str());
	}
}


static
void
set_special_char(char macro_ch, char param_ch,
		 char open_ch, char close_ch,
		 char module_ch)
{
	special_char.macro = macro_ch;
	special_char.param = param_ch;
	special_char.open_brace = open_ch;
	special_char.close_brace = close_ch;
	special_char.module = module_ch;
}

int
main(int argc, char **argv)
{
	ostream* os = NULL;
	bool debug_mode = false;
	char ch;
	while ((ch = getopt(argc, argv, "dD:o:")) != -1)
		switch (ch) {
		case 'd':	// debug mode
			debug_mode = true;
			break;
		case 'o':
			if (os != NULL) {
				cerr << "multiple output file!" << endl;
				exit(EXIT_FAILURE);
			}
			os = new ofstream(optarg);
			if (os->fail()) {
				cerr << "file \"" << optarg <<
				     "\" could not open" << endl;
				exit(EXIT_FAILURE);
			}
			break;
		case 'D':	// define macro
			cerr << "-D is not implemented" << endl;
			break;
		case '?':
		default:
			cerr << "option error" << endl;
			exit(EXIT_FAILURE);
		}
	argc -= optind;
	argv += optind;

	if (os == NULL)
		os = &cout;


	// init
	current_module = ":user";
	set_special_char('%', '$', '{', '}', ':');
	debugger = new Debugger(debug_mode);
	primitive_macro();
	Macro_output_file mo(os);
	Macro_input mi;

	// input files
	if (argc == 0) {
		cerr << "more than one file is needed" << endl;
		exit(EXIT_FAILURE);
	} else {
		for (int i = argc - 1; i >= 0; i--) { 
			ifstream *is = new ifstream(argv[i]);
			if (is->fail()) {
				cerr << "cannot open \"" << argv[i] << 
				     "\"" << endl;
				exit(EXIT_FAILURE);
			}
			Input_position pos;
			pos.name = string("file \"") + argv[i] + "\"";
			pos.row = pos.column = 1;
			pos.is = is;
			pos.pos = pos.is->tellg();
			debugger->in_sub(pos, pos.name);
			mi.push(is, pos.name);
		}
	}

	main_loop(&mi, &mo);

	delete debugger;
	if (os != &cout)
		delete os;

	return 0;
}

