/*
 * Copyright (c) 2004, SHIRAMOTO Takeyuki
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: primitive.cc,v 1.7 2004/05/17 07:31:52 siramoto Exp $
 */


#include<cassert>
#include<cstdio>
#include<cstdlib>
#include<exception>
#include<fstream>
#include<iosfwd>
#include<iostream>
#include<sstream>
#include<stack>
#include<string>
#include<vector>

using namespace std;

extern "C" {
#include<unistd.h>
};

#include"define.h"
#include"global.h"
#include"primitive.h"
#include"Macro_definition.h"
#include"Macro_set.h"
#include"Macro_output_file.h"



// primitive: :sys:if/3
// primitive: :sys:eq/2
// primitive: :sys:define/2
// primitive: :sys:globaldefine/2
// primitive: :sys:alias/2
// primitive: :sys:begin/0
// primitive: :sys:end/0
// primitive: :sys:module/1
// primitive: :sys:pushmodule/1
// primitive: :sys:popmodule/0
// primitive: :sys:currentmodule/0
// primitive: :sys:input/1
// primitive: :sys:fileout/2
// primitive: :sys:fileappend/2
// primitive: :unix:system/1
// primitive: :unix:command/1

//
// primitive: :sys:if/3
//
static
istream*
primitive_sys_if(const vector<string>& param_list)
{
	return new istringstream((param_list[0] == "t")?
				 param_list[1] : param_list[2]);
}

//
// primitive: :sys:eq/2
//
static
istream*
primitive_sys_eq(const vector<string>& param_list)
{
	return new istringstream((param_list[0] == param_list[1])? "t": "f");
}

//
// primitive: :sys:define/2
//
static
string
define_name(string name)
{
	for (string::const_iterator p = name.begin();
	     p != name.end();
	     p ++)
	{
		if (! (isalnum(*p) || *p == '_' || *p & 0x80 ||
		       *p == special_char.macro || *p == special_char.module))
		{
			cerr << "Illegal macro name" << endl;
			debugger->error();
			/* NOTREACHED */
		}
	}

	ostringstream name_s;
	while (name[0] == special_char.macro)
		name.erase(name.begin());
	if (name[0] != special_char.module)
		name_s << current_module << special_char.module;
	name_s << name;
	return name_s.str();
}

static
int
define_param_num(const string& def)
{
	int max_param = 0;
	string::const_iterator p = def.begin();
	while (p != def.end()) {
		if (*p == special_char.param) {
			p ++;
			assert(p != def.end());
			if (isdigit(*p) && *p != '0') {
				if (*p - '0' > max_param)
					max_param = *p - '0';
			}
		}
		p ++;
	}
	return max_param;
}

static
istream*
primitive_sys_define(const vector<string>& param_list)
{
	string name = define_name(param_list[0]);
	int param_num = define_param_num(param_list[1]);
	macro_set.define(name, Macro_definition(param_num, param_list[1]));
	return new istringstream("");
}


//
// primitive: :sys:globaldefine/2
//
static
istream*
primitive_sys_globaldefine(const vector<string>& param_list)
{
	string name = define_name(param_list[0]);
	int param_num = define_param_num(param_list[1]);
	macro_set.global_define(name,
			        Macro_definition(param_num, param_list[1]));
	return new istringstream("");
}

//
// primitive: :sys:undefine/1
//
static
istream*
primitive_sys_undefine(const vector<string>& param_list)
{
	string name = define_name(param_list[0]);
	macro_set.undefine(name);
	return new istringstream("");
}

//
// primitive: :sys:alias/2
//
static
istream*
primitive_sys_alias(const vector<string>& param_list)
{
	ostringstream name_s;
	name_s << current_module << special_char.module << param_list[1];
	string name(name_s.str());
	const Macro_definition src = macro_set.get_definition(param_list[0]);
	macro_set.define(name, src);

	return new istringstream("");
}

//
// primitive: :sys:begin/0
//
static
istream*
primitive_sys_begin(const vector<string>& param_list)
{
	macro_set.scope_in();
	return new istringstream("");
}

//
// primitive: :sys:end/0
//
static
istream*
primitive_sys_end(const vector<string>& param_list)
{
	macro_set.scope_out();
	return new istringstream("");
}

//
// primitive: :sys:module/1
//
static
istream*
primitive_sys_module(const vector<string>& param_list)
{
	if (param_list[0][0] == special_char.module)
		current_module = param_list[0];
	else {
		ostringstream name_s;
		name_s << current_module << special_char.module <<
		       param_list[0];
		current_module = name_s.str();
	}
	return new  istringstream("");
}


static stack<string> module_stack;
//
// primitive: :sys:pushmodule/1
//
static
istream*
primitive_sys_pushmodule(const vector<string>& param_list)
{
	module_stack.push(current_module);
	if (param_list[0][0] == special_char.module)
		current_module = param_list[0];
	else {
		ostringstream name_s;
		name_s << current_module << special_char.module <<
		       param_list[0];
		current_module = name_s.str();
	}
	return new istringstream("");
}

//
// primitive: :sys:popmodule/0
//
static
istream*
primitive_sys_popmodule(const vector<string>& param_list)
{
	if (module_stack.empty()) {
		cerr << "module stack is empty" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	current_module = module_stack.top();
	module_stack.pop();
	return new istringstream("");
}

//
// primitive: :sys:currentmodule/0
//
static
istream*
primitive_sys_currentmodule(const vector<string>& param_list)
{
	ostringstream s;
	s << current_module << special_char.module;
	return new istringstream(s.str());
}




//
// primitive: :sys:input/1
//
static
istream*
primitive_sys_input(const vector<string>& param_list)
{
	ifstream* is = new ifstream(param_list[0].c_str());
	if (is->fail()) {
		cerr << "file \"" << param_list[0] << "\" cannot open" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	return is;
}

//
// primitive: :sys:fileout/2
//
static
istream*
primitive_sys_fileout(const vector<string>& param_list)
{
	ofstream os(param_list[0].c_str());
	if (os.fail()) {
		cerr << "file \"" << param_list[0] << "\" cannot open" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	Macro_output_file of(&os);
	of.out(param_list[1]);
	return new istringstream("");
}

//
// primitive: :sys:fileappend/2
//
static
istream*
primitive_sys_fileappend(const vector<string>& param_list)
{
	ofstream os(param_list[0].c_str(), ofstream::app);
	if (os.fail()) {
		cerr << "file \"" << param_list[0] << "\" cannot open" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	Macro_output_file of(&os);
	of.out(param_list[1]);
	return new istringstream("");
}

//
// primitive: :sys:message/1
//
static
istream*
primitive_sys_message(const vector<string>& param_list)
{
	cerr << "\n" << param_list[0] << endl;
	return new istringstream("");
}

//
// primitive: :sys:defined/1
//
static
istream*
primitive_sys_defined(const vector<string>& param_list)
{
	string name = define_name(param_list[0]);
	if (macro_set.is_defined(name))
		return new istringstream("t");
	else
		return new istringstream("f");
}


//
// primitive: :unix:system/1
//
static
istream*
primitive_unix_system(const vector<string>& param_list)
{
	if (system(NULL) == 0) {
		cerr << "system sh(1) not available" << endl;
		debugger->error();
		/* NOTREACHED */
	}

	string cmd;
	string::const_iterator p = param_list[0].begin();
	while (p != param_list[0].end()) {
		if (*p == special_char.macro)
			p ++;
		cmd += *p++;
	}

	int state = system(cmd.c_str());
	return new istringstream((state == 0)? "t": "f");
}

//
// primitive: :unix:command/1
//
static
istream*
primitive_unix_command(const vector<string>& param_list)
{
	string cmd;
	string::const_iterator p = param_list[0].begin();
	while (p != param_list[0].end()) {
		if (*p == special_char.macro)
			p ++;
		cmd += *p++;
	}

	FILE* fp = popen(cmd.c_str(), "r");
	if (fp == NULL) {
		cerr << "could not exec " << cmd << endl;
		debugger->error();
		/* NOTREACHED */
	}
	int ch;
	string s;
	while ((ch = fgetc(fp)) != EOF) {
		if (ch == special_char.macro ||
		    ch == special_char.open_brace ||
		    ch == special_char.close_brace ||
		    ch == special_char.param)
		{
			s += special_char.macro;
		}
		s += ch;
	}
	if (pclose(fp) != 0) {
		cerr << "command execuiton failed" << endl;
		debugger->error();
		/* NOTREACHED */
	}
	if (! s.empty() && s[s.size() - 1] == '\n')
		s.erase(s.end() - 1);

	return new istringstream(s);
}






void
primitive_macro()
{
#define DEF_SYS_PRIM(name, param) \
	macro_set.define(":sys:"#name,\
			 Macro_definition(param, primitive_sys_##name))

	DEF_SYS_PRIM(define,	2);
	DEF_SYS_PRIM(globaldefine,	2);
	DEF_SYS_PRIM(undefine,	1);
	DEF_SYS_PRIM(if,	3);
	DEF_SYS_PRIM(eq,	2);
	DEF_SYS_PRIM(alias,	2);
	DEF_SYS_PRIM(begin,	0);
	DEF_SYS_PRIM(end,	0);
	DEF_SYS_PRIM(module,	1);
	DEF_SYS_PRIM(pushmodule,1);
	DEF_SYS_PRIM(popmodule,	0);
	DEF_SYS_PRIM(currentmodule,	0);
	DEF_SYS_PRIM(input,	1);
	DEF_SYS_PRIM(fileout,	2);
	DEF_SYS_PRIM(fileappend,2);
	DEF_SYS_PRIM(message,	1);
	DEF_SYS_PRIM(defined,	1);

#define DEF_UNIX_PRIM(name, param) \
	macro_set.define(":unix:"#name,\
			 Macro_definition(param, primitive_unix_##name))

	DEF_UNIX_PRIM(system,	1);
	DEF_UNIX_PRIM(command,	1);

}


