/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java
 ||
 || It is assumed that a Tamino database called "mydb" has
 || been created and is running.
 || For storing data the default collection ino:etc is used
 || which should only be used for examples/test code.
 || We do not recommend this for real applications.
 ||
 || The following tasks are performed:
 ||
 || - establish a connection to the Tamino database
 || - obtain an XML accessor
 || - start a local transaction mode and
 ||   o retrieve first object stored
 ||   o update object just inserted
 ||   o on success commit transaction otherwise roll back
 || - switch back to auto-commit mode
 || - query updated document
 || - delete document
 || - close the connection
 */
package com.softwareag.tamino.db.api.examples.greeting;

import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TDeleteException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.jdom.TJDOMObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.StringReader;
import java.io.StringWriter;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;


public class ProcessXMLGreeting {
	
	public ProcessXMLGreeting(String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Obtain a TXMLObjectAccessor with a JDOM object model
		accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) ,
												   TJDOMObjectModel.getInstance() );
	}
	
	private void performInsert(TXMLObject xmlObject) throws TException  {
		// TResponse represents an access response from Tamino
		TResponse response = null;
		// Insert the XML document
		try  {
			// Invoke the insert operation and obtain the response
			response = accessor.insert( xmlObject );
			// Show the collection, doctype and id
			System.out.println( "Insert succeeded, ino:collection:" + xmlObject.getCollection() +
							   ", ino:doctype:" + xmlObject.getDoctype() +
							   ", ino:id:" + xmlObject.getId() );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
			// Rethrow the exception
			throw insertException;
		}
	}
	
	private void performUpdate(TQuery query,String updatedText) throws TException  {
		TLocalTransaction localTransaction = null;
		// Perform query and update operations within a transaction
		try  {
			// Switch to local transaction mode
			localTransaction = connection.useLocalTransactionMode();
			// Invoke the query to obtain the document and to lock it
			TResponse response = accessor.query( query );
			// Obtain the TXMLObject from the response
			TXMLObject xmlObject = response.getFirstXMLObject();
			if ( xmlObject == null )
				return;
			// Obtain the JDOM element and update its content
			Element element = (Element)xmlObject.getElement();
			element.setText( updatedText );
			// Invoke the update
			response = accessor.update( xmlObject );
			System.out.println( "Update succeeded!" );
			// Commit the transaction
			localTransaction.commit();
		}
		// TQueryException and TUpdateException are both derived from TAccessorException
		// so we simply use the base class here
		catch (TAccessorException accessorException)  {
			showAccessFailure( accessorException );
			localTransaction.rollback();
			throw accessorException;
		}
		finally  {
			// Switch back to auto-commit mode
			connection.useAutoCommitMode();
		}
	}
	
	private void performQuery(TQuery query) throws TException  {
		// Now we query the updated object
		try  {
			TResponse response = accessor.query( query );
			TXMLObject xmlObject = response.getFirstXMLObject();
			// Write the XML content into a StringWriter
			StringWriter stringWriter = new StringWriter();
			xmlObject.writeTo( stringWriter );
			System.out.println( "Queried document:" + stringWriter );
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	private void performDelete(TQuery query) throws TException  {
		// Finally, we delete the document again
		try  {
			TResponse response = accessor.delete( query );
			System.out.println( "Deleted the document!" );
		}
		catch (TDeleteException deleteException)  {
			showAccessFailure( deleteException );
			throw deleteException;
		}
	}
	
	// Build a JDOM Element from the given XML
	private Element toJDOM(String xml) throws JDOMException  {
		SAXBuilder saxBuilder = new SAXBuilder();
		Document document = saxBuilder.build( new StringReader( xml ) );
		return document.getRootElement();
	}
	
	// Inform about the reason for the failure
	private void showAccessFailure(TAccessorException accessorException)  {
		// Obtain an access failure message with the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			System.out.println( "Access failed:" + accessFailure );
		else
			System.out.println( "Access failed:" + accessorException.getMessage() );
	}
	
	private void show()  throws TException {
		try  {
			// Instantiate a TXMLObject with the JDOM object model
			TXMLObject xmlObject = TXMLObject.newInstance( toJDOM( XML ) );
			// Initiate the insert
			performInsert( xmlObject );
			// Construct a query expression so that the inserted document can be referenced
			TQuery query = TQuery.newInstance( xmlObject.getDoctype() + "[@ino:id=\"" +
											  xmlObject.getId() + "\"]" );
			// Initiate the update
			performUpdate( query , "Hello World, updated :-)" );
			// Initiate the query
			performQuery( query );
			// Initiate the removal
			performDelete( query );
		}
		catch (JDOMException jdomException)  {
			jdomException.printStackTrace();
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close the connection
			connection.close();
		}
	}
	
	public static void main(String[] args) throws TException  {
		// Use ino:etc collection in the specified Tamino database
		ProcessXMLGreeting processXMLGreeting = new ProcessXMLGreeting( DATABASE_URI , "ino:etc" );
		processXMLGreeting.show();
	}
	
	// URI of the Tamino database, please edit accordingly
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
	
	// XML document to be written to the connected database
	private final static String XML = "<Greeting by='ProcessGreetingApplication'>Hello World</Greeting>";
	
	// The database connection
	private TConnection connection = null;
	
	// The accessor, here a high-level TXMLObjectAccessor
	private TXMLObjectAccessor accessor = null;
	
}
