/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 * InsertUniqueDOM4J.java
 */
package com.softwareag.tamino.db.api.examples.jazz;


import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TIsolationLevel;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.dom4j.TDOM4JObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.FileInputStream;
import java.util.List;
import org.dom4j.Document;
import org.dom4j.Element;


public class InsertUniqueDOM4J {
	
	/** Creates new InsertUniqueDOM4J */
	public InsertUniqueDOM4J (String databaseURI,String collection) throws TConnectionException {
		
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the first connection to the database
		connectionA = connectionFactory.newConnection( databaseURI );
		// Instantiate the specific TDOM4JObjectModel
		TXMLObjectModel dom4jObjectModelA = TDOM4JObjectModel.getInstance();
		// Do the object model registration.
		TXMLObjectModel.register( dom4jObjectModelA );
		
		// Obtain the concrete TXMLObjectAccessor with an underyling DOM4J object model
		accessorA = connectionA.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) ,dom4jObjectModelA  );
		// Set local transaction mode to "shared"
		connectionA.setIsolationLevel(TIsolationLevel.SHARED) ;
		// Instantiate the specific TDOM4JObjectModel
		TXMLObjectModel dom4jObjectModelB = TDOM4JObjectModel.getInstance();
		// Do the object model registration.
		TXMLObjectModel.register( dom4jObjectModelB );
		// Obtain the second connection to the database
		connectionB = connectionFactory.newConnection( databaseURI );
		// Obtain the second accessor
		accessorB = connectionB.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) , dom4jObjectModelB );
		// Set isolation level of connection 2 to "unprotected"
		connectionB.setIsolationLevel(TIsolationLevel.UNPROTECTED) ;
	}
	
	private void performInsert(TXMLObject xmlObject) throws Exception  {
		try  {
			// Invoke the insert operation
			accessorA.insert( xmlObject );
		}
		catch (TInsertException insertException)	{
			showAccessFailure( insertException );
		}
	}
	
	// get number of matching documents
	private int getCount(String path) throws Exception	{
		try  {
			// Construct TQuery object
			TQuery query = TQuery.newInstance("count("+path+")");
			// perform the query
			TResponse response = accessorB.query(query );
			// get the number of documents found
			String s = response.getQueryContentAsString();
			// convert to integer
			return Integer.valueOf(s).intValue();
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			return 0;
		}
	}
	
	// Show the reason for the access failure.
	private void showAccessFailure(TAccessorException accessorException) throws Exception {
		// Obtain an access failure message telling the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			throw new Exception( "Access failed:" + accessFailure );
		else
			throw new Exception( "Access failed:" + accessorException.getMessage() );
	}
	
	
	private void processTransaction(String filename, String key) throws Exception {
		TLocalTransaction myTransaction = null;
		// Read file into a DOM4J Tamino XML object.
		// Instantiate an empty TXMLObject instance related to the DOM4J object model.
		TXMLObject xmlObject = TXMLObject.newInstance( TDOM4JObjectModel.getInstance() );
		// Establish the DOM4J representation by reading the content from a file input stream.
		xmlObject.readFrom( new FileInputStream(filename ));
		// get DOM4J document
		Document doc = (Document) xmlObject.getDocument();
		// get top level element
		Element root = (Element) xmlObject.getElement();
		// get key value
		String keyValue = null;
		// check if key is an attribute or an element
		if (key.startsWith("@")) {
			// get attribute value
			keyValue = root.attributeValue(key.substring(1));
		} else {
			// get element node list
			List list = doc.getRootElement().elements(key);
			if (list.isEmpty()) throw new Exception("Key not found");
			// get only element
			Element elem = (Element) list.get(0);
			// get element content
			keyValue = elem.getText();
		}
		// Check for proper content
		if (keyValue == "") throw new Exception("Key not found");
		// Start the transaction
		myTransaction = connectionA.useLocalTransactionMode();
		// Insert the document
		performInsert( xmlObject );
		// Get number of matching documents
		int c = getCount( xmlObject.getDoctype() + "["+key+"='" + keyValue + "']" );
		if (c == 1) {
			// Unique - commit the transaction
			myTransaction.commit();
			System.out.println("Transaction committed");
		} else {
			// Bad - rollback the transaction
			myTransaction.rollback();
			throw new Exception("Key not unique: "+c+" occurrences. Transaction aborted.");
		}
		connectionA.close();
		connectionB.close();
	}
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(String[] args) throws Exception  {
		long startTime = System.currentTimeMillis();
		try{
			System.out.println("Start InsertUniqueDOM4J ...");
			InsertUniqueDOM4J insertUnique = new InsertUniqueDOM4J( DATABASE_URI , COLLECTION );
			insertUnique.processTransaction(args[0],args[1]);
		} catch (Throwable thr) {
			thr.printStackTrace();
		}
		System.out.println("InsertUniqueDOM4J needs "+(System.currentTimeMillis() - startTime)+" milliSeconds");
	}
	
	// Constant for the database URI.
	private final static String DATABASE_URI = "http://localhost/tamino/jazz";
	
	// Constant for the collection.
	private final static String COLLECTION = "encyclopedia";
	
	// Database connection A
	private TConnection connectionA = null;
	
	// Accessor A
	private TXMLObjectAccessor accessorA = null;
	
	// Database connection B
	private TConnection connectionB = null;
	
	// Accessor B
	private TXMLObjectAccessor accessorB = null;
}

