/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 || Example for the Tamino API for Java.
 ||
 || Assumes that there is a Tamino database called "mydb".
 || The example simply uses the default collection ino:etc.
 || This collection should only be used for examples/test code.
 || It shouldn't generally be used for real applications.
 ||
 || The example does the following:
 ||
 || establishes a connection to the Tamino database
 || establishes a sax specific object model
 || registers this object model
 || obtains an XML accessor for the sax object model
 || inserts a new document of <Message>...</Message> into ino:etc
 || starts a local transaction mode
 || queries for the newly inserted document
 || updates the newly inserted document
 || commits the transaction if succeeded otherwise performs a rollback
 || switches back to auto-commit mode
 || queries for the updated document
 || Finally the document is deleted
 || closes the connection
 */
package com.softwareag.tamino.db.api.examples.message;

import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TDeleteException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.TXMLObjectModel;
import com.softwareag.tamino.db.api.objectModel.sax.TSAXObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.StringWriter;


public class ProcessMessage {
	
	public ProcessMessage(String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Instantiate the default handler that processes the sax events
		messageDefaultHandler = new MessageDefaultHandler();
		// Instantiate the document and element event handlers each of which delegates its events to the
		// messageDefaultHandler
		DocumentDefaultHandler docDefHandler = new DocumentDefaultHandler( messageDefaultHandler );
		ElementDefaultHandler elDefHandler = new ElementDefaultHandler( messageDefaultHandler );
		// Instantiate the specific TSAXObjectModel
		TSAXObjectModel saxObjectModel = new TSAXObjectModel( "MessageSAXObjectModel" , Message.class , Message.class ,	docDefHandler , elDefHandler );
		// Do the object model registration.
		TXMLObjectModel.register( saxObjectModel );
		// Obtain the concrete TXMLObjectAccessor with an underyling JDOM object model
		accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) , saxObjectModel );
	}
	
	private void performInsert(TXMLObject xmlObject) throws TException  {
		// TResponse represents an access response from Tamino.
		TResponse response = null;
		// Do the insert document
		try  {
			StringWriter stw = new StringWriter();
			xmlObject.writeTo( stw );
			System.out.println( "Going to insert " + stw );
			// Invoke the insert operation and obtain the response.
			response = accessor.insert( xmlObject );
			// Show the collection, doctype and id
			System.out.println( "Insert succeeded, ino:collection:" + xmlObject.getCollection() + ", ino:doctype:" + xmlObject.getDoctype() +
							   ", ino:id:" + xmlObject.getId() );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
			// Rethrow the exception
			throw insertException;
		}
	}
	
	private void performUpdate(TQuery query,String updatedText) throws TException  {
		TLocalTransaction localTransaction = null;
		// Do the query and update within a transaction
		try  {
			// Switch to the local transaction mode
			localTransaction = connection.useLocalTransactionMode();
			// Invoke the query to obtain the document and to lock it
			TResponse response = accessor.query( query );
			// Obtain the TXMLObject from the response
			TXMLObject xmlObject = response.getFirstXMLObject();
			if ( xmlObject == null )
				return;
			// Obtain the Message element and clear and update its content
			Message element = (Message)xmlObject.getElement();
			element.clearContent();
			element.appendContent( updatedText );
			// Invoke the update
			response = accessor.update( xmlObject );
			System.out.println( "Update succeeded!" );
			// Commit the transaction
			localTransaction.commit();
		}
		// TQueryException and TUpdateException are both derived from TAccessorException so we simply use the base class here
		catch (TAccessorException accessorException)  {
			showAccessFailure( accessorException );
			localTransaction.rollback();
			throw accessorException;
		}
		finally  {
			// Switch back to the auto commit mode
			connection.useAutoCommitMode();
		}
	}
	
	private void performQuery(TQuery query) throws TException  {
		// Now lets make a query on the updated object
		try  {
			TResponse response = accessor.query( query );
			TXMLObject xmlObject = response.getFirstXMLObject();
			// Write the XML content to a StringWriter
			StringWriter stringWriter = new StringWriter();
			xmlObject.writeTo( stringWriter );
			System.out.println( "Queried document:" + stringWriter );
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			throw queryException;
		}
	}
	
	private void performDelete(TQuery query) throws TException  {
		// Finally, lets delete the document again
		try  {
			TResponse response = accessor.delete( query );
			System.out.println( "Deleted the document!" );
		}
		catch (TDeleteException deleteException)  {
			showAccessFailure( deleteException );
			throw deleteException;
		}
	}
	
	// Show the reason for the access failure.
	private void showAccessFailure(TAccessorException accessorException)  {
		// Obtain an access failure message telling the exact reason if Tamino request failed.
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			System.out.println( "Access failed:" + accessFailure );
		else
			System.out.println( "Access failed:" + accessorException.getMessage() );
	}
	
	private void show()  throws TException {
		try  {
			Message message = new Message();
			message.appendContent( "Hello World" );
			// Instantiate a TXMLObject with an underyling SAX based object model
			TXMLObject xmlObject = TXMLObject.newInstance( message );
			// Initiate the insert.
			performInsert( xmlObject );
			// Build a query so that the inserted document can be referenced
			TQuery query = TQuery.newInstance( xmlObject.getDoctype() + "[@ino:id=\"" + xmlObject.getId() + "\"]" );
			// Initiate the update
			performUpdate( query , "Hello World, updated :-)" );
			messageDefaultHandler.reset();
			// Initiate the query
			performQuery( query );
			// Initiate the removal
			performDelete( query );
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close the connection.
			connection.close();
		}
	}
	
	public static void main(String[] args) throws TException  {
		// Change the database uri here to work with an appropiate one.
		ProcessMessage processMessage = new ProcessMessage( DATABASE_URI , "ino:etc" );
		processMessage.show();
	}
	
	// Constant for the database URI. Please edit to use your uri of interest.
	private final static String DATABASE_URI = "http://localhost/tamino/mydb";
	
	// The database connection.
	private TConnection connection = null;
	
	// The message default handler for all the sax events.
	private MessageDefaultHandler messageDefaultHandler = null;
	
	// The concrete accessor, here a high level TXMLObjectAccessor.
	private TXMLObjectAccessor accessor = null;
	
}

