/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.ext.LexicalHandler;
import java.util.ArrayList;
import java.io.IOException;

/**
 * A fast and memory efficient FONIDocument implementation for parsed documents.
 *  Once built, it is readonly and thread safe.
 */
public class IDocument extends FONIWriter implements FONIDocument
{
    public IDocument( ) {
	blocks = new int[8][];
	blocks[0] = new int[BLOCK_SIZE];
	blockCnt = 1;
	charBlocks = new char[8][];
	charBlocks[0] = new char[CHARBLOCK_SIZE];
	lastCharBlock = 0;
	charBlockPtr = 1;
	docSize = CONTENT_OFFSET;	// dummy node at 0
	openNode(Node.DOCUMENT);
	curNode = CONTENT_OFFSET;
    }

    // -------- FONIDocument interface: -------------

    public String getBaseURI() {
	return baseURI;
    }

    public int   getRootNode() {
	return CONTENT_OFFSET;
    }

    public void setDocId(int value) {
        docId = value;
    }

    public int getDocId() {
        return docId;
    }

    public QName getName( int nodeId ) {
	switch(getKind(nodeId)) {
	    case Node.ELEMENT:
		return elementNames.getName(getNameId(nodeId));
	    case Node.PROCESSING_INSTRUCTION:
		return otherNames.getName(getNameId(nodeId));
	    default:
		return null;
	}
    }

    public int getNameId( int offset ) {
	return hNameId( dataAt(offset) );
    }

    public QName  pnGetName( int/*NID*/ nodeId ) {
	return otherNames.getName(getNameId(nodeId));
    }

    public int  pnGetNameId( int/*NID*/ nodeId ) {
	return getNameId(nodeId);
    }

    public int  getParent( int/*NID*/ nodeId ) {
	return dataAt( nodeId + PARENT_OFFSET );
    }

    public int  getNextSibling( int/*NID*/ nodeId ) {
	return dataAt( nodeId + NEXT_OFFSET );
    }

    public int  getNodeNext (int/*NID*/ nodeId) {	// in document order
	int/*NID*/ kid = getFirstChild( nodeId );
	return (kid != 0)? kid : getNodeAfter( nodeId );
    }

    public int  getNodeAfter( int/*NID*/ nodeId ) {
	int/*NID*/ nxt;
	while((nxt = getNextSibling(nodeId)) == 0) {
	    int/*NID*/ parent = getParent(nodeId);
	    if (parent == 0)
		return 0;
	    nodeId = parent;
	}
	return nxt;
    }

    public int/*NID*/  getNodeSpan( int/*NID*/ nodeId ) {
	int/*NID*/ nxt = getNodeNext(nodeId);
	if (nxt == 0)
	    nxt = docSize;
	return nxt - nodeId;
    }

    public int/*NID*/  getFirstChild( int/*NID*/ nodeId ) {
	int header = dataAt(nodeId), kind = header & KIND_MASK;
	if (kind != Node.ELEMENT && kind != Node.DOCUMENT ||
	    (header & HAS_CHILDREN) == 0)
	    return 0;
	int/*NID*/ kid = nodeId +
	                 ATTR_OFFSET + 2 * (hAttrCount(header) + hNamespaceCount(header));
	return kid;
    }

    public NodeIterator childrenIterator(int/*NID*/ id) {
	return new SiblingIterator( - getFirstChild(id) );
    }


    public int getAttrCount( int/*NID*/ nodeId ) {
	return hAttrCount(dataAt(nodeId + HEADER_OFFSET));
    }

    public NodeIterator attrIterator(int/*NID*/ id) {
	return new AttrIterator(id);
    }

    public int/*NID*/   getAttribute( int/*NID*/ nodeId, int nameId ) {
	int/*NID*/ off = nodeId + ATTR_OFFSET + 2 * getNamespaceCount(nodeId);
	for(int cnt = getAttrCount(nodeId); cnt > 0; --cnt, off += 2)
	    if(getNameId(off) == nameId)
		return off;
	return 0;
    }

    public String getStringValue( int/*NID*/ nodeId ) {
	switch(getKind(nodeId)) {
	case Node.ELEMENT:
	case Node.DOCUMENT:
	    StringBuffer sb = new StringBuffer(getNodeSpan(nodeId));
	    for(int/*NID*/kid= getFirstChild(nodeId); kid != 0; kid=getNextSibling(kid)) {
		recStringValue(kid, sb);
	    }
	    return sb.toString();
	case Node.ATTRIBUTE:
	case Node.NAMESPACE:
	    return decodeString( dataAt(nodeId + 1) );
	default:
	    return decodeString( dataAt(nodeId + CONTENT_OFFSET) );
	}
    }

    void recStringValue(int/*NID*/ nodeId, StringBuffer sb) {
	switch(getKind(nodeId)) {
	case Node.ELEMENT:
	case Node.DOCUMENT:
	    for(int/*NID*/kid= getFirstChild(nodeId); kid != 0; kid=getNextSibling(kid)) {
		recStringValue(kid, sb);
	    }
	    break;
	case Node.TEXT:
	    decodeString( dataAt(nodeId + CONTENT_OFFSET), sb );
	    break;
	}
    }
    /**
     *	Gets the string value for pseudo-nodes Attributes and Namespaces.
     */
    public String pnGetStringValue( int/*NID*/ nodeId ) {
	return decodeString( dataAt(nodeId + 1) );
    }

    public char[]  getCharValue( int/*NID*/ nodeId, int reserve ) {
	switch(getKind(nodeId)) {
	case Node.ELEMENT:
	case Node.DOCUMENT:
	    throw new RuntimeException("not allowed on non-atoms");
	case Node.ATTRIBUTE:
	case Node.NAMESPACE:
	    return decodeChars( dataAt(nodeId + 1), reserve );
	default:
	    return decodeChars( dataAt(nodeId + CONTENT_OFFSET), reserve );
	}
    }

    public char[]  pnGetCharValue( int/*NID*/ nodeId, int reserve ) {
	return decodeChars( dataAt(nodeId + 1), reserve );
    }

    // dummy
    public Object getValue( int/*NID*/ nodeId ) { return null;  }
    public long getIntegerValue( int/*NID*/ nodeId ) { return -1; }

    public int getDefinedNSCount( int/*NID*/ nodeId ) {
	return getNamespaceCount(nodeId);
    }

    public NodeIterator namespaceIterator( int/*NID*/ nodeId, boolean inScope ) {
	if(inScope)
	    // this is correct because on each node that defines namespaces,
	    // there is a merged mapping
	    while(nodeId != 0 && getNamespaceCount(nodeId) == 0)
		nodeId = getParent(nodeId);
	return new NSIterator(nodeId);
    }

    public int  getElementNameCount( ) {
	return elementNames.size();
    }

    public QName  getElementName( int nameId ) {
	return elementNames.getName(nameId);
    }

    public int    internElementName( QName name ) {
	return elementNames.find(name);
    }

    public int  getOtherNameCount( ) {
	return otherNames.size();
    }

    public QName  getOtherName( int nameId ) {
	return otherNames.getName(nameId);
    }

    public int    internOtherName( QName name ) {
	return otherNames.find(name);
    }

    /**
     *	
     */
    protected class SiblingIterator implements FONIDocument.NodeIterator {
	SiblingIterator( int/*NID*/ startId ) {
	    this.startId = this.curId = startId;
	}

	public boolean next() {
	    if(curId < 0)
		 curId = -curId;	// trick for children iterator
	    else curId = getNextSibling(curId);
	    return curId != 0;
	}

	public int/*NID*/     currentId() {
	    return curId;
	}

	public FONIDocument.NodeIterator reborn(){
	    return new SiblingIterator(startId);
	}

	int/*NID*/ startId, curId;
    }

    /**
     *	
     */
    public class AttrIterator implements FONIDocument.NodeIterator {
	protected int/*NID*/ startId;
	protected int/*NID*/ offset;
	protected int count = 0;

	AttrIterator() { }

	AttrIterator( int/*NID*/ nodeId ) { 
	    startId = nodeId;
	    int header = dataAt( nodeId + HEADER_OFFSET );
	    // negative means at start:
	    offset = - (nodeId + ATTR_OFFSET + 2 * hNamespaceCount(header));
	    count = hAttrCount(header);
	}

	public boolean next() {
	    if(count <= 0)
		return false;
	    -- count;
	    if(offset < 0)
		offset = -offset;
	    else
		offset += 2;
	    return true;
	}

	public int/*NID*/ currentId() {
	    return offset < 0 ? -offset : offset;
	}

	public FONIDocument.NodeIterator reborn(){
	    return new AttrIterator(startId);
	}
    }

    /**
     *	
     */
    protected class NSIterator extends AttrIterator {

	NSIterator( int/*NID*/ nodeId ) { 
	    
	    count = nodeId == 0 ? 0 : getNamespaceCount(nodeId);
	    offset = nodeId + ATTR_OFFSET - 2;
	}

	public boolean next() {
	    if(count <= 0)
		return false;
	    -- count;
	    offset += 2;
	    return true;
	}

	public int/*NID*/ currentId() {
	    return offset;
	}

	public FONIDocument.NodeIterator reborn(){
	    return new NSIterator(startId);
	}
    }

    // ----- SAX2 construction:

    Locator currentLocator;

    public void setDocumentLocator(Locator locator) {
	currentLocator = locator;
    }

    public void close() throws IOException, SAXException {
    }

    public void startDocument () throws SAXException {
	if(currentLocator != null)
	    baseURI = currentLocator.getSystemId();
    }

    public void endDocument() throws SAXException {
	
    }

    public void startElement (String namespaceURI, String localName,
			      String qName, Attributes attrs)
	throws SAXException
    {
	endText();
	int/*NID*/ nodeId = openNode(Node.ELEMENT);
	if(localName == null || localName.length() == 0) {
	    namespaceURI = ""; localName = qName;
	}
	int nameId = elementNames.enter( namespaceURI, localName );
	int attrCnt = attrs == null ? 0 : attrs.getLength();
	int nsCnt = 0, stoff = nodeId + CONTENT_OFFSET;
	// namespaces ?
	if(prefixMapping != null) {
	    // merge with upper mapping:
	    if(namespaceStack.size() > 0) {
		ArrayList upmap = (ArrayList) namespaceStack.get(namespaceStack.size()-1);
		for(int u = upmap.size() - 2, pos; u >= 0; u -= 2) {
		    for(pos = prefixMapping.size() - 2; pos >= 0; pos -= 2)
			if(upmap.get(u).equals(prefixMapping.get(pos)))
			    break;
		    if(pos < 0) {
			prefixMapping.add(0, upmap.get(u));
			prefixMapping.add(1, upmap.get(u+1));
		    }
		}
	    }
	    // store:
	    nsCnt = prefixMapping.size();
	    for(int i = 0; i < nsCnt; i += 2) {
		int nsnameId = otherNames.enter("", (String) prefixMapping.get(i));
		setData(stoff ++, Node.NAMESPACE + (nsnameId << NAME_SHIFT));
		setData(stoff ++, storeString( (String) prefixMapping.get(i+1) ));
	    }
	    namespaceStack.add(prefixMapping);
	    prefixMapping = null;
	}
	// store attributes:
	for(int a = 0;  a < attrCnt; a++)
	{
	    int id = otherNames.enter(attrs.getURI(a), attrs.getLocalName(a));
	    setData(stoff ++, Node.ATTRIBUTE + (id << NAME_SHIFT) );
	    setData(stoff ++, storeString(attrs.getValue(a)) );
	}
	docSize = stoff;
	setData( nodeId + HEADER_OFFSET,
		 Node.ELEMENT + (nameId << NAME_SHIFT) +
		 (attrCnt << ATTR_SHIFT) + ((nsCnt/2) << NS_SHIFT) );
    }

    public void endElement (String namespaceURI, String localName, String qName)
	throws SAXException  {
	endText();
	closeNode();
    }

    public void characters (char ch[], int start, int length) {
	int nPtr = charPtr + length;
	if(nPtr > charBuffer.length) {
	    char[] old = charBuffer;
	    charBuffer = new char[nPtr + 1000];
	    System.arraycopy(old, 0, charBuffer, 0, charPtr);
	}
	System.arraycopy(ch, start, charBuffer, charPtr, length);
	charPtr = nPtr;
	++ charChunks; charCnt += length;
    }

    public void characters (String chars) {
	characters(chars.toCharArray(), 0, chars.length());
    }
 
    public void processingInstruction (String target, String data)
	throws SAXException  {
	endText();
	int/*NID*/ node = openNode(Node.PROCESSING_INSTRUCTION);
	int nameId = otherNames.enter("", target);
	setData( node + HEADER_OFFSET,
		 dataAt(node + HEADER_OFFSET) + (nameId << NAME_SHIFT));
	setData( docSize ++, storeString(data));
	closeNode();
    }

    public void skippedEntity (String name)
	throws SAXException
    {
	throw new SAXException("skipped entity '"+name+"': cannot be handled");
    }

    public void startPrefixMapping ( String prefix, String uri )
	throws SAXException
    {
	if(prefixMapping == null)
	    prefixMapping = new ArrayList();
	prefixMapping.add(prefix);
	prefixMapping.add(uri);
    }

    public void endPrefixMapping( String prefix ) 
	throws SAXException {
    }

    public void comment(char[] ch, int start, int length) {
	endText();
	int/*NID*/ node = openNode(Node.COMMENT);
	setData( docSize ++, storeString(new String(ch, start, length)));
	closeNode();
    }

    public void startCDATA() { }
    public void endCDATA() { }
    public void startDTD(String name, String publicId, String systemId) { }
    public void endDTD() { }
    public void startEntity(String name) { }
    public void endEntity(String name) { }

    public void  atom(Object value) throws SAXException {
	System.err.println("IDocument.atom: not implemented");
    }

    // ------------------------

    public int/*NID*/ getCurrentNode() {
	return curNode;
    }

    int/*NID*/ openNode( int impl )
    {
	int/*NID*/ nodeId = docSize;
	docSize += CONTENT_OFFSET;
	if (prevNode != 0)
	    setData( prevNode + NEXT_OFFSET, nodeId );
	setData( nodeId + HEADER_OFFSET, impl );
	setData( nodeId + PARENT_OFFSET, curNode );
	setData( nodeId + NEXT_OFFSET, 0 );
	curNode = lastNode = nodeId;
	prevNode = 0;
	return curNode;
    }

    void  closeNode ()
    {
	prevNode = curNode;
	curNode = getParent(curNode);
	if(curNode != 0)
	    setFlag( curNode, HAS_CHILDREN );
    }

    void  endText() {
	if( charPtr > 0 ) {
	    ++ textChunks;
	    openNode(Node.TEXT);
	    setData( docSize ++, storeString());
	    closeNode();
	}
	//textChunk = null;
	charPtr = 0;
    }

    void setFlag( int/*NID*/ nodeId, int flag ) {
	setData( nodeId + HEADER_OFFSET, dataAt(nodeId + HEADER_OFFSET) | flag );
    }

    void checkDump( int/*NID*/ node, int/*NID*/ parent, int depth ) {
	System.out.print(node);
	for(int d = 0; d < depth; d++) System.out.print("  ");
	++ depth;
	switch(getKind(node)) {
	case Node.ELEMENT:
	case Node.DOCUMENT:
	    System.out.println("ELEMENT "+getName(node));
	    for(AttrIterator attrit = new AttrIterator(node); attrit.next(); ) {
		checkDump(attrit.currentId(), node, depth);
	    }
	    for(int/*NID*/ kid = getFirstChild(node); kid != 0; kid = getNextSibling(kid))
		checkDump(kid, node, depth);
	    break;
	case Node.ATTRIBUTE:
	    System.out.println("  ATTR "+ getName(node) +" |"+getStringValue(node)+"|");
	    break;
	case Node.TEXT:
	    System.out.println("TEXT |"+getStringValue(node)+"|");
	    break;
	case Node.COMMENT:
	    System.out.println("COMMENT |"+getStringValue(node)+"|");
	    break;
	case Node.PROCESSING_INSTRUCTION:
	    System.out.println("PI |"+getStringValue(node)+"|");
	    break;
	default:
	    System.err.println("*** bad node id="+node+" kind="+getKind(node));
	    break;
	}
    }

    // ----- implementation specific:

    final static int HEADER_OFFSET = 0;
    final static int PARENT_OFFSET = 1;
    final static int NEXT_OFFSET = 2;
    final static int ATTR_OFFSET = 3;
    final static int CONTENT_OFFSET = 3;

    final static int KIND_BITS = 3;	// bits for node kind
    final static int KIND_MASK = (1 << KIND_BITS) - 1;
    final static int HAS_CHILDREN = 8;

    final static int NAME_SHIFT = KIND_BITS + 1;	// reserve bits for flags
    final static int NAME_BITS = 12;	// bits for name id
    final static int NAME_MASK = (1 << NAME_BITS) - 1;	// reserve 12 bits for name id

    final static int ATTR_SHIFT = NAME_SHIFT + NAME_BITS;  // 
    final static int ATTR_BITS = 9;  // bits for element attr count
    final static int ATTR_MASK = (1 << ATTR_BITS) - 1; 

    final static int NS_SHIFT = ATTR_SHIFT + 9;    // the rest for NS node count

    final static int ATTR_NAME_SHIFT = 2; // 1 bit reserved in attr and NS pseudonode names

    final static int BLOCK_SHIFT = 9;
    final static int BLOCK_SIZE = 1 << BLOCK_SHIFT;
    final static int BLOCK_MASK = BLOCK_SIZE - 1;

    final static int CHARBLOCK_SHIFT = 12;	// 4K chars. must stay <= 15
    final static int CHARBLOCK_SIZE = 1 << CHARBLOCK_SHIFT;
    final static int CHARBLOCK_MASK = CHARBLOCK_SIZE - 1;

    protected int docId;
 
    private String  baseURI;
    private NSTable elementNames = new NSTable();
    private NSTable otherNames = new NSTable();
    private ArrayList  prefixMapping;	// non-null when the current node has namespace nodes
    private ArrayList  namespaceStack = new ArrayList();

    //int[] data = new int[1024];
    int[][] blocks;	//OLD
    int blockCnt;
    int docSize;
    int/*NID*/ curNode;
    int/*NID*/ prevNode = 0;
    int/*NID*/ lastNode;	// last node open

    int charCnt, charChunks, textChunks;
    // temporary buffer
    char[] charBuffer = new char[CHARBLOCK_SIZE / 2];
    int charPtr;
    // storage blocks for small strings
    char[][] charBlocks;
    int lastCharBlock;	// index of current char block
    int charBlockPtr;	// ptr inside current block
    // Big strings (directly stored in the ArrayList)
    ArrayList bigStrings = new ArrayList();

    public int estimateMemorySize() {
	return blockCnt * BLOCK_SIZE * 4 + lastCharBlock * CHARBLOCK_SIZE * 2
	    + bigStrings.size() * CHARBLOCK_SIZE;	// minimum
    }

    private int dataAt( int/*NID*/ offset ) {
	return blocks[ offset >> BLOCK_SHIFT ] [ offset & BLOCK_MASK ];
	//return data[offset];
    }

    private void setData( int/*NID*/ offset, int value ) {
	/*
	if(offset >= data.length) {
            int[] old = data;
	    int incr = old.length; //Math.min(old.length, 1000000);
            data = new int[ old.length + incr ];
            System.arraycopy(old, 0, data, 0, old.length);
	    System.out.println("alloc "+data.length);
        }
	data[offset] = value;*/
	
	int block = offset >> BLOCK_SHIFT, cell = offset & BLOCK_MASK;
	if(block >= blockCnt) {
	    if(blockCnt >= blocks.length) {
		int[][] old = blocks;
		blocks = new int[old.length + 64][];
		System.arraycopy(old, 0, blocks, 0, old.length);
	    }
	    blocks[ blockCnt ++ ] = new int[BLOCK_SIZE];
	}
	blocks[ block ][ cell ] = value;
	
	
    }

    public int  getKind( int/*NID*/ nodeId ) {
	return dataAt(nodeId + HEADER_OFFSET) & KIND_MASK;
    }

    private final int hNameId( int header ) {
	return (header >> NAME_SHIFT) & NAME_MASK;
    }

    int getNamespaceCount( int/*NID*/ nodeId ) {
	return hNamespaceCount( dataAt(nodeId) );
    }

    private final int hNamespaceCount( int header ) {
	return header >> NS_SHIFT;
    }

    private final int hAttrCount( int header ) {
	return (header >> ATTR_SHIFT) & ATTR_MASK;
    }

    int storeString( String str ) {
	int L = str.length();
	if(L > charBuffer.length)
	    return storeBigString(str);
	str.getChars(0, L, charBuffer, 0);	// big enough
	charPtr = L;
	int result = storeString();
	charPtr = 0;
	return result;
    }

    int storeString( ) {
	if(charPtr > CHARBLOCK_SIZE/2 ) {
	    return storeBigString( new String(charBuffer, 0, charPtr) );
	}
	int rlen = charPtr + 1;	// required length
	if( charBlockPtr + rlen > CHARBLOCK_SIZE) {
	    // spanning blocks is not allowed: allocate a new block
	    if(++ lastCharBlock >= charBlocks.length) {
		char[][] old = charBlocks;
		charBlocks = new char[old.length + 100][];
		System.arraycopy(old, 0, charBlocks, 0, old.length);
	    }
	    charBlocks[lastCharBlock] = new char[CHARBLOCK_SIZE];
	    charBlockPtr = 0;
	}
	int index = lastCharBlock * CHARBLOCK_SIZE + charBlockPtr;
	char[] chb = charBlocks[lastCharBlock];
	chb[charBlockPtr] = (char) charPtr;
	System.arraycopy( charBuffer, 0, chb, charBlockPtr + 1, charPtr);
	charBlockPtr += rlen;
	return index << 1;
    }

    int storeBigString( String s ) {
	int index = bigStrings.size();
	bigStrings.add(s);
	return (index << 1) + 1;
    }

    String decodeString( int code ) {
	if( (code & 1) != 0)
	    // big string:
	    return (String) bigStrings.get( code >> 1 );
	else {
	    code >>= 1;
	    char[] block = charBlocks[code >> CHARBLOCK_SHIFT];
	    int pos = code & CHARBLOCK_MASK;
	    // the first char is the length of the string
	    return new String(block, pos + 1, (int) block[pos] );
	}
    }

    void decodeString( int code, StringBuffer buffer ) {
	if( (code & 1) != 0)
	    // big string:
	    buffer.append( (String) bigStrings.get( code >> 1 ) );
	else {
	    code >>= 1;
	    char[] block = charBlocks[code >> CHARBLOCK_SHIFT];
	    int pos = code & CHARBLOCK_MASK;
	    // the first char is the length of the string
	    buffer.append( block, pos + 1, (int) block[pos] );
	}
    }

    char[] decodeChars( int code, int reserve ) {
	if( (code & 1) != 0) {
	    // big string:
	    String s = (String) bigStrings.get( code >> 1 );
	    char[] res = new char[ s.length() + reserve ];
	    s.getChars(0, s.length(), res, reserve);
	    return res;
	}
	else {
	    code >>= 1;
	    char[] block = charBlocks[code >> CHARBLOCK_SHIFT];
	    int pos = code & CHARBLOCK_MASK;
	    // the first char is the length of the string
	    int L = (int) block[pos];
	    char[] res = new char[ L + reserve ];
	    System.arraycopy(block, pos + 1, res, reserve, L );
	    return res;
	}
    }
}
