/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import java.text.Collator;

/**
 * Node as defined by the XQuery1/XPath2 <em>Data Model</em>, plus extensions.
 */
public interface Node
{
    /**
     * Accessor dm:node-kind().
     * 
     * @return the node kind as a string: "element", "attribute" etc.
     */
    String getNodeKind();
    /**
     * Accessor dm:node-name().
     * 
     * @return null if the node has no name (document, text, comment, pi).
     */
    QName  getNodeName();
    /**
     * Accessor dm:parent().
     * 
     * @return null if the node has no parent.
     */
    Node   parent();
    /**
     * Accessor dm:string-value().
     * <p>Returns the string value of the node. For an element, it is the
     * concatenation of text nodes contained in the element.
     */
    String getStringValue();
    /**
     * Accessor dm:base-uri().
     */
    String getBaseURI();

    /**
     * Returns the URI of the document.
     */
    String getDocumentURI();

    /**
     * Accessor dm:children. Returns the children in document order.
     * <p>Only for Document and Elements: other kinds return an empty sequence.
     */
    NodeSequence children();

    /**
     * Accessor dm:attributes. Only for Elements: other kinds return
     * an empty sequence.
     */
    NodeSequence attributes();

    /**
     * Accessor dm:namespaces(). According to the parameter, returns in-scope
     * namespaces for this node, or namespaces defined by this node.
     * On non-element nodes, always returns the empty sequence.
     * 
     * @param inScope if true, return the closure of in-scope namespaces,
     * else the namespaces defined specifically on this node.
     */
    NodeSequence namespaces( boolean inScope );

    // ---- extensions for internal use: -------------------------------------

    /** Node nature (returned by getNature) for a document node. */
    static int  DOCUMENT = 1;
    /** Node nature (returned by getNature) for an Element node. */
    static int  ELEMENT = 2;
    /** Node nature (returned by getNature) for an Attribute node.*/
    static int  ATTRIBUTE = 3;
    /** Node nature (returned by getNature) for a naemespacenode.*/
    static int  NAMESPACE = 4;
    /** Node nature (returned by getNature) for a processing instruction node.*/
    static int  PROCESSING_INSTRUCTION = 5;
    /** Node nature (returned by getNature) for a Comment node. */
    static int  COMMENT = 6;
    /** Node nature (returned by getNature) for a leaf text node. */
    static int  TEXT = 7;
    /** Node nature (returned by getNature) for a typed leaf node
     * holding a boolean value.*/
    static int  ATOM_BOOL = 8; 
    /** Node nature (returned by getNature) for a typed integer node,
     *  holding a long integer value.*/
    static int  ATOM_INT = 9;
    /** Node nature (returned by getNature) for a typed leaf node
	holding a double value.*/
    static int  ATOM_DOUBLE = 10;
    /** Node nature (returned by getNature) for typed leaf node
	holding a value of any serializable class.*/
    static int  ATOM_ANY = 11; 
    /** Node nature (returned by getNature) for typed leaf node
	holding a date/time value. */
    static int  ATOM_DATE = 12; 

    /**
     * Returns a numeric node kind: DOCUMENT, ELEMENT etc.
     */
    int   getNature();

    /**
     * Returns the document node if any, else the top-level node.
     */
    Node  document();

    /**
     * Returns an arbitrary value that is stable on the whole
     * document (for order comparison).
     */
    int docPosition();

    /**
     * Convenience: quick test of elements.
     */
    boolean isElement();
    /**
     * Convenience: quick test of atomic nodes (text, comment, typed atoms).
     */
    boolean isAtom();

    /**
     * Gets an attribute of this (element) by its name.
     * 
     * @return An attribute node, or null if not found or if applied to a non-element.
     */
    Node  attribute( QName name );

    /**
     * Gets text contents as a char array. Applicable only to text nodes, PI nodes and
     * attributes.
     */
    char[]  getChars();

    /**
     * Number of Namespaces defined on this particular node.
     */
    int     getDefinedNSCount();
    /**
     * Returns a matching prefix for the Namespace by looking up the namespace nodes.
     * 
     * @return the first suitable prefix, otherwise null if none is found.
     */
    String getNsPrefix( String nsuri );
    /**
     * Returns a matching Namespace for the prefix by looking up the namespace nodes.
     * 
     * @return the first suitable namespace URI, otherwise null if none is found.
     */
    String getNsUri( String prefix );

    /**
     * Returns -1 if this node is strictly before the argument node in document order, 0
     * if nodes are identical, 1 if after the argument node. If the two nodes belong to
     * different documents, returns an arbitrary but stable "order of documents".
     */
    int orderCompare( Node node );

    /**
     * Compares the string values of two nodes, optionally using a collation.
     */
    int compareStringValues( Node node, Collator collator );

    /**
     * Returns true if this node is an ancestor of the parameter node or the node
     * itself.
     */
    boolean contains( Node node );

    // ---- extensions for efficient XPath steps:

    /**
     * Returns an iterator on ancestors that match the node test.
     * 
     * @param nodeTest concrete classes are BaseNodeTest, UnionNodeTest.
     */
    NodeSequence ancestors( NodeTest nodeTest );

    /**
     * Returns an iterator on ancestors (including the node itself) that match the node
     * test.
     * 
     * @param nodeTest concrete classes are BaseNodeTest, UnionNodeTest.
     */
    NodeSequence ancestorsOrSelf( NodeTest nodeTest );

    /**
     * Returns a sequence of length 1 (if the parent matches the nodeTest) else 0.
     */
    NodeSequence parent( NodeTest nodeTest );
    /**
     * Returns an iterator on children that match the node test.
     */
    NodeSequence children( NodeTest nodeTest );
    /**
     * Returns an iterator on descendants that match the node test.
     */
    NodeSequence descendants( NodeTest nodeTest );
    /**
     * Returns an iterator on descendants (including the node itself) that match the
     * node test.
     */
    NodeSequence descendantsOrSelf( NodeTest nodeTest );
    /**
     * Returns an iterator on attributes of this element that match the node test.
     */
    NodeSequence attributes( NodeTest nodeTest );
    /**
     * Returns an iterator on following siblings that match the node test.
     */
    NodeSequence followingSiblings( NodeTest nodeTest );
    /**
     * Returns an iterator on preceding siblings (in document order) that match the node
     * test.
     */
    NodeSequence precedingSiblings( NodeTest nodeTest );
    /**
     * Returns an iterator on following nodes that match the node test.
     */
    NodeSequence following( NodeTest nodeTest );
    /**
     * Returns an iterator preceding nodes (in document order) that match the node test.
     */
    NodeSequence preceding( NodeTest nodeTest );

    // ----- Construction (optional): ----------------------------------------

    /**
     * Adds text inside the node. For an element, creates a text child node if
     * necessary.
     * <p>Caution: Some implementations do not support modification and
     * can raise a run-time exception.
     */
    void addText( String text );

    // ----- Extensions for typed atomic values ------------------------------

    /**
     * Returns the value of an atomic node.
     * 
     * @return Generally a String, but for typed atoms it is an object of
     * class Boolean (ATOM_BOOL), Long (ATOM_INT), Double (ATOM_BOOL) or
     * any class (ATOM_ANY).
     */
    Object   getValue();        // 

    /**
     * Returns the integer value (attempts to convert).
     * 
     * @exception DataModelException if not convertible to integer.
     */
    long     getIntegerValue() throws DataModelException;
}
