/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.ext;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.dt.*;
import net.xfra.qizxopen.xquery.impl.PredefinedModule;
import net.xfra.qizxopen.dm.NodeSequence;
import net.xfra.qizxopen.dm.NodeTest;
import net.xfra.qizxopen.dm.DataModelException;
import net.xfra.qizxopen.util.time.DateTimeBase;

import java.sql.*;

/**
 *  SQL Database connection via JDBC.
 *  
 *  Wrapper class for static methods.
 */
public class SqlConnection 
{
    public static String NS = "java:net.xfra.qizxopen.ext.SqlConnection";

    public static ItemType TYPE_CONNECTION =
        new WrappedObjectType(Connection.class);
    public static ItemType TYPE_RESULTSET =
        new WrappedObjectType(ResultSet.class);
    public static ItemType TYPE_PSTATEMENT =
        new WrappedObjectType(PreparedStatement.class);

    public static void plugHook(PredefinedModule module) throws XQueryException{
	if(module.simpleFunctionLookup(QName.get(NS, "execute")) != null)
	    return;
	module.declareFunction(new SqlExecQuery());
	module.declareFunction(new SqlExecUpdate());
	module.declareFunction(new SqlRawExec());
    }

    /**
     *	Explicitly registers a driver.
     *	(not necessary if system property jdbc.drivers is used).
     */
    public static void registerDriver(String className) throws Exception {
	// The newInstance() call is a work around for some 
	// broken Java implementations (?)
	Class.forName(className).newInstance(); 
    }

    /**
     *	Opens a Connection.
     */
    public static
    Connection getConnection( String url, String user, String passwd )
      throws SQLException {
	Connection conn = DriverManager.getConnection(url, user, passwd);
	return conn;
    }

    /**
     *	Closes a Connection.
     */
    public static void close( Connection conn )
      throws SQLException {
	conn.close();
    }

    /**
     * Creates a prepared statement.
     * Just to make it easier: write sql:prepare($conn, "...") rather than that:
     *	<pre>declare namespace jdbc = "java:java.sql";
     *	jdbc:Connection.prepareStatement($conn, "...")
     *	</pre>
     */
    public static PreparedStatement prepare( Connection conn, String statement)
      throws SQLException {
	return conn.prepareStatement(statement);
    }

    public static class RawResult extends BaseValue
    {
	static QName ROW = QName.get("row");

	ResultSet rset;
	ResultSetMetaData meta = null;
	QName[] colNames;

	RawResult(ResultSet rset) {
	    this.rset = rset;
	}

	public boolean next() {
	    try {
		return rset.next();
	    }
	    catch (SQLException e) {
		//e.printStackTrace();
		return false;
	    }
	}

	public Item asItem() throws TypeException {
	    return new SingleWrappedObject(rset);
	}

	public String  asString() throws TypeException {
	    return rset.toString();
	}

	public Value bornAgain() {
	    // cant clone the result set: Problem! might give strange results
	    // if reset (beforeFirst) not implemented.
	    try {
		rset.beforeFirst();
	    }
	    catch (SQLException e) { }	// just try
	    return new RawResult(rset); 
	}
    }

    /**
     *	Node wrapper: presents the result set as a sequence of nodes.
     *	<p>Each node represents a row of data. Its children elements have
     *	the names of fields returned by the query.
     *	<p>Caches the node that represents the current row in result set.
     */
    public static class Result extends RawResult
    {
	EventDrivenBuilder builder = new EventDrivenBuilder();

	Result(ResultSet rset) {
	    super(rset);
	}

	public ItemType  getType() {
	    return Type.NODE;
	}

	public Item  asItem() throws TypeException {
	    return asNode();
	}

	public boolean  isNode() {
	    return true;
	}
	
	public Node asNode() throws TypeException {
	    try {
		if(meta == null) {
		    meta = rset.getMetaData();
		    int ccnt = meta.getColumnCount();
		    colNames = new QName[ccnt];
		    for(int c = 1; c <= ccnt; c++) {
			colNames[c - 1] = QName.get(meta.getColumnLabel(c));
  
		    }
		}
		builder.reset();
		builder.startElement(ROW);
		for(int c = 0; c < colNames.length; c++) {
		    builder.startElement(colNames[c]);
		    String s = rset.getString(c + 1); //TODO conversion
		    if(s != null) 
			builder.atom(s);
		    builder.endElement(colNames[c]);
		}
		builder.endElement(ROW);
		return builder.crop();
	    }
	    catch (SQLException e) {
		throw new TypeException("error in row construction", e);
	    }
	    catch (DataModelException e) {
		throw new TypeException("error in row construction", e);
	    }
	}

	public Value bornAgain() {
	    // cant clone the result set: Problem! might give strange results
	    // if reset (beforeFirst) not implemented.
	    try {
		rset.beforeFirst();
	    }
	    catch (SQLException e) { }	// just try
	    return new Result(rset); 
	}
    } // end of class Result

    /**
     *	
     */
    static class Stat {
	PreparedStatement stat;
	ParameterMetaData meta;

	Stat(PreparedStatement st) {
	    stat = st;
	}

	void setArg(int rank, Item value) throws TypeException, SQLException {
	    
	    ItemType type = value.getType();
	    if(value instanceof StringValue) {
		stat.setString(rank, value.asString());
	    }
	    else if(Type.INTEGER.isSuperType(type)) {
		long v = value.asInteger();
		stat.setLong(rank, v);
	    }
	    else if(Type.NUMERIC.isSuperType(type)) {
		if(type == Type.DOUBLE)
		    stat.setDouble(rank, ((DoubleValue) value).asDouble() );
		else if(type == Type.DECIMAL)
		    stat.setBigDecimal(rank, ((DecimalValue) value).getValue());
		else if(type == Type.FLOAT)
		    stat.setFloat(rank, ((FloatValue) value).asFloat() );
		else System.err.println("OOPS NUMERIC: "+value);
	    }
	    else if(value instanceof MomentValue) {
		DateTimeBase dt = ((MomentValue) value).getValue();
		stat.setDate(rank,
			     new Date((long) dt.getMillisecondsFromEpoch()));
	    }
	    else if(value == null) {
		if(meta == null)
		    meta = stat.getParameterMetaData();
		stat.setNull(rank, meta.getParameterType(rank));
	    }
	}
    } // end of class Stat

}
