/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.gui;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.dm.XMLSerializer;
import net.xfra.qizxopen.dm.DataModelException;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.*;
import net.xfra.qizxopen.xquery.impl.SyntaxColorer;
import net.xfra.qizxopen.xquery.impl.Version;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import javax.swing.table.*;
import javax.swing.tree.*;
import javax.swing.text.*;

import java.util.Locale;
import java.util.ResourceBundle;
import java.util.MissingResourceException;
import java.util.Enumeration;
import java.util.Vector;
import java.io.*;
import java.net.URL;

/**
 *	A simple GUI for XQuest / Qizx/open.
 */
public class XQuestGUI extends JFrame
    implements XQueryProcessor.PauseHandler
{
    private static ResourceBundle resourceBundle = null;
    private static String MSG_ATTR = "MESSG";

    private static String APP = "Qizx/open";

    JSplitPane topSplit, dispSplit;
    JTabbedPane toolTabs;	// left-side tabbed tools

    QEdit queryEdit;
    MessageView messages;
    XMLView xmlView;

    JScrollPane itemView, nodeView;
    JTabbedPane nodeTabs; // tree view + XML view
    JTable itemList;
    JTree nodeDisplay;
    DefaultTreeModel treeModel;

    JMenuBar mbar;
    JMenu filesMenu, helpMenu;

    JFileChooser fileChooser;
    JDialog helpDialog;

    // ---- XQuery:
    XQueryProcessor xproc;
    XMLSerializer   serialOut;
    Log log;
    Item[] itemArray = new Item[100];
    int itemCount;
    Waiter waiter;
    // asynchronous execution:
    XQuery execQuery;
    long   execTime;
    XQueryException execError;
    EvalContext execPauseContext;

    
    // options:
    public String  queryFile;
    public boolean traceQuery = false;

    public String  base = ".";
    public String  inputURI = null;
    public String  collInput = null;
    public String  xmlInput = null;
    public String[] globals = new String[0];
    public String[] xmloptions = new String[0];
    public String[] appargs = new String[0];
    public String  timezone;
    public String  collation;

    static CLOptions options = new CLOptions("XQuest");
    static {
	options.declare("-base", CLOptions.NEXT_ARG,
		"default base URI for libraries, documents and modules");
	options.declare("-D", "globals", CLOptions.STICKY_ARG,
	      "variable_name=value!initialize a global variable defined in the query.");
	options.declare("--", "appargs", CLOptions.ALL_ARG,
	    "pass all following arguments to XQuery processor in variable '$arguments'");
	options.declare("-X", "xmloptions", CLOptions.STICKY_ARG,
			"option=value!set a XML serialization option.");
	options.declare("-input", "inputURI", CLOptions.NEXT_ARG,
			"URI of a document used as input (XQuery function input())");
	options.declare("-cinput", "collInput", CLOptions.NEXT_ARG, 
			"URI of a collection used as input (XQuery function input())");
	options.declare("-xinput", "xmlInput", CLOptions.NEXT_ARG, 
			"a XML fragment used as input (XQuery function input())");
	options.declare("-timezone", CLOptions.NEXT_ARG,
			"implicit timezone in duration format");
	options.declare("-collation", CLOptions.NEXT_ARG, "default collation");

	options.declare("-help", null, CLOptions.HELP_ARG, "print this help");
	options.argument("<query file>", "queryFile", 0,
			 "a file containing a query to load.");
	options.declare("-tex", "traceExceptions", CLOptions.SET_ARG,
			"verbose display of exceptions");
	// undocumented options:
	options.declare("-tq", "traceQuery", CLOptions.SET_ARG, null);
    }

    public XQuestGUI(String appName) {
	super(appName);
    }

    public void start()
	throws XQueryException, DataModelException, IOException {
	xproc = new XQueryProcessor(base, base);
	if(inputURI != null)
	    xproc.setDocumentInput(inputURI);
	if(collInput != null)
	    xproc.setCollectionInput(collInput);
	if(xmlInput != null)
	    xproc.setInput(xmlInput);
	if(timezone != null)
	    xproc.setImplicitTimezone(timezone);
	if(collation != null)
	    xproc.setDefaultCollation(collation);
	xproc.initGlobal(QName.get("arguments"), appargs);

	addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    quit();
		}
	    });
	createGUI();
	log = new GUILog();
	waiter = new Waiter();
	waiter.start();
	if(queryFile != null)
	    queryEdit.loadFrom(new File(queryFile));
	serialOut = new XMLSerializer();
 	// XML options:
	for(int g = 0; g < xmloptions.length; g++) {
	    int eq = xmloptions[g].indexOf('=');
	    if(eq < 0) {
		throw new XQueryException("invalid XML option: "+xmloptions[g]);
	    }
	    serialOut.setOption( xmloptions[g].substring(0, eq),
				 xmloptions[g].substring(eq + 1));
	}
     }
 
    static public void usage( ) {
        System.err.println("usage!");
        System.exit(1);
    }

    static public void main( String args[] )
    {
        try {
	    XQuestGUI tool = new XQuestGUI(APP);
	    options.parse(args, tool);
	    tool.start();
	    tool.setSize(850, 700);
	    tool.setVisible(true);
        }
	catch (CLOptions.Exception e) {
	    System.exit(0);
        }
	catch (Exception e) {
            e.printStackTrace();
	    System.exit(1);
        }
    }

    // ------- XQuery -------------------------------------------------------

    boolean doExecute(String src) {
	log.reset();
	changeNodeDisplay(null);
	for(int i = itemCount; --i >= 0; )
	    itemArray[i] = null; // recover memory
	itemCount = 0;
	try {
	    execQuery = xproc.compileQuery( src, "<input>", log );
	}
        catch (XQueryException e) {
	    message("compilation:");
	    displayLog();
	    message(e.getMessage(), messages.errorStyle);
	    return false;
        }
	waiter.perform(queryExecution);
	return true;
    }

    // executed by the service thread
    Runnable queryExecution = new Runnable() {
	    public void run() {
		long T0 = System.currentTimeMillis();
		queryEdit.setRunning(true);
		try {
		    for(int g = 0; g < globals.length; g++) {
			int eq = globals[g].indexOf('=');
			if(eq < 0) {
			    message("illegal variable initializer: "+
				    globals[g]);
			    return;
			}
			xproc.initGlobal(QName.get(globals[g].substring(0, eq)),
					 globals[g].substring(eq + 1));
		    }
		    execTime = -1;
		    execError = null;
		    Value v = xproc.executeQuery( execQuery );
		    itemCount = 0;
		    for( ; v.next(); ) {
			if(itemCount >= itemArray.length) {
			    Item[] old = itemArray;
			    itemArray = new Item[ 2 * old.length ];
			    System.arraycopy(old, 0, itemArray, 0, old.length);
			}
			itemArray[itemCount++] = v.asItem();
		    }
		    execTime = System.currentTimeMillis() - T0;
		}
		catch (XQueryException e) {
		    execError = e;
		}
		catch (OutOfMemoryError oom) {
		    execError = new XQueryException("out of memory error", oom);
		}
		SwingUtilities.invokeLater(finishQueryExecution);
	    }
	};

    // executed by the AWT thread through invokeLater
    Runnable finishQueryExecution = new Runnable() {
	    public void run() {
		execPauseContext = null;
		if(execError != null) {
		    message("execution:");
		    if(execError instanceof EvalException) {
			((EvalException) execError).printStack(log, 20);
			displayLog(); // printStack does not display..
		    }
		    else {
			displayLog();
			message(execError.getMessage(), messages.errorStyle);
		    }
		}
		queryEdit.setRunning(false);
		itemList.getSelectionModel().clearSelection();
		if(itemCount > 0)
		    itemList.getSelectionModel().setSelectionInterval(0, 0) ;
		else itemList.getSelectionModel().clearSelection();
		itemList.revalidate();
		itemList.repaint();
		if(execTime >= 0)
		 message(itemCount+" item(s), execution time: "+execTime+" ms");
	    }
	};

    public void pauseAt(EvalContext ctx) {	//TODO? do the wait here
	execPauseContext = ctx;
	SwingUtilities.invokeLater(pausedExecution);
    }

    synchronized void unpause() {
	execPauseContext = null;
	this.notify();	// restart
    }

    Runnable pausedExecution = new Runnable() {
	    public void run() {
		message("stopped in:");
		execPauseContext.printStack(log, 20);
		displayLog();
	    }
	};

    int message( String msg ) {
	return message(msg, null);
    }

    int message( String msg, AttributeSet style) {
	int pos = messages.appendText(msg, style);
	messages.appendText("\n", style);
	return pos;
    }

    void displayLog() {
	Log.Message msg;
	for(int m = 0; (msg = log.getMessage(m)) != null; m++) {
	    Style bst = (msg.severity == Log.ERROR)?
		messages.errorStyle : messages.noStyle;
	    MutableAttributeSet must = messages.mutableStyle(bst);
	    must.addAttribute(MSG_ATTR, msg);
	    String prefix = (msg.severity == Log.ERROR)? "*** " : "    ";
	    messages.appendText(prefix, must);
	    message(msg.expand(), must);
	}
	log.reset();
    }

    // --------------- GUI ------------------------------------------------

    private void createGUI() {
	JPanel workArea = (JPanel) getContentPane(); 

	createItemView();

	nodeTabs = new JTabbedPane(JTabbedPane.TOP,
				   JTabbedPane.WRAP_TAB_LAYOUT);
	treeModel = new DefaultTreeModel(new DisplayNode(null));
	nodeDisplay = new JTree(treeModel);
	//nodeDisplay.setRoot();
	nodeView = new JScrollPane(nodeDisplay);
	//setHeader(nodeView, new JLabel("XML tree: "));
	nodeTabs.addTab("Tree View", null, nodeView,
			"Display of a Node item as a tree");

	xmlView = new XMLView();
	nodeTabs.addTab("Tag View", null, new JScrollPane(xmlView),
			"Display of a Node item as tags");

	dispSplit = new JSplitPane( JSplitPane.VERTICAL_SPLIT, 
				    itemView, nodeTabs );
	dispSplit.setOneTouchExpandable(true);
	dispSplit.setDividerLocation(200); // TODO settings

	createTabs();

	topSplit = new JSplitPane( JSplitPane.HORIZONTAL_SPLIT, 
				   toolTabs, dispSplit );
	topSplit.setOneTouchExpandable(true);
	topSplit.setResizeWeight(0.3);
	topSplit.setDividerLocation(480); // TODO settings

	workArea.add(topSplit, BorderLayout.CENTER);

	buildMenu();
    }

    private void createTabs() {
	toolTabs = new JTabbedPane(JTabbedPane.TOP,
				   JTabbedPane.WRAP_TAB_LAYOUT);
	toolTabs.addChangeListener(new ChangeListener() {
		public void stateChanged(ChangeEvent e) {
		}
	    }) ;

	// tab: query edit and messages
	queryEdit = new QEdit();
	messages = new MessageView();
	JSplitPane querySplit = new JSplitPane( JSplitPane.VERTICAL_SPLIT, 
						queryEdit, messages );
	querySplit.setResizeWeight(0.8);
	querySplit.setDividerLocation(300); // TODO settings
	
	toolTabs.addTab("Query input", null, querySplit,
			"Type and edit a XQuery");

    }

    void createItemView() {
	AbstractTableModel itemModel = new ItemTableModel();
	itemList = new JTable(itemModel);
	ListSelectionModel lsm = itemList.getSelectionModel();
	lsm.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
	lsm.addListSelectionListener(new ListSelectionListener() {
		public void valueChanged(ListSelectionEvent e) {
		    if(e.getValueIsAdjusting())
			return;
		    int index =
			itemList.getSelectionModel().getMinSelectionIndex();
		    if(index >= 0 && index < itemCount)
			try {
			    if(itemArray[index].isNode())
				changeNodeDisplay(itemArray[index].asNode());
			    else changeNodeDisplay(null);
			}
			catch (XQueryException ex) {
			    ex.printStackTrace();
			}
		}
	    });
	// settings:
 	TableColumnModel cm = itemList.getColumnModel();
 	cm.getColumn(0).setMinWidth(10); cm.getColumn(0).setMaxWidth(50);
 	cm.getColumn(1).setMinWidth(20); cm.getColumn(1).setMaxWidth(150);
 	cm.getColumn(2).setMinWidth(50); cm.getColumn(2).setPreferredWidth(200);

	itemView = new JScrollPane(itemList);
	JToolBar toolbar = new JToolBar();
	toolbar.add(new JLabel("Results:"));
	setHeader(itemView, toolbar);
    }

    private void buildMenu()
    {
	mbar = new JMenuBar();
	setJMenuBar(mbar);

	filesMenu = new JMenu(loc("Files"));
	addCmd( filesMenu, "Open...", KeyEvent.VK_O, new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    haveFileChooser();
		    int resp = fileChooser.showOpenDialog(XQuestGUI.this);
		    if(resp != JFileChooser.APPROVE_OPTION)
			return;
		    File file = fileChooser.getSelectedFile();
		    try {
			queryEdit.loadFrom(file);
		    }
		    catch (IOException ioe) {
		      JOptionPane.showMessageDialog(XQuestGUI.this, "error",
						    "cannot read file "+file+
						    ": "+ioe.getMessage(),
						    JOptionPane.ERROR_MESSAGE);
		    }
		}
	    });
	addCmd( filesMenu, "Save as...", KeyEvent.VK_S, new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    haveFileChooser();
		    int resp = fileChooser.showSaveDialog(XQuestGUI.this);
		    if(resp != JFileChooser.APPROVE_OPTION)
			return;
		    File file = fileChooser.getSelectedFile();
		    try {
			queryEdit.saveTo(file);
		    }
		    catch (IOException ioe) {
		      JOptionPane.showMessageDialog(XQuestGUI.this, "error",
						    "cannot save to file "+file+
						    ": "+ioe.getMessage(),
						    JOptionPane.ERROR_MESSAGE);
		    }
		}
	    });
	addCmd( filesMenu, null, 0, null );
	addCmd( filesMenu, "Quit", KeyEvent.VK_Q, new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    quit();
		}
	    } );
	addCmd( filesMenu, null, 0, null );// insertion position of recent files
	mbar.add( filesMenu );

	helpMenu = new JMenu(loc("Help"));
	addCmd( helpMenu, "Help", 0, new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    help();
		}
	    } );
	addCmd( helpMenu, "About "+APP, 0, new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    about();
		}
	    } );
	mbar.add( helpMenu );
    }

    void haveFileChooser() {
	if(fileChooser != null)
	    return;
	fileChooser = new JFileChooser();
	URL examp = getClass().getClassLoader().getResource("examples");
	File src = examp != null? FileUtil.urlToFile(examp) : new File(".");
	fileChooser.setCurrentDirectory(src);
    }

    private void addCmd( JMenu menu, String label, int accel,
			 ActionListener listener ) {
	if (label == null)
	    menu.addSeparator();
	else {
	    JMenuItem item = accel == 0 ? 
		new JMenuItem(loc(label)) : new JMenuItem(loc(label), accel);
	    if (accel != 0)
		item.setAccelerator(KeyStroke.getKeyStroke( accel,
							    Event.CTRL_MASK));
	    item.addActionListener(listener);
	    item.setActionCommand(label);
	    menu.add(item);
	}
    }

    static void setHeader(JScrollPane pane, JComponent header) {
	JViewport hport = new JViewport();
	hport.setView(header);
	pane.setColumnHeader(hport);
    }

    public static String loc(String id) {
	String message = null;
	try {
	    return resourceBundle.getString(id);
	} catch (Exception e) {
	    return id;
	}
    }

    ImageIcon getIcon(String name) {
	String iconLoc = "resources/"+ name +".gif";
	URL iconURL = ClassLoader.getSystemResource(iconLoc);
	return iconURL == null? null : new ImageIcon(iconURL, name);
    }

    // ------------------ events --------------------------------------------

    private void quit() {
	if(queryEdit.isModified() &&
	   JOptionPane.showConfirmDialog(
	       this,
	       "The current query is modified:\ndo you really want to quit?",
	       "Warning", JOptionPane.OK_CANCEL_OPTION)
	   != JOptionPane.YES_OPTION)
	    return;
	System.exit(0);
    }

    private void help() {
	if(helpDialog == null) {
	    helpDialog = new JDialog(this, "Help", false) ;
	    JTextPane contents = new JTextPane();
	    JScrollPane port = new JScrollPane(contents);
	    JToolBar toolbar = new JToolBar();
	    toolbar.add(new JLabel("Help:   "));
	    JButton close = new JButton("Close");
	    toolbar.add(close);
	    close.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent e) {
			helpDialog.setVisible(!true);
		    }
		});
	    ImageIcon iicon = getIcon("zap");
	    if (iicon != null)
		close.setIcon(iicon);
	    setHeader(port, toolbar);
	    helpDialog.getContentPane().add(port);
	    helpDialog.setSize(500, 500);
	    Rectangle pr = getBounds();
	    helpDialog.setLocation(pr.x + pr.width/2, pr.y + pr.height/2);
	    // contents:
	    URL url = ClassLoader.getSystemResource("resources/help.html");
	    if(url != null)
		try {
		    contents.setPage(url);
		}
		catch (IOException e) {
		    contents.setText("error loading help file: "+e);
		}
	    else contents.setText("cannot locate help file");
	    contents.setEditable(false);
	}
	helpDialog.setVisible(true);
    }

    private void about() {
	Runtime runtime = Runtime.getRuntime();
        runtime.gc();
        long totalMemory = runtime.totalMemory();
        long usedMemory = (totalMemory - runtime.freeMemory()) / 1024;
        long maxMemory = runtime.maxMemory() / 1024;
	String msg = APP + " v"+ Version.get() +"\n\n"
	+ "Copyright \u00A9 2003-2004 Xavier C. Franc, all rights reserved.\n"
	+ "\n"
	+ "For more information, please visit http://www.xfra.net/qizxopen/\n"
	+ "\n\n"
	    + "Memory used "+usedMemory+" Kb, maximum "+maxMemory+" Kb\n\n";
	
	JOptionPane.showMessageDialog(this, msg, "About "+APP,
                                      JOptionPane.PLAIN_MESSAGE, null);
    }

    void changeNodeDisplay(Node root) {
	treeModel.setRoot(new DisplayNode(root));
	nodeDisplay.revalidate();
	nodeDisplay.repaint();
	// xml view:
	if(root != null) {
	    StringWriter disp = new StringWriter(500);
	    serialOut.setOutput(disp);
	    try {
		serialOut.output(root);
	    }
	    catch (DataModelException e) {
		e.printStackTrace();
	    }
	    xmlView.text.setText(disp.toString());
	}
	else xmlView.text.setText("");
    }

    // ----------------- auxiliary objects -------------------------------

    /**
     *	Query editor: syntax highlighting, query history
     */
    public abstract class TextPort extends JScrollPane implements ActionListener
    {
	JTextPane  text;
	//JTextArea  text;
	JToolBar   toolbar;
	StyleContext styles = new StyleContext();
	StyledDocument doc;

	TextPort(String title /*, int rows, int columns*/) {
	    doc = new DefaultStyledDocument();
	    text = new JTextPane(doc);
	    //text = new JTextArea(doc, "", rows, columns );
	    //text.setLineWrap(false);
	    setViewportView(text);
	    if(title != null) {
		toolbar = new JToolBar();
		JLabel label = new JLabel(title);
		toolbar.add(label);
		setHeader(this, toolbar);
	    }
	    //
	}

	int appendText( String txt, AttributeSet style ) {
	    try {
		int pos = doc.getLength();
		doc.insertString( pos, txt, style );
		Rectangle r = text.modelToView(doc.getLength());
		getViewport().scrollRectToVisible(r);
		return pos;
	    } catch(BadLocationException e) {
		e.printStackTrace(); return doc.getLength();
	    }
	}

	JButton addButton(String text, String icon,
			  String actionCommand, String toolTip) {
	    JButton button = new JButton();
	    button.setActionCommand(actionCommand);
	    button.setToolTipText(toolTip);
	    button.addActionListener(this);

	    ImageIcon iicon = getIcon(icon);
	    if (iicon != null)
		button.setIcon(iicon);
	    else if(text == null)
		button.setText(actionCommand);
	    if(text != null)
		button.setText(text);

	    toolbar.add(button);
	    return button;
	}

	Style addStyle(String name, Color fontColor) {
	    Style style = doc.addStyle(name, null);
	    StyleConstants.setForeground( style, fontColor);
	    StyleConstants.setBold(style, true);
	    return style;
	}

	MutableAttributeSet mutableStyle( Style parent ) {
	    return new SimpleAttributeSet(parent);
	}

	public abstract void actionPerformed(ActionEvent e);

    } // end of class TextPort

    /**
     *	Query editor: syntax highlighting, query history
     */
    public class QEdit extends TextPort implements DocumentListener
    {
	boolean modified;
	boolean locked;
	Thread timer;
	Style[] tokenStyles;
	Vector history = new Vector();
	int    histoLoc;
	String savedCurrent;	// save modified query
	JButton execButton, stopButton, pauseButton, stepButton;

	QEdit() {
	    super("Query:  "/*, 12, 30*/);
	    //text.setBorder(BorderFactory.createEtchedBorder());
	    text.setBorder(BorderFactory.createCompoundBorder(
			       BorderFactory.createEtchedBorder(),
			       BorderFactory.createEmptyBorder(0, 4, 0, 4)));
	    execButton = addButton("Execute", "exec", "exec",
				   "Compile and execute query");
	    stopButton = addButton(null, "stop", "stop",
				   "Abort execution");
	    pauseButton = addButton(null, "pause", "pause",
				   "Pause execution");
	  //stepButton = addButton(null, "step", "step","One-step execution");

	    setRunning(false);
	    toolbar.add(new JToolBar.Separator());
	    addButton(null, "clear", "clear",
		      "clear the query edition area");
	    toolbar.add(new JToolBar.Separator());
	    addButton(null, "back", "previous",
		      "previous query in history");
	    addButton(null, "forward", "next",
		      "next query in history");
	    doc.addDocumentListener(this);
	    tokenStyles = new Style[] {
		addStyle("dummy", Color.black),
		addStyle("TAG", new Color(0x009080)),
		addStyle("SPACE", Color.white),	// whatever
		addStyle("NUMBER", new Color(0xc07000)),
		addStyle("STRING", new Color(0xc07000)),
		addStyle("MISC", new Color(0x303030)),
		addStyle("NAME", new Color(0x3050e0)),
		addStyle("KEYWORD", new Color(0x004090)),
		addStyle("COMMENT", new Color(0x30e070)),
		addStyle("PRAGMA", new Color(0xc09070)),
		addStyle("FUNC", new Color(0x7030f0))
	    };
	}

	public void actionPerformed(ActionEvent e) {
	    String cmd = e.getActionCommand();
	    if(cmd.equals("clear"))
		text.setText("");
	    else if(cmd.equals("previous")) {
		if(histoLoc > 0) {
		    if(modified && histoLoc == history.size())
			savedCurrent = text.getText();
		    -- histoLoc;
		    locked = true;
		    text.setText( (String) history.get(histoLoc) );
		    locked = false;
		}
		debugHistory();
	    }
	    else if(cmd.equals("next")) {
		locked = true;
		if(histoLoc < history.size() - 1) {
		    ++ histoLoc;
		    text.setText( (String) history.get(histoLoc) );
		}
		else if(savedCurrent != null) {
		    text.setText( savedCurrent );
		}
		locked = false;
		debugHistory();
	    }
	    else if(cmd.equals("exec")) {
		String q = text.getText();
		if(doExecute(q) && modified) {
		    history.add(q);
		    histoLoc = history.size();
		    modified = false;
		    savedCurrent = null;
		    debugHistory();
		}
	    }
	    else if(cmd.equals("stop")) {
		unpause();
		xproc.stopExecution();
	    }
	    else if(cmd.equals("pause")) {
		if(execPauseContext != null) {
		    unpause();
		}
		else xproc.pauseExecution(XQuestGUI.this);
	    }
	    else if(cmd.equals("step")) {
		xproc.pauseExecution(XQuestGUI.this);
		unpause();
	    }
	}

	void loadFrom(File input) throws IOException {
	    String contents = FileUtil.loadString(input);
	    text.setText(contents);
	    modified = false;
	}

	void saveTo(File output) throws IOException {
	    FileUtil.saveString(text.getText(), output);
	    modified = false;
	}

	boolean isModified() {
	    return modified;
	}

	void setRunning(boolean running) {
	    execButton.setEnabled(!running);
	    stopButton.setEnabled(running);
	    pauseButton.setEnabled(running);
	}

	void debugHistory() {
	    if(false) {
		System.err.println(" ptr "+histoLoc);
		for(int h = 0; h < history.size(); h++)
		    System.err.println(h+" "+history.get(h));
		System.err.println("saved: "+savedCurrent);
		System.err.println("modified "+modified);
	    }
	}

	void gotChange() {
	    if(!locked) {
		modified = true;
		histoLoc = history.size(); // if old is modified it becomes new
	    }
	    waiter.perform(refreshSyntax);
	}

	final Runnable refreshSyntax = new Runnable() {
		public void run() {
		    try {
			Thread.sleep(500);
			SwingUtilities.invokeLater(redoSyntax);
		    }
		    catch (InterruptedException e) { //e.printStackTrace(); 
		    }
		}
	    };

	final Runnable redoSyntax = new Runnable() {
		public void run() {
		    timer = null;
		    try {
			SyntaxColorer scolo = new SyntaxColorer(text.getText());
			int token = scolo.nextToken();
			for( ; token > 0; token = scolo.nextToken()) {
			    int start = scolo.getTokenStart();
			    int tlen = scolo.getTokenLength();
			    doc.setCharacterAttributes(start, tlen,
						      tokenStyles[token], true);
			}
		    }
		    catch (XQueryException e) {
			//e.printStackTrace(); // errors can happen
		    }
		    text.revalidate();
		}
	    };

	public void changedUpdate(DocumentEvent e) { }
	public void insertUpdate(DocumentEvent e) { gotChange(); }
	public void removeUpdate(DocumentEvent e) { gotChange(); } 
    } // end of class QEdit
    
    /**
     *	Displays compilation and execution messages.
     */
    public class MessageView extends TextPort implements CaretListener
    {
	Style errorStyle, noStyle;

	MessageView() {
	    super("Messages:  ");
	    addButton("Clear", "clear", "clear", "clear messages");
	    errorStyle = addStyle("ERROR", new Color(0xf03020));
	    noStyle = addStyle("PLAIN", new Color(0x505050));
	    text.addCaretListener(this);
	}

	public void actionPerformed(ActionEvent e) {
	    String cmd = e.getActionCommand();
	    if(cmd.equals("clear"))
		text.setText("");
	}

	public void caretUpdate(CaretEvent e) {
	    Element elem = doc.getCharacterElement(e.getDot());
	    if(elem == null) return;
	    Object attr = elem.getAttributes().getAttribute(MSG_ATTR);
	    if(attr == null) return;
	    Log.Message msg = (Log.Message) attr;
	    
	    if(msg.module != null) {
		queryEdit.text.requestFocusInWindow();
		queryEdit.text.select(msg.location, msg.location + 1);
	    }
	}

    } // end of class MessageView

    /**
     *	Display of items
     */
    public class ItemTableModel extends AbstractTableModel 
    {
	ItemTableModel() {
	}
	public int getColumnCount() { return 3; }

	public int getRowCount() {
	    return itemCount;
	}

	public Object getValueAt(int row, int col) {
	    switch(col) {
	    case 0: return new Integer(row);
	    case 1:
		return itemArray[row].getType().toString();
	    case 2:
		try {
		    if(!itemArray[row].isNode())
			return itemArray[row].asString();
		    return "see below";
		}
		catch (XQueryException e) {
		    return "<error>";
		}
	    }
	    return "cell "+row+":"+col;
	}

	public String 	getColumnName(int columnIndex) {
	    switch(columnIndex) {
	    case 0: return "rank";
	    case 1: return "type";
	    case 2: return "value";
	    }
	    return "?";
	}

    } // end of class ItemTableModel

    
    /**
     *	
     */
    public class DisplayNode implements TreeNode
    {
	Node displayed;
	DisplayNode[] children;	//TODO optimize
	int childCnt = -1;

	DisplayNode( Node displayed ) {
	    this.displayed = displayed;
	}

	public TreeNode getChildAt(int childIndex) {
	    expand();
	    return children[childIndex];
	}

	public int getChildCount() {
	    if(displayed == null)
		return 0;
	    expand();
	    return childCnt;
	}

	public boolean isLeaf() {
	    if(displayed == null)
		return false;
	    expand();
	    return childCnt == 0;
	}

	public TreeNode getParent() {
	    if(displayed == null)
		return null;
	    return new DisplayNode(displayed.getParent());
	}

	public int getIndex(TreeNode node) {
	    return -1;
	}

	public boolean getAllowsChildren() {
	    return true;
	}

	public Enumeration children() {
	    return null;
	}

	public String toString() {
	    if(displayed == null)
		return "<no node>";
	    String kind = displayed.getNodeKind();
	    QName name = displayed.getNodeName();
	    try {
		switch(displayed.getNature()) {
		case Node.ELEMENT:
		    StringBuffer buf = new StringBuffer(kind +" "+name);
		    for(Value a = displayed.getAttributes(); a.next(); ) {
			Node aNode = a.asNode();
			buf.append(' ');
			buf.append(aNode.getNodeName().toString());
			buf.append('=');
			buf.append(aNode.getStringValue());
		    }	
		    return buf.toString();
		case Node.TEXT:
		case Node.COMMENT:
		    return kind +" "+displayed.getStringValue();
		case Node.PROCESSING_INSTRUCTION:
		    return kind +" "+name+" "+displayed.getStringValue();
		default:
		    return kind;
		}
	    }
	    catch (XQueryException e) { // should not happen
		return e.toString();
	    }
	}

	void expand() {
	    if(childCnt >= 0)
		return;
	    children = new DisplayNode[4];
	    childCnt = 0;
	    try {
		for(Value seq = displayed.getChildren(); seq.next(); ) {
		    if(childCnt >= children.length) {
			DisplayNode[] old = children;
			children = new DisplayNode[ old.length * 2 ];
			System.arraycopy(old, 0, children, 0, old.length);
		    }
 		    children[childCnt++] = new DisplayNode(seq.asNode());
		}
	    }
	    catch (XQueryException e) {
		e.printStackTrace();
	    }
	}
    }

    public class XMLView extends TextPort {

	XMLView() {
	    super(null);
	}
	public void actionPerformed(ActionEvent e) { }
    }

    /**
     *	Service thread: refresh timer and asynchronous execution
     */
    public class Waiter extends Thread
    {
	Object lock = new Object();
	Runnable task = null;

	void perform(Runnable task) {
	    this.task = task;
	    synchronized(lock) {
		lock.notify();
	    }
	}

	public void run() {
	    for(;;) {
		synchronized(lock) {
		    try {
			lock.wait();
		    } catch (InterruptedException e) {  }
		}
		if(task != null)
		    task.run();
	    }
	}
    }

    /**
     *	
     */
    public class GUILog extends Log
    {
	GUILog() {
	}

	// silent: we collect messages as objects
	public void printMessage( Message msg ) { }
    } // end of class MyLog

} // end of class XQuestGUI
