/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

///package net.xfra.qizxopen.server;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.FileUtil;
import net.xfra.qizxopen.dm.XMLSerializer;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Pragma;

import javax.servlet.*;
import javax.servlet.http.*;
import javax.xml.transform.*;
import javax.xml.transform.sax.*;
import javax.xml.transform.stream.*;
import org.xml.sax.InputSource;

import java.util.*;
import java.io.*;
import java.net.*;

/**
 *  XML Query Servlet: implements "XQuery Server Pages", a mechanism that
 *  uses XML Query as a template language for dynamic Web Applications.
 *  <p>Technically, this servlet executes a query and serializes the resulting
 *  tree to the servlet output stream, typically in HTML or XHTML.
 *  <p>Optionally, it can "pipe" the evaluated XML tree into a XSLT stylesheet,
 *  which in turn writes to the HTTP stream, thus providing the "View" 
 *  functionality of a MVC (Model, View, Controller) processing scheme.
 *  <p>The servlet manages a cache of XQuery pages, and loads/reloads pages as
 *  needed.
 */
public class XQServlet extends HttpServlet implements ErrorListener
{
    int    cacheSize, templCacheSize;
    ArrayList cache;		// compiled query cache
    ArrayList templateCache;	// loaded XSLT template cache
    XQueryProcessor master;
    boolean logQueryCache = false, logTemplatesCache = false;
    boolean logRequests = false;

    int CHECK_DELAY = 1000; // interval in ms between two checkings of query changes
    TransformerFactory transformerFactory;
    ServletLog templateErrLog;


    static class CacheSlot {
	String path;
	Object loaded;
	long   loadTimeStamp;	// to check if it needs reloading
	long   checkTimeStamp;	// to avoid checking the date too often
    }

    // namespace/prefix used for accessing the current page context (class PageEnv)
    // functions with ns prefix 'xqsp' use a hidden variable as auto 1st arg:
    static String SERVLET_CLASS = PageEnv.class.getName();
    static String SERVLET_PREFIX = "xqsp";
    static QName  SERVLET_VARNAME = QName.get("@page@");   // inaccessible name
    static String SERVLET_NS = bindingWithAutoArg( SERVLET_CLASS, SERVLET_VARNAME);

    // functions with ns prefix 'request' use variable $request as auto 1st arg:
    static String REQUEST_CLASS = "javax.servlet.http.HttpServletRequest";
    static String REQUEST_PREFIX = "request";
    static QName  REQUEST_VARNAME = QName.get(SERVLET_NS, "request");
    static String REQUEST_NS = bindingWithAutoArg( REQUEST_CLASS, REQUEST_VARNAME );

    // functions with ns prefix 'response' use variable $response as auto 1st arg:
    static String RESPONSE_CLASS = "javax.servlet.http.HttpServletResponse";
    static String RESPONSE_PREFIX = "response";
    static QName  RESPONSE_VARNAME = QName.get(SERVLET_NS, "response");
    static String RESPONSE_NS =
        bindingWithAutoArg( RESPONSE_CLASS, RESPONSE_VARNAME );

    static String COMPILATION_ERROR = "<html><body>"+
        "<h1 style='color: red'>XQuery Server compilation errors:</h1><pre>";
    static String TEMPLATE_ERROR = "<html><body>"+
        "<h1 style='color: red'>XQuery Server XSLT template errors:</h1><pre>";
    static String EXECUTION_ERROR =
        "<h1 style='color: red'>XQuery Server execution messages:</h1><pre>";
    static QName SERIALIZATION = QName.get("qizx:serialization");
    static QName SERIALIZE = QName.get("x:serialize");
    static QName TRANSFORM = QName.get("x:transform");

    private static String bindingWithAutoArg( String className, QName global ) {
	return "java:" + className + "?" + global.getLocalName() + '=' + global.getURI();
    }

    public void init() {
	logRequests = "true".equals(getInitParameter("log-requests"));
	logQueryCache = "true".equals(getInitParameter("log-query-cache"));
	logTemplatesCache =
	    "true".equals(getInitParameter("log-templates-cache"));
	String qcs = getInitParameter("query-cache-size");
	cacheSize = (qcs == null )? 8 : Integer.parseInt(qcs);
	String tcs = getInitParameter("templates-cache-size");
	templCacheSize = (tcs == null )? cacheSize : Integer.parseInt(tcs);
	cache = new ArrayList();
	templateCache = new ArrayList();

    }

    public void doGet( HttpServletRequest request, HttpServletResponse response)
	throws ServletException, IOException {

	PageEnv env = new PageEnv(request, response);
        PrintWriter out = response.getWriter();
        response.setContentType("text/html");
	if(logRequests) {
	    String q = request.getQueryString();
	    log(request.getRequestURI()+ " " +
		(q == null? "" : q.substring(0, Math.min(100, q.length()))));
	}
	String path = request.getServletPath();
	XQuery q = loadQuery(path, out);	// from cache or compiled on the fly
	if(q == null)
	    return;
	// create a processor for execution (always SAX but can be used normally):
	SAXXQueryProcessor processor = new SAXXQueryProcessor(master);

	ServletLog log = new ServletLog(out);
	log.header = EXECUTION_ERROR;
	processor.setLog(log);
	// initialize globals:
	processor.initGlobal(SERVLET_VARNAME, env);
	processor.initGlobal(REQUEST_VARNAME, request);
	processor.initGlobal(RESPONSE_VARNAME, response);
	// Security: check and enable classes that are allowed for Java extension
	String allowed = getInitParameter("allowed-classes");
	if(allowed != null) {
	    StringTokenizer tok = new StringTokenizer(allowed);
	    for( ; tok.hasMoreTokens(); ) {
		String className = tok.nextToken();
		processor.authorizeClass(className);
		
	    }
	}

	try {
	    XMLSerializer serializer = new XMLSerializer();
	    serializer.setOutput(out);
	    serializer.setOption("indent", "no"); // default
	    serializer.setOption("method", "HTML"); // default

	    // examine pragmas: may specify serialization options or XSLT transform
	    String stylesheet = null;
	    Pragma pragmas[] = q.getPragmas(), transformPragmas = null;
	    for(int p = 0; p < pragmas.length; p++) {
		if(pragmas[p].name == SERIALIZATION || pragmas[p].name == SERIALIZE) {
		    Pragma.Iterator it = pragmas[p].contentIterator();
		    for(; it.next(); )
			serializer.setOption(it.getAttrName(), it.getAttrValue());
		}
		else if(pragmas[p].name == TRANSFORM) {
		    transformPragmas = pragmas[p];
		    Pragma.Iterator it = pragmas[p].contentIterator();
		    for(; it.next(); )
			if(it.getAttrName().equals("stylesheet"))
			    stylesheet = it.getAttrValue();
		}		
	    }
	    if(stylesheet == null)
		// simple serialization of the output
		processor.executeQuery(q, serializer);
	    else {
		// execution piped with a XSLT transformation:
		Templates sheet = loadTemplates(stylesheet, out);
		if(sheet == null)
		    return;
		Transformer transfo = sheet.newTransformer();
		processor.setQuery(q);
		Pragma.Iterator it = transformPragmas.contentIterator();
		for(; it.next(); ) {
		    String name = it.getAttrName(), value = it.getAttrValue();
		    if(!name.equals("stylesheet"))
			transfo.setParameter(name, value);
		}
		// some XSLT processors are pretty boring: dont accept dummy input
		InputSource dummySource = new InputSource(new StringReader("<dummy/>"));
		transfo.transform( new SAXSource(processor, dummySource),
				   new StreamResult(out) );
	    }
	}
	catch (EvalException ev) {
	     ev.printStack(log, 20);
	     if(ev.getCause() != null) {
		 out.print("Caused by: ");
		 ev.getCause().printStackTrace(out);
	     }
	     //else ev.printStackTrace(out);
	}
	catch (Exception xe) {
	    throw new ServletException("servlet error", xe); 
	}
    }

    public void doPost( HttpServletRequest request, HttpServletResponse response)
	throws ServletException, IOException {
	doGet(request, response);
    }

    // find query in the cache ? if not, load it:
    XQuery loadQuery( String path, PrintWriter msgOut ) throws IOException {

	URL url = getServletContext().getResource(path);
	if(url == null)
	    throw new IOException("query not found: "+path);
	for(;;) {
	    CacheSlot e = null;
	    boolean inCharge = false;
	    synchronized (cache) {
		int c = cache.size();
		for( ; --c >= 0; )
		    if( (e = (CacheSlot) cache.get(c)) != null && e.path.equals(path))
			break;
		if(c < 0) {
		    // not found: insert a dummy slot to avoid multiple compilations
		    e = new CacheSlot();
		    e.path = path;
		    cache.add(0, e);
		    inCharge = true;
		    if(logQueryCache) log(path+" not found in cache");
		}
		else {
		    if(c > 0) {	// put at head
			cache.remove(c);
			cache.add(0, e);
		    }
		    // check modification
		    if(e.loaded != null) {
			long now = System.currentTimeMillis();
			// dont check each time: if checked very recently, assume it is OK
			if(e.checkTimeStamp + CHECK_DELAY > now) {
			    if(logQueryCache) log("cache hit: "+url);
			    return (XQuery) e.loaded;
			}
			long lastModified = url.openConnection().getLastModified();
			e.checkTimeStamp = now;
			if(lastModified <= e.loadTimeStamp) {
			    if(logQueryCache) log("cache hit : "+url);
			    return (XQuery) e.loaded;
			}
			if(logQueryCache) log(path+"modified: "+url);
			inCharge = true;	// have to reload it
		    }
		    // if not inCharge: another thread is currently compiling it.
		    // We dont want to wait here, to avoid locking the cache
		}
	    }
	    // query not yet available
	    if(!inCharge) {
		// another thread is compiling it. just wait
		if(logQueryCache) log(path+" wait for completion");
		try {
		    synchronized(e) {
			e.wait();
		    }
		} catch(InterruptedException ex) { }
		continue;	// check again
	    }
	    // try to load:
	    makeMaster();
	    InputStream input = url.openStream();
	    XQuery query = null;
	    ServletLog log = new ServletLog(msgOut);
	    try {
		query = master.compileQuery( new InputStreamReader(input), 
					     url.toString(), log);
	    }
	    catch (XQueryException xex) {
		synchronized(cache) {
		    cache.remove(e);
		}
		synchronized(e) {
		    e.notify();		// unblock waiting threads
		}
		msgOut.println("</pre><h2 style='color: red'>*** "+
			       xex.getMessage()+"</h2>");
	    }
	    input.close();
	    if(logQueryCache) log(query == null ? (path+" failed") : (path+" load done"));

	    if(query == null)
		synchronized(cache) {
		    e.loaded = null;	// otherwise we would return the former query
		    cache.remove(e);
		}
	    else synchronized(e) {
		e.loadTimeStamp = e.checkTimeStamp = System.currentTimeMillis();
		e.loaded = query;
		e.notify();	// unblock waiting threads
	    }
	    synchronized(cache) {
		if(cache.size() > cacheSize)
		    cache.remove(templateCache.size() - 1);
	    }
	    return (XQuery) e.loaded;
	}

    }

    // cached loading of XSLT templates
    Templates loadTemplates( String path, PrintWriter msgOut ) throws IOException {
	URL url = getServletContext().getResource(path);
	if(url == null)
	    throw new IOException("stylesheet not found: "+path);
	// simpler implementation: block on compilation
	synchronized(templateCache) {
	    CacheSlot e = null;
	    int c = templateCache.size();
	    for( ; --c >= 0; )
		if( (e = (CacheSlot) templateCache.get(c)) != null && e.path.equals(path))
		    break;
	    if(c >= 0) {
		// found: check modification
		long now = System.currentTimeMillis();
		// dont check each time: if checked very recently, assume it is OK
		if(e.checkTimeStamp + CHECK_DELAY > now) {
		    if(logTemplatesCache) log("T-cache hit: "+url);
		    return (Templates) e.loaded;
		}
		long lastModified = url.openConnection().getLastModified();
		e.checkTimeStamp = now;
		if(lastModified <= e.loadTimeStamp) {
		    if(logTemplatesCache) log("T-cache hit : "+url);
		    return (Templates) e.loaded;
		}
		if(logTemplatesCache) log(path+"modified: "+url);
		templateCache.remove(e);
	    }
	    // load or reload:
	    if(logTemplatesCache) log("load templates: "+path);
	    e = new CacheSlot();
	    e.path = path;
	    if(transformerFactory == null) {
		transformerFactory = TransformerFactory.newInstance();
		transformerFactory.setErrorListener(this);
	    }
	    templateErrLog = new ServletLog(msgOut);
	    templateErrLog.header = TEMPLATE_ERROR;
	    try {
		Templates templ = transformerFactory.newTemplates (
		    new StreamSource( getServletContext().getResourceAsStream(path) ));
		e.loaded = templ;
	    }
	    catch (TransformerConfigurationException ex) {
		templateErrLog.println("template compilation error: "+ex); //.getMessage()
		return null;
	    }
	    e.loadTimeStamp = e.checkTimeStamp = System.currentTimeMillis();
	    templateCache.add(0, e);
	    if(templateCache.size() > templCacheSize)
		templateCache.remove(templateCache.size() - 1);
	    return (Templates) e.loaded;
	}
    }

    // -------------- ErrorListener

    public void warning(TransformerException e) 
	throws TransformerException {
	log("template warning "+e.getMessage());
	templateErrLog.println("*** warning: "+ e.getMessage());
    }

    public void error(TransformerException e) 
	throws TransformerException {
	log("template error "+e.getMessage());
	templateErrLog.println("*** error: "+ e.getMessage());
    }

    public void fatalError(TransformerException e) 
	throws TransformerException {
	templateErrLog.println("*** fatal error: "+ e.getMessage());
	// Just in case.
	throw e;
    }

    synchronized void makeMaster() throws IOException {
	if(master != null)
	    return;
	// create a master XQueryProcessor, used for compilation
	String root = getServletContext().getResource("/").toString();
	log("XQSP base-uri: "+root);
	master = new XQueryProcessor(root, root);

	// predefine global variables and NS:
	master.predefineNamespace(SERVLET_PREFIX, SERVLET_NS);
	master.predefineNamespace(REQUEST_PREFIX, REQUEST_NS);
	master.predefineNamespace(RESPONSE_PREFIX, RESPONSE_NS);

	master.predefineGlobal(SERVLET_VARNAME, Type.WRAPPED_OBJECT);
	master.predefineGlobal(REQUEST_VARNAME, Type.WRAPPED_OBJECT);
	master.predefineGlobal(RESPONSE_VARNAME, Type.WRAPPED_OBJECT);

	master.authorizeClass("XQServlet$PageEnv");
	master.authorizeClass("javax.servlet.http.HttpServlet");
	master.authorizeClass("javax.servlet.http.HttpServletRequest");
	master.authorizeClass("javax.servlet.http.HttpServletResponse");
	master.authorizeClass("javax.servlet.http.HttpSession");
	master.authorizeClass("javax.servlet.http.Cookie");
    }

    //	A Log that writes both to the container and to the servlet
    //
    public class ServletLog extends Log {
	boolean atStart = true;
	String header = COMPILATION_ERROR;

	ServletLog( PrintWriter out ) {
	    super(out);
	}

	protected void println( String s ) {
	    if(atStart)
		super.println(header);
	    atStart = false;
	    super.println( escapeML(s) );
	    log(s);
	}
    } // end of class ServletLog

    private static String escapeML(String s) {
	StringBuffer buf = new StringBuffer(s.length() + s.length()/8);
	for(int c = 0, C = s.length(); c < C; c++) {
	    char ch = s.charAt(c);
	    if(ch == '<')
		buf.append("&lt;");
	    else if(ch == '&')
		buf.append("&amp;");
	    else
		buf.append(ch);
	}
	return buf.toString();
    }

    private static String[] strings(Enumeration enu) {
	Vector v = new Vector();
	while(enu.hasMoreElements())
	    v.add(enu.nextElement());
	return (String[]) v.toArray(new String[v.size()]);
    }

    
    //
    //	Support of functions relative to the current page.
    //
    public class PageEnv {
	HttpServletRequest request;
	HttpServletResponse response;
	HttpSession session;
	ServletContext appContext;
	HashMap attributes;

	PageEnv(HttpServletRequest request, HttpServletResponse response) {
	    this.request = request;
	    this.response = response;
	    session = request.getSession();
	    appContext = getServletContext();
	}
	/**
	 * Convenience for request.getHeaderNames(): returns a list of Strings.	
	 */
	public String[] headerNames() {
	    return strings(request.getHeaderNames());
	}
	/**
	 * Convenience for request.getParameterNames(): returns a list of Strings.
	 */
	public String[] parameterNames() {
	    return strings(request.getParameterNames());
	}
	/**
	 * Convenience for request.getInitParameterNames(): returns a list of Strings.
	 */
	public String[] initParameterNames() {
	    return strings(getInitParameterNames());
	}

	public String initParameter(String name) {
	    return getInitParameter(name);
	}

	public URL getResource(String name) throws MalformedURLException {
	    return appContext.getResource(name);
	}
	/**
	 *	Loads the text value of a resource.
	 */
	public String getResourceAsString(String name) throws IOException {
	    URL res = getResource(name);
	    return (res == null)? null : FileUtil.loadString(res);
	}
	/**
	 *	Forward request to another servlet: returns true if no error.
	 */
	public boolean forward(String path) throws IOException {
	    RequestDispatcher rqd = appContext.getRequestDispatcher(path);
	    System.err.println("path "+path+" "+rqd);
	    try {
		if(rqd != null)
		    rqd.forward(request, response);	
		return true;
	    }
	    catch (Exception e) {
		System.err.println("*** forward: "+e);
	    }
	    return false;
	}
	/**
	 *	Gets an attribute of this page.
	 */
	public Object getAttribute(String name) {
	    return attributes == null ? null : attributes.get(name);
	}
	/**
	 *	Sets an attribute of this page.
	 */
	public void setAttribute(String name, Object value) {
	    if(attributes == null)
		attributes = new HashMap();
	    attributes.put(name, value);
	}
	/**
	 *	Removes an attribute from this page.
	 */
	public void removeAttribute(String name) {
	    if(attributes != null)
		attributes.remove(name);
	}
	/**
	 *	Gets a scoped attribute.
	 */
	public Object getAttribute( String id, String scope ) {
	    if(scope.equals("page")) {
		return getAttribute(id);
	    }
	    else if(scope.equals("request")) {
		return request.getAttribute(id);
	    }
	    else if(scope.equals("session")) {
		if(session == null)
		    return null;
		synchronized(session) {
		    return session.getAttribute(id);
		}
	    }
	    else if(scope.equals("application")) {
		synchronized(appContext) {
		    return appContext.getAttribute(id);
		}
	    }
	    throw new IllegalArgumentException("bad scope: "+scope);
	}
	/**
	 *  Gets a scoped attribute which is an instantiated Bean.
	 *  <p>If the bean not yet instantiated, the class name is used for instantiation.
	 *  @param id name of the attribute
	 *  @param beanClass fully qualified name of the class implementing the Bean.
	 *  @param scope the scope in which the bean instance is stored: can be
	 *  "page" (local to the current page), "request", "session" or "application".
	 */
	public Object useBean( String id, String beanClass, String scope )
	    throws Exception {
	    Object obj = null;
	    if(scope.equals("page")) {
		obj = getAttribute(id);
		if(obj == null)
		    setAttribute(id, obj = instantiate(beanClass));
	    }
	    else if(scope.equals("request")) {
		obj = request.getAttribute(id);
		if(obj == null)
		    request.setAttribute(id, obj = instantiate(beanClass));
	    }
	    else if(scope.equals("session")) {
		if(session == null)
		    throw new RuntimeException("no session");
		synchronized(session) {
		    obj = session.getAttribute(id);
		    if(obj == null)
			session.setAttribute(id, obj = instantiate(beanClass));
		}
	    }
	    else if(scope.equals("application")) {
		synchronized(appContext) {
		    obj = appContext.getAttribute(id);
		    if(obj == null)
			appContext.setAttribute(id, obj = instantiate(beanClass));
		}
	    }
	    else
		throw new IllegalArgumentException("bad scope: "+scope);
	    return obj;
	}

	private Object instantiate( String className ) throws Exception {
	    return java.beans.Beans.instantiate(getClass().getClassLoader(), className);
	}
    }

} // end of class XQServlet
