/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.util;

import java.io.IOException;

/**
 *	A set of (positive) integer identifiers. 
 *	A bit optimized for sparse sets.
 */
public class BitIdSet extends IdSet
{
    // use bytes as cell:
    final static int SHIFT = 3, BITS = 1 << SHIFT, MASK = BITS - 1;
    final static int INCREMENT = 4;
    final static int INCBITS = (1 << SHIFT) * INCREMENT, INCMASK = INCBITS - 1;

    //TODO allocate by chunks of 4 ou 8 bytes or in proportion of size.
    int   minId;	// always multiple of BITS (aligned)
    int   idSpan;
    byte[] bits;

    public BitIdSet() {
	idSpan = 0;
    }

    public BitIdSet(int id) {
	idSpan = 0;
	add(id);
    }

    public boolean test( int id ) {
	int bit = id - minId;
	if(bit < 0 || bit >= idSpan)
	    return false;
	return (bits[ bit >> SHIFT ] & (1 << (bit & MASK))) != 0;
    }

    public IdSet copy() {
	BitIdSet clone = new BitIdSet();
	clone.minId = minId;
	clone.idSpan = idSpan;
	if(bits != null) {
	    clone.bits = new byte[ bits.length ];
	    System.arraycopy(bits, 0, clone.bits, 0, bits.length);
	}
	return clone;
    }

    public void add( int id ) {
	if(idSpan == 0)	{ // empty: take as minId
	    minId = id & ~INCMASK;	// truncate
	    bits = new byte[INCREMENT];
	    idSpan = INCBITS;
	}
	
	int bit = id - minId;
	if(bit < 0 || bit >= idSpan) {
	    // need to resize:
	    byte[] old = bits;
	    int ext = 1 + ((bit < 0) ? -bit : (bit - idSpan)) / BITS;	// in units
	    int byteExt = ((ext + INCREMENT - 1) / INCREMENT) * INCREMENT;
	    bits = new byte[bits.length + byteExt];
	    if(bit < 0) {
		System.arraycopy(old, 0, bits, byteExt, old.length);
		minId -= byteExt * BITS;
		bit = id - minId;		// minId changed: need to adjust
	    }
	    else {
		System.arraycopy(old, 0, bits, 0, old.length);
	    }
	    idSpan += byteExt * BITS;
	}
	bits[ bit >> SHIFT ] |= (1 << (bit & MASK));
    }

    public void remove( int id ) {
	int bit = id - minId;
	if(bit >= 0 && bit < idSpan)
	    bits[ bit >> SHIFT ] &= ~(1 << (bit & MASK));
    }

    /**
     *	Merges an IdSet with this set.
     */
    public IdSet unionWith( BitIdSet other ) {
	if(other.bits == null)
	    return this;
	// adjust bit array:
	int min = Math.min(minId, other.minId);
	int max = Math.max(minId + idSpan, other.minId + other.idSpan);
	if(min < minId || max > minId + idSpan) {
	    byte[] nbits = new byte[ (max - min) / BITS ];
	    if(bits != null)
		System.arraycopy(bits, 0, nbits, (minId - min) / BITS, bits.length);
	    bits = nbits;
	    minId = min;
	    idSpan = max - min;
	}
	// OR with other set:
	for(int b = 0; b < other.bits.length; b++)
	    bits[ b + (other.minId - min) / BITS ] |= other.bits[b];
	return this;
    }

    /**
     *	Iteration mechanism: returns the first id in the set that is >= to argument.
     */
    public int getNext( int id ) {
	int bit = id - minId;
	if(bit < 0)
	    bit = 0;
	int cell = bit >> SHIFT;
	bit -= cell * BITS;
	for( ; cell < idSpan / BITS; bit = 0, cell ++) {
	    byte byt = bits[cell];
	    if(byt == 0)
		continue;
	    for( ; bit < BITS; bit++)
		if( (byt & (1 << bit)) != 0)
		    return minId + cell * BITS + bit;
	}
	return -1;
    }

    final static int[] counts = { 0, 1, 1, 2,  1, 2, 2, 3,  1, 2, 2, 3,  2, 3, 3, 4 };

    public int size() {
	if(bits == null)
	    return 0;
	int sz = 0;
	for( int c = bits.length; --c >= 0; ) {
	    byte b = bits[c];
	    sz += counts[b & 0xf] + counts[(b >> 4) & 0xf];
	}
	return sz;
    }

    public void save( OutputByteStream output ) throws IOException {
	output.putVint(minId);
	output.putVint(idSpan);
	if(bits != null)
	    output.putBytes(bits);
    }

    public void load( InputByteStream input ) throws IOException {
	minId = input.getVint();
	idSpan = input.getVint();
	bits = new byte[idSpan / BITS];
	if(input.getBytes(bits) != bits.length)
	    throw new IOException("truncated IdSet");
    }


    public void dump() {
	System.out.println("= min "+minId+" span "+idSpan+" size "+size());
	if(bits == null)
	    return;
	for(int c = 0; c < bits.length; c++)
	    System.out.print(' '+Integer.toHexString(bits[c] & 0xff));
	System.out.println();
    }
/*
    static public void main( String args[] )
    {
        try {
	    BitIdSet set = new BitIdSet();
	    set.dump();
	    set.add(133); set.add(130); set.dump();
	    set.add(128); set.add(127); set.dump();
	    set.remove(130); set.remove(100); set.dump();
	    set.add(120);
	    set.add(96);
	    for(int j = 234; j <= 267; j++) set.add(j);
	    set.dump();

            int id;
	    id = 120; System.err.println("id "+id+" "+set.test(id));
	    id = 130; System.err.println("id "+id+" "+set.test(id));
	    id = 133; System.err.println("id "+id+" "+set.test(id));

	    id = 0;
	    for( ; (id = set.getNext(id)) >= 0; ++id)
		System.out.print(" + "+id);
	    System.out.println();

		System.out.println("------- union ");
	    BitIdSet set2 = new BitIdSet();
	    set2.add(51);
	    set2.add(291);
	    set.unionWith(set2);
	    set.dump();
	    id = 0;
	    for( ; (id = set.getNext(id)) >= 0; ++id)
		System.out.print(" +"+id);
	    System.out.println();
	    IdSet set3 = set.copy();
	    id = 0;
	    for( ; (id = set3.getNext(id)) >= 0; ++id)
		System.out.print(" +"+id);
	    System.out.println();
	    
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
*/
} // end of class IdSet

