/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.lang.reflect.*;
import java.io.PrintStream;

/**
 *	Command line option analyzer.
 *	<p>
 */
public class CLOptions
{
    String appName;
    Option[] options = new Option[0];
    Option[] arguments = new Option[0];	// anonymous options
    String[] args;
    int argp, anonymous;

    /**
     *	Gets the following argument to a String field.  */
    public final static int NEXT_ARG = 1;
    /**
     *	Gets the end of the argument to a String field.  */
    public final static int STICKY_ARG = NEXT_ARG + 1;
    /**
     *	Sets a boolean field (no argument).  */
    public final static int SET_ARG = STICKY_ARG + 1;
    /**
     *	Resets a boolean field (no argument).  */
    public final static int RESET_ARG = SET_ARG + 1;
    /**
     *	Gets all following arguments to a String array field.  */
    public final static int ALL_ARG = RESET_ARG + 1; // 
    /**
     *	Prints help.  */
    public final static int HELP_ARG = ALL_ARG + 1; // 

    protected static class Option {
	String key, field;
	int type;
	String help;

	Option( String key, String field, int type, String help) {
	    this.key = key; this.field = field; this.type = type; this.help = help;
	}
    }

    public static class Exception extends java.lang.Exception {   }

    /**
     *	Creation with an application name.
     *	<p>Options must then be defined by declare().
     *	@param appName name of the application
     */
    public CLOptions( String appName ) {
	this.appName = appName;
    }

    /**
     *	Declares an option.
     *	<p>
     *	@param key appearance of the option switch. A null key means an anonymous argument.
     *	an anonymous argument may be repeatable if the type is ALL_ARG.
     *	@param field name of a Java field of the 'storage' argument of next():
     *	receives the value. Its type is used for automatic conversion.
     *	A field ALL_ARG must be of type String[].
     *	@param type values: NEXT_ARG, SET_ARG, RESET_ARG, ALL_ARG, HELP_ARG.
     *	@param help option help description.
     */
    public void declare(String key, String field, int type, String help) {
	Option[] old = options;
	options = new Option[ old.length + 1 ];
	System.arraycopy(old, 0, options, 0, old.length);
	options[old.length] = new Option(key, field, type, help);
    }

    public void declare(String key, int type, String help) {
	String field = key;
	if(!Character.isJavaIdentifierStart(field.charAt(0)))
	    field = field.substring(1);
	declare(key, field, type, help);
    }

    /**
     *	Declares an argument (aka Anonymous Option).
     *	@param cardinality A value of 1 means that the argument must be present once,
     *	0 that the argument is optional, 2 or more that the argument may be repeated
     *	(in that case, each call to next will get the next value).
     *	Only the last argument may be optional or repeated.
     */
    public void argument(String key, String field, int cardinality, String help) {
	Option[] old = arguments;
	arguments = new Option[ old.length + 1 ];
	System.arraycopy(old, 0, arguments, 0, old.length);
	arguments[old.length] = new Option(key, field, cardinality, help);
    }	    

    /**
     *	Parses the command line.<p>
     *	The values of options are stored into the (public) fields of 'storage' that have
     *	the name specified in declare().
     *	<p>Checks
     */
    public void parse( String[] args, Object storage ) throws Exception {
	this.args = args;
	argp = 0;
	anonymous = 0;

	for( ; argp < args.length; ) {
	    String arg = args[argp++];
	    int op = lookup(arg);
	    if(op < 0) {	// simple argument
		if(anonymous >= arguments.length) {
		    if(arguments[arguments.length-1].type <= 1 )	// not repeatable
			error("too many arguments: "+arg);
		    anonymous = arguments.length - 1;
		}
		setField( storage, "argument", arguments[anonymous++].field, arg);
	    }
	    else switch(options[op].type) {
		case NEXT_ARG:
		    if(argp >= args.length) error(options[op].key+" requires an argument");
		    setField(storage, options[op].key, options[op].field, args[argp++]);
		    break;
		case STICKY_ARG:
		    String tail = arg.substring(options[op].key.length());
		    setField(storage, options[op].key, options[op].field, tail);
		    break;
		case SET_ARG:
		case RESET_ARG:
		    setField( storage, options[op].key, options[op].field,
			      options[op].type == SET_ARG? "1" : "0");
		    break;
		case ALL_ARG:
		    String[] nargs = new String[args.length - argp];
		    System.arraycopy(args, argp, nargs, 0, nargs.length);
		    setField( storage, options[op].key, options[op].field, nargs);
		    return;
		case HELP_ARG:
		    printHelp(System.err);
		    throw new Exception();
	    }
	}
	if(anonymous < arguments.length && arguments[arguments.length-1].type >= 1)
	    error("not enough arguments");
    }

    // looks up the option table.
    private int lookup( String arg ) {
	for(int op = options.length; --op >= 0; )
	    if( options[op].key.equals(arg) ||
		options[op].type == STICKY_ARG && arg.startsWith(options[op].key) )
		return op;
	return -1;
    }

    void error(String msg) throws Exception {
	System.err.println("*** "+ msg);	
	printHelp(System.err);
	throw new Exception();
    }

    public void printHelp( PrintStream out ) {
	out.print("usage: "+ appName +" [options]");
	for(int op = 0; op < arguments.length; op++) {
	    out.print(" " + arguments[op].key);
	    if(arguments[op].type == ALL_ARG)
		out.print("...");
	}
	out.println();
	if(arguments.length > 0) {
	    out.println(" arguments:");
	    for(int op = 0; op < arguments.length; op++) {
		out.print("  " + arguments[op].key);
		int ptr = arguments[op].key.length();
		for(; ptr < 12; ptr++)
		    out.print(' ');
		out.println(' '+arguments[op].help);
	    }
	}

	out.println(" options:");
	for(int op = 0; op < options.length; op++) {
	    if(options[op].help == null)
		continue;
	    int ptr = options[op].key.length();
	    out.print("  "+options[op].key);
	    String help = options[op].help;
	    if(options[op].type == NEXT_ARG) {
		out.print(" <"+options[op].field+'>');
		ptr += 3 + options[op].field.length();
	    }
	    else if(options[op].type == STICKY_ARG) {
		int pos = help.indexOf('!');
		if(pos >= 0) {
		    String compl = help.substring(0, pos);
		    out.print(compl);
		    ptr += compl.length();
		    help = help.substring(pos+1);
		}
	    }
	    for(; ptr < 22; ptr++)
		out.print(' ');
	    out.println(" "+help);
	}
    }

    void setField(Object target, String key, String field, Object value) throws Exception
    {
	Class fieldClass = null;
	try {
	    Field f = target.getClass().getField( field );
	    fieldClass = f.getType();

	    if (fieldClass == int.class) 
		f.setInt(target, Integer.parseInt((String) value));
	    else if (fieldClass == float.class)
		f.setFloat(target, Float.parseFloat((String) value));
	    else if (fieldClass == double.class)
		f.setDouble(target, Double.parseDouble((String) value));
	    else if (fieldClass == boolean.class)
		f.setBoolean( target, value.equals("1") 
			      || ((String) value).equalsIgnoreCase("yes")
			      || ((String) value).equalsIgnoreCase("true"));
	    else if(fieldClass.isArray()) {
		// only String[] supported:
		String[] old = (String[]) f.get(target), na = null;
		if(value instanceof String) {
		    na = new String[ old.length + 1 ];
		    System.arraycopy(old, 0, na, 0, old.length);
		    na[old.length] = (String) value;
		}
		else {
		    String[] nv = (String[]) value;
		    na = new String[ old.length + nv.length ];
		    System.arraycopy(old, 0, na, 0, old.length);
		    System.arraycopy(nv, 0, na, old.length, nv.length);
		}
		f.set( target, na );
	    }
	    else
		f.set( target, value );
	}
	catch(NoSuchFieldException e1) {
	    throw new RuntimeException("!!! no such field "+ field);
	}
	catch(java.lang.Exception e2) {
            e2.printStackTrace();
	    error("illegal value for "+key+ ": "+e2.getMessage());
	}
    }
} // end of class CLOptions


