/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.text.Collator;
import java.text.RuleBasedCollator;
import java.text.CollationElementIterator;
import java.util.Locale;

/**
 *	Utility class for manipulation of text Collators,
 *	providing collator-based search functions.
 */
public class Collations
{
    public static final String CODEPOINT =
        "http://www.w3.org/2003/05/xpath-functions/collation/codepoint";
    public static final String CODEPOINT_SHORT = "codepoint";

    /**
     *	gets a locale-based collator from an uri of the form 'll' or 'll-CC' where
     *	'll' is a ISO-639 language name and CC a ISO-3166 country name.
     */
    public static Collator getInstance( String uri ) {
	if(uri.equals(CODEPOINT) || uri.equals(CODEPOINT_SHORT))
	    return null;
	int dash = uri.indexOf('-');
	Locale loc = dash < 0 ? new Locale(uri)
	    : new Locale(uri.substring(0, dash), uri.substring(dash+1));
	return Collator.getInstance(loc);
    }

    final static int END = CollationElementIterator.NULLORDER;

    /**
     *	gets a locale-based collator from an uri of the form 'll' or 'll-CC' where
     *	'll' is a ISO-639 language name and CC a ISO-3166 country name.
     *	The uri can be followed by an anchor of the form '#primary', '#secondary' etc.
     *	that changes the strength of the collator.
     */
    public static Collator getInstanceWithStrength( String uri ) {
	if(uri == null)
	    return null;
	int sharp = uri.indexOf('#');
	String root = sharp < 0 ? uri : uri.substring(0, sharp);
	Collator coll = getInstance(root);
	if(uri.endsWith("#primary"))
	    coll.setStrength(Collator.PRIMARY);
	else if(uri.endsWith("#secondary"))
	    coll.setStrength(Collator.SECONDARY);
	if(uri.endsWith("#tertiary"))
	    coll.setStrength(Collator.TERTIARY);
	return coll;
    }

    /**
     *	Simple helper. If the collator is null, does codepoint comparison.
     */
    public static int compare( String s1, String s2, Collator collator ) {
	return Util.comparison( collator == null ?
				s1.compareTo(s2) : collator.compare(s1, s2) );
    }

    // expand a pattern into an array of collation elements
    private static int[] preCollate( String pattern, RuleBasedCollator coll ) {
	int[] tmp = new int[ pattern.length() ];
	int ptr = 0;
	CollationElementIterator it = coll.getCollationElementIterator(pattern);
	for(int n; (n = it.next()) != END; ) {
	    if(n == 0)	// CollationElementIterator emits spurious '0'
		continue;
	    if(ptr >= tmp.length) {
		int[] old = tmp;
		tmp = new int[ old.length * 2 ];
		System.arraycopy(old, 0, tmp, 0, old.length);
	    }
	    tmp[ptr++] = n;
	}
	//for(int i = 0; i < ptr; i++) System.out.print(" "+Integer.toHexString(tmp[i]));System.out.println(" precol");
	if(ptr == tmp.length)	// true in many cases
	    return tmp;
	int[] result = new int[ptr];
	System.arraycopy(tmp, 0, result, 0, ptr);
	return result;
    }

    static boolean tailMatch( CollationElementIterator it, int[] elements ) {
	for(int i = 1; i < elements.length; ) {
	    int e = it.next();
	    if(e == 0)
		continue;
	    if(e == END || e != elements[i])
		return false;
	    i++;
	}
	return true;
    }

    /**
     *	Returns the index of the first occurrence of pattern. 
     */
    public static int indexOf( String src, String pattern, Collator collator ) {

	if(collator == null)
	    return src.indexOf(pattern);
	RuleBasedCollator coll = (RuleBasedCollator) collator; // checked before
	int[] pat = preCollate(pattern, coll);
	CollationElementIterator it = coll.getCollationElementIterator(src);

	for(int e; (e = it.next()) != END; ) {
	    
	    if(e == 0 || e != pat[0])
		continue;
	    int pos = it.getOffset();
	    if(tailMatch(it, pat)) {
		it.setOffset(pos);
		it.previous();	// not necessarily same as pos - 1
		return it.getOffset();
	    }
	    // resume at saved pos:
	    it.setOffset(pos);
	}
	return -1;
    }

    public static boolean endsWith( String src, String pattern, Collator collator ) {

	if(collator == null)
	    return src.endsWith(pattern);
	RuleBasedCollator coll = (RuleBasedCollator) collator; // checked before
	int[] pat = preCollate(pattern, coll);
	CollationElementIterator it = coll.getCollationElementIterator(src);

	for(int e; (e = it.next()) != END; ) {
	    if(e == 0 || e != pat[0])
		continue;
	    int pos = it.getOffset();
	    if(tailMatch(it, pat))
		return (it.next() < 0);	// must be at end
	    // resume at saved pos:
	    it.setOffset(pos);
	}
	return false;
    }

} // end of class Collations

        
