/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.io.*;

/**
 *	
 */
public class InputByteStream 
{
    final static int BUFSIZE = 4096;
    byte[] data;
    int    size;
    int    ptr;
    InputStream in;

    public InputByteStream( File file ) throws FileNotFoundException {
	this(new FileInputStream(file));
    }

    public InputByteStream( InputStream in ) {
	this.in = in;
	data = new byte[BUFSIZE];
	size = ptr = 0;
    }

    public InputByteStream() { }

    public void restartOn(byte[] buffer, int size) throws IOException {
	data = buffer;
	this.size = size;
	ptr = 0;
    }

    public void close() throws IOException {
	in.close();
    }

    public int getByte() throws IOException {
	if(ptr >= size)
	    return getBlock();
	return data[ptr ++] & 0xff;
    }

    public int getBytes(byte[] buffer) throws IOException {
	return getBytes(buffer, 0, buffer.length);
    }

    public int getBytes(byte[] buffer, int start, int reqsize)
	throws IOException {
	int readsize = 0, L;
	
	for(; readsize < reqsize; ) {
	    if(ptr >= size) {
		L = in.read(data, 0, BUFSIZE);
		if(L <= 0)
		    return readsize;
		size = L;
		ptr = 0;
	    }
	    L = Math.min(size - ptr, reqsize - readsize);
            System.arraycopy(data, ptr, buffer, start, L);
	    ptr += L;
	    readsize += L;
	    start += L;
	}
	return readsize;
    }

    private int getBlock() throws IOException {
	if(in == null)
	    return -1;
	int L = in.read(data, 0, BUFSIZE);
	if(L <= 0)
	    return -1;
	size = L;
	ptr = 1;
	return data[0] & 0xff;
    }

    public int getVint() throws IOException {
	if(ptr <= size - 5)
	    return (int) decode();
	else
	    return (int) slowGetVlong();
    }

    public long getVlong() throws IOException {
 	if(ptr <= size - 9)
 	    return decode();
 	else
	    return slowGetVlong();
    }

    public double getDouble() throws IOException {
	long r = getByte();
	for(int i = 0; i < 7; i++)
	    r = (r << 8) | getByte();
	
	return Double.longBitsToDouble(r);
    }

    public char[] getChars() throws IOException {
	int L = getVint();
	char[] chars = new char[L];
	getChars(chars, L);
	return chars;
    }

    public void getChars(char[] chars, int length) throws IOException {
	if(ptr + 2 * length <= size) {
	    for(int i = 0; i < length; i++) {	
		int p = ptr + 2 * i;
		chars[i] = (char) (((int) data[p] << 8) + data[p + 1]);
	    }
	    ptr += 2 * length;
	}
	else
	    for(int i = 0; i < length; i++) {
		int c = getByte() << 8;
		chars[i] = (char) (c + getByte());
	    }
    }

    private long slowGetVlong() throws IOException {
	int b = getByte(), b2, b3, b4, b5, b6, b7;
	
	int bytesMore = 0;
	if(b < 128)
	    return b;
	switch( b & 0x7f ) {
	default:
	    // 14 bits
	    return ((b & 0x3f) << 8) | getByte();

        case 64: case 65: case 66: case 67: case 68: case 69: case 70: case 71: 
        case 72: case 73: case 74: case 75: case 76: case 77: case 78: case 79: 
        case 80: case 81: case 82: case 83: case 84: case 85: case 86: case 87: 
        case 88: case 89: case 90: case 91: case 92: case 93: case 94: case 95:
	    // 21 bits:
	    b2 = getByte();
	    b3 = getByte();
	    return ((b & 0x1f) << 16) | (b2 << 8) | b3;

        case 96: case 97: case 98: case 99:
	case 100: case 101: case 102: case 103: 
        case 104: case 105: case 106: case 107:
	case 108: case 109: case 110: case 111:
	    // 28 bits:
	    return slowGetBytes(b & 0x0f, 3);
        case 112: case 113: case 114: case 115:
	case 116: case 117: case 118: case 119:
	    // 35 bits:
	    return slowGetBytes(b & 0x07, 4);
        case 120: case 121: case 122: case 123:  // 42 bits:
	    return slowGetBytes(b & 0x03, 5);
	case 124:
	    return slowGetBytes(0, 6);	// 48 bits
	case 125:
	    return slowGetBytes(0, 7);	// 56 bits
	case 126:
	    return slowGetBytes(0, 8);	// 64 bits
	case 127:	    
	    throw new RuntimeException("bad header for Vlong");
	}
    }

    private long slowGetBytes(long prefix, int bytes) throws IOException {
	int shift = (bytes - 1) * 8;
	prefix <<= shift + 8;
	for(int i = 0; i < bytes; i++, shift -= 8)
	    prefix |= ((long) getByte()) << shift;
	return prefix;
    }

    public long decode() {
	byte b = data[ptr];
	
	if(b >= 0) {
	    ++ ptr; return b;
	}
	switch( b & 0x7f ) {
	default:
	    // 14 bits
	    ptr += 2;
	    return ((b & 0x3f) << 8) | (data[ptr - 1] & 0xff);

        case 64: case 65: case 66: case 67: case 68: case 69: case 70: case 71: 
        case 72: case 73: case 74: case 75: case 76: case 77: case 78: case 79: 
        case 80: case 81: case 82: case 83: case 84: case 85: case 86: case 87: 
        case 88: case 89: case 90: case 91: case 92: case 93: case 94: case 95:
	    // 21 bits:
	    ptr += 3;
	    return ((b & 0x1f) << 16)
		 | ((data[ptr-2] & 0xff) << 8)
		 | (data[ptr-1] & 0xff);

        case 96: case 97: case 98: case 99:
	case 100: case 101: case 102: case 103: 
        case 104: case 105: case 106: case 107:
	case 108: case 109: case 110: case 111:
	    // 28 bits:
	    ptr += 4;
	    return ((b & 0xf) << 24) | ((data[ptr-3] & 0xff) << 16) |
		   ((data[ptr-2] & 0xff) << 8) | (data[ptr-1] & 0xff);
        case 112: case 113: case 114: case 115:
	case 116: case 117: case 118: case 119:
	    // 35 bits:
	    return decodeBytes(b & 0x07, 4);
        case 120: case 121: case 122: case 123:  // 42 bits:
	    return decodeBytes(b & 0x03, 5);
	case 124:
	    return decodeBytes(0, 6);	// 48 bits
	case 125:
	    return decodeBytes(0, 7);	// 56 bits
	case 126:
	    return decodeBytes(0, 8);	// 64 bits
	case 127:	    
	    throw new RuntimeException("bad header for Vlong");
	}
    }

    private long decodeBytes(long prefix, int bytes) {
	int shift = (bytes - 1) * 8;
	prefix <<= shift + 8;
	for(int i = 1; i <= bytes; i++, shift -= 8)
	    prefix |= ((long)(data[ptr + i] & 0xff)) << shift;
	ptr += bytes + 1;
	return prefix;
    }
} // end of class InputByteStream
