/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.io.*;

/**
 *	An efficient BufferedOutputStream with support for more data types.
 */
public class OutputByteStream
{
    final static int BUFSIZE = 4096;
    byte[] data;
    int    size;
    OutputStream out;
    boolean trace = false;

    public OutputByteStream( File file ) throws FileNotFoundException {
	this(new FileOutputStream(file));
    }

    public OutputByteStream( OutputStream out ) {
	this.out = out;
	data = new byte[BUFSIZE];
    }

    public void putByte(int b) throws IOException {
	if(size >= BUFSIZE)
	    flush();
	data[size ++] = (byte) b;
    }

    public void putBytes(byte[] buf) throws IOException {
	putBytes(buf, 0, buf.length);
    }

    public void putBytes(byte[] buf, int start, int bsize) throws IOException {
	if(trace) System.err.println(" put bytes "+bsize);
	for(; bsize > 0; ) {
	    int L = Math.min(BUFSIZE - size, bsize);
            System.arraycopy(buf, start, data, size, L);
	    size += L;
	    if(size >= BUFSIZE)
		flush();
	    bsize -= L;
	    start += L;
	}
    }

    public void flush() throws IOException {
	out.write(data, 0, size);
	out.flush();
	size = 0;
    }

    public void close() throws IOException {
	flush();
	out.close();
    }

    /**
     *	Closes and synchronizes the file on disk.
     *	CAUTION: assumes the underlying OutputStream is a FileOutputStream.
     */
    public void syncClose() throws IOException {
	flush();
	((FileOutputStream) out).getFD().sync();
	close();
    }

    public void putVint( int code ) throws IOException {
	if(trace) System.err.println(" put int "+code);
	if(size <= BUFSIZE - 5)
	    size = encodeInt(code, data, size);
	else
	    slowPutVlong(code);
    }

    public void putVlong( long code ) throws IOException {
	if(trace) System.err.println(" put long "+code);
	
	if(size <= BUFSIZE - 9)
	    size = encodeLong(code, data, size);
	else
	    slowPutVlong(code);
	
    }

    public void  putDouble(double value) throws IOException {
	if(trace) System.err.println(" put double "+value);
	long r = Double.doubleToRawLongBits(value);
	
	if(size <= BUFSIZE - 8) {
	    data[size + 0] = (byte) (r >> 56);
	    data[size + 1] = (byte) (r >> 48);
	    data[size + 2] = (byte) (r >> 40);
	    data[size + 3] = (byte) (r >> 32);
	    data[size + 4] = (byte) (r >> 24);
	    data[size + 5] = (byte) (r >> 16);
	    data[size + 6] = (byte) (r >> 8);
	    data[size + 7] = (byte) r;
	    size += 8;
	}
	else
	    slowPutBytes(r, 8);
    }

    public void  putString(String s) throws IOException {
	putChars(s.toCharArray());	// OPTIM
    }

    //	fixed-width encoding (Unicode MSB first)
    //
    public void  putChars(char[] chars) throws IOException {
	if(trace) System.err.println(" put chars "+new String(chars));
	int L = chars.length;
	putVint(L);
	if(size + 2 * L <= BUFSIZE) {
	    for(int i = 0; i < L; i++) {	
		int p = size + 2 * i;
		data[p] = (byte) (chars[i] >> 8);
		data[p + 1] = (byte) chars[i];
	    }
	    size += 2 * L;
	}
	else
	    for(int i = 0; i < L; i++) {	
		int p = size + 2 * i;
		putByte( (byte) (chars[i] >> 8) );
		putByte( (byte) chars[i] );
	    }
    }

    private void slowPutVlong( long code ) throws IOException {
	if(code < 0)
	    throw new RuntimeException("negative code "+code);
	int bytesMore = 0;
	if(code < 128)
	    putByte( (int) code );
	else if(code < (1 << 14)) {
	    putByte( (int) ((code >> 8) | 0x80) );
	    bytesMore = 1;
	}
	else if(code < (1 << 21)) {
	    putByte( (int) ((code >> 16) | 0xc0) );
	    bytesMore = 2;
	}
	else if(code < (1 << 28)) {
	    putByte( (int) ((code >> 24) | 0xe0) ); 
	    bytesMore = 3;
	}
	else if(code < (1L << 35)) {
	    putByte( (int) ((code >> 32) | 0xf0) );
	    bytesMore = 4;
	}
	else if(code < (1L << 42)) {
	    putByte( (int) ((code >> 40) | 0xf8) );
	    bytesMore = 5;
	}
	else if(code < (1L << 48)) {
	    putByte( (int) 0xfc ); 
	    bytesMore = 6;
	}
	else if(code < (1L << 56)) {
	    putByte( (int) 0xfd ); 
	    bytesMore = 7;
	}
	else {
	    putByte( (int) 0xfe ); 
	    bytesMore = 8;
	}
	slowPutBytes(code, bytesMore);
    }

    private void slowPutBytes(long code, int bytes) throws IOException {
	int shift = (bytes - 1) * 8;
	for(int i = 0; i < bytes; i++, shift -= 8)
	    putByte( (int) (code >> shift) );
    }

    /**
     *	Stores a positive int in variable-length encoding. Assumes that the buffer
     *	is large enough.
     *	@return the new buffer size.
     */
    public static int encodeInt(int code, byte[] buffer, int bufSize) {
	if(code < 0)
	    throw new RuntimeException("negative code "+code);
	if(code < 128) {
	    buffer[bufSize] = (byte) code;
	    return bufSize + 1;
	}
	else if(code < (1 << 14)) {
	    buffer[bufSize] = (byte) ((code >> 8) | 0x80);
	    buffer[bufSize + 1] = (byte) code;
	    return bufSize + 2;
	}
	else if(code < (1 << 21)) {
	    buffer[bufSize] = (byte) ((code >> 16) | 0xc0);
	    buffer[bufSize + 1] = (byte) (code >> 8); 
	    buffer[bufSize + 2] = (byte) code;
	    return bufSize + 3;
	}
	else if(code < (1 << 28)) {
	    buffer[bufSize] = (byte) ((code >> 24) | 0xe0); 
	    return encodeBytes(code, 4, buffer, bufSize);
	}
	else {
	    buffer[bufSize] = (byte) 0xf0; 
	    return encodeBytes(code, 5, buffer, bufSize);
	}
    }

    public static int encodeLong(long code, byte[] buffer, int bufSize) {
	if(code < 0)
	    throw new RuntimeException("negative code "+code);
	if(code < (1 << 28))
	    return encodeInt((int) code, buffer, bufSize);
	if(code < (1L << 35)) {
	    buffer[bufSize] = (byte) ((code >> 32) | 0xf0);
	    return encodeBytes(code, 5, buffer, bufSize);
	}
	else if(code < (1L << 42)) {
	    buffer[bufSize] = (byte) ((code >> 40) | 0xf8);
	    return encodeBytes(code, 6, buffer, bufSize);
	}
	else if(code < (1L << 48)) {
	    buffer[bufSize] = (byte) 0xfc; 
	    return encodeBytes(code, 7, buffer, bufSize);
	}
	else if(code < (1L << 56)) {
	    buffer[bufSize] = (byte) 0xfd; 
	    return encodeBytes(code, 8, buffer, bufSize);
	}
	else {
	    buffer[bufSize] = (byte) 0xfe; 
	    return encodeBytes(code, 9, buffer, bufSize);
	}
    }

    // puts (byteCnt - 1) lower bytes. (total -> byteCnt)
    private 
    final static int encodeBytes( long code, int byteCnt, byte[] buffer, int bufSize) {
	
	int shift = (byteCnt - 2) * 8;
	for(int i = 1; i < byteCnt; i++, shift -= 8)
	    buffer[bufSize + i] = (byte) (code >> shift);
	return bufSize + byteCnt;
    }


} // end of class OutputByteStream
