/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util;

import java.util.HashMap;

/**
 *  Representation of a Qualified Name: Namespace + localName.
 *  <p>A QName has a unique representation: comparison can be performed with ==
 *  <p>This object has no notion of prefix.
 */
public final class QName
{
    private static HashMap names = new HashMap();
    private static QName probe = new QName(null, null);

    /** the xml:lang qualified name */
    public static final QName XML_LANG;
    /** the xml:space qualified name */
    public static final QName XML_SPACE;
    /** the xml:base qualified name */
    public static final QName XML_BASE;
    /** the xsi:type qualified name */
    public static final QName XSI_TYPE;
    /** the xsi:nil qualified name */
    public static final QName XSI_NIL;
    /** the xsi:schemaLocation qualified name */
    public static final QName XSI_SCHEMA_LOCATION;
    /** the xsi:noNamespaceSchemaLocation qualified name */
    public static final QName XSI_NO_NAMESPACE_SCHEMA_LOCATION;

    static {
	XML_LANG = get(Namespace.XML, "lang");
	XML_SPACE = get(Namespace.XML, "space");
	XML_BASE = get(Namespace.XML, "base");
	XSI_TYPE = get(Namespace.XSI, "type");
	XSI_NIL = get(Namespace.XSI, "nil");
	XSI_SCHEMA_LOCATION = get(Namespace.XSI, "schemaLocation");
	XSI_NO_NAMESPACE_SCHEMA_LOCATION =
	    get(Namespace.XSI, "noNamespaceSchemaLocation");
    }

    private Namespace namespace; 
    private String localName; 

    // ------------------------------------------------------------------------

    /**
     *	Obtains a unique representation of a QName from
     *	a namespace URI and a NCname.
     */
    public static synchronized QName get(String namespaceURI,
					 String localName) {
	return get(Namespace.get(namespaceURI), localName);
    }

    /**
     *	Obtains a unique representation of a QName from
     *	a Namespace object and a NCname.
     */
    public static synchronized QName get(Namespace namespace,
					 String localName) {
	if (namespace == null)
	    throw new IllegalArgumentException("null namespace");
	if (localName == null)
	    throw new IllegalArgumentException("null localName");

	probe.namespace = namespace;
	probe.localName = localName;
	QName name = (QName) names.get(probe);
	if (name == null) {
	    name = new QName(namespace, localName);
	    names.put(name, name);
	}
	return name;
    }

    /**
     *	Obtains a unique representation of a QName with empty namespace.
     */
    public static synchronized QName get(String localName) {
	return get(Namespace.NONE, localName);
    }

    public boolean equals(Object other) {
	if (other instanceof QName) {
	    QName n = (QName) other;
	    return (namespace == n.namespace && localName.equals(n.localName));
	}
	if (other instanceof String) {
	    return namespace == Namespace.NONE &&
		   localName.equals((String) other);
	}
	return false;
    }

    public int hashCode() {
	// identityHashCode returns 0 for null.
	return (System.identityHashCode(namespace) ^ localName.hashCode());
    }

    private QName(Namespace namespace, String localName) {
	this.namespace = namespace;
	this.localName = localName == null ? null : localName.intern();
    }

    /** Returns the namespace as an interned Namespace object */
    public Namespace getNamespace() {
	return namespace;
    }

    /** Returns the namespace URI as a String  */
    public String getURI() {
	return namespace.getURI();
    }

    /** Returns the localname as a String  */
    public String getLocalName() {
	return localName;
    }

    public String toString() {
	if (namespace == Namespace.NONE)
	    return localName;
	return '{' + namespace.getURI() + '}' + localName;
    }

    public int compareTo(QName other) {
	if (namespace == other.namespace)
	    return localName.compareTo(other.localName);
	else
	    return namespace.compareTo(other.namespace);
    }
}
