/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.util.time;

import net.xfra.qizxopen.util.Util;

/**
 *	Representation and manipulation of a Duration value.
 */
public class Duration
{
    private boolean negative;
    private int year;
    private int month;
    private int day;
    private int hour;
    private int minute;
    private double second;

    // -----------------------------------------------------------------------

    public static Duration parseDuration(String s)  throws DateTimeException {

	int charCount = s.length();
	if (charCount <= 1)
	    throw new DateTimeException("duration syntax", s);

	Duration duration = new Duration();

	int pos = 0;
	if (s.charAt(0) == '-') {
	    duration.negative = true;
	    ++pos;
	}

	// At least P0Y.
	if (pos + 2 >= charCount || s.charAt(pos) != 'P')
	    throw new DateTimeException("duration syntax", s);
	++pos;

	boolean isDate = false;
	boolean isTime = false;
	int timePos = s.indexOf('T');

	int nextPos = s.indexOf('Y', pos);
	if (nextPos > pos) {
	    duration.year = DateTimeBase.parseNonNegativeInt(s, pos, nextPos);
	    isDate = true;
	    pos = nextPos + 1;
	}

	nextPos = s.indexOf('M', pos);
	if (nextPos > pos && (timePos < 0 || nextPos < timePos)) {
	    duration.month = DateTimeBase.parseNonNegativeInt(s, pos, nextPos);
	    isDate = true;
	    pos = nextPos + 1;
	}

	nextPos = s.indexOf('D', pos);
	if (nextPos > pos) {
	    duration.day = DateTimeBase.parseNonNegativeInt(s, pos, nextPos);
	    isDate = true;
	    pos = nextPos + 1;
	}

	if (timePos >= 0) {
	    if (timePos == pos) {
		++pos;

		nextPos = s.indexOf('H', pos);
		if (nextPos > pos) {
		    duration.hour = 
			DateTimeBase.parseNonNegativeInt(s, pos, nextPos);
		    isTime = true;
		    pos = nextPos + 1;
		}

		nextPos = s.indexOf('M', pos);
		if (nextPos > pos) {
		    duration.minute = 
			DateTimeBase.parseNonNegativeInt(s, pos, nextPos);
		    isTime = true;
		    pos = nextPos + 1;
		}

		nextPos = s.indexOf('S', pos);
		if (nextPos > pos) {
		    duration.second = 
			DateTimeBase.parseNonNegativeDouble(s, pos, nextPos);
		    isTime = true;
		    pos = nextPos + 1;
		}
	    }

	    if (!isTime)
		throw new DateTimeException("duration syntax", s);
	}

	if (pos != charCount || !(isDate || isTime))
	    throw new DateTimeException("duration syntax", s);

	duration.normalize();
	return duration;
    }

    public Duration() {}

    public Duration( int year, int month, int day, int hour, int minute, double second) {
	setYear(year);
	setMonth(month);
	setDay(day);
	setHour(hour);
	setMinute(minute);
	setSecond(second);

	normalize();
    }

    /**
     *	Accepts negative values (should be of the same sign).
     */
    public Duration( int months, double seconds ) {
	negative = months < 0 || seconds < 0;
	year = 0;
	month = Math.abs(months);
	day = 0;
	hour = 0;
	minute = 0;
	second = Math.abs(seconds);
	normalize();
    }

    public void setNegative(boolean negative) {
	this.negative = negative;
    }

    public boolean isNegative() {
	return negative;
    }

    public void setYear(int year) {
	if (year < 0)
	    throw new IllegalArgumentException("negative year component");
	this.year = year;
    }

    public int getYear() {
	return year;
    }

    public void setMonth(int month) {
	if (month < 0)
	    throw new IllegalArgumentException("negative month component");
	this.month = month;
    }

    public int getMonth() {
	return month;
    }

    public void setDay(int day) {
	if (day < 0)
	    throw new IllegalArgumentException("negative day component");
	this.day = day;
    }

    public int getDay() {
	return day;
    }

    public void setHour(int hour) {
	if (hour < 0)
	    throw new IllegalArgumentException("negative hour component");
	this.hour = hour;
    }

    public int getHour() {
	return hour;
    }

    public void setMinute(int minute) {
	if (minute < 0)
	    throw new IllegalArgumentException("negative minute component");
	this.minute = minute;
    }

    public int getMinute() {
	return minute;
    }

    public void setSecond(double second) {
	if (second < 0)
	    throw new IllegalArgumentException("negative second component");
	this.second = second;
    }

    public double getSecond() {
	return second;
    }

    public String toString() {
	if (year == 0 && month == 0 && day == 0 &&
	    hour == 0 && minute == 0 && second == 0)
	    return "PT0S";

	StringBuffer buffer = new StringBuffer();

	if (negative)
	    buffer.append('-');

	buffer.append('P');

	if (year > 0) {
	    buffer.append(year);
	    buffer.append('Y');
	}
	if (month > 0) {
	    buffer.append(month);
	    buffer.append('M');
	}
	if (day > 0) {
	    buffer.append(day);
	    buffer.append('D');
	}

	if (hour > 0 || minute > 0 || second > 0) {
	    buffer.append('T');

	    if (hour > 0) {
		buffer.append(hour);
		buffer.append('H');
	    }
	    if (minute > 0) {
		buffer.append(minute);
		buffer.append('M');
	    }
	    if (second > 0) {
		buffer.append(DateTimeBase.SECOND_FORMAT.format(second));
		buffer.append('S');
	    }
	}

	return buffer.toString();
    }

    /**
     *	Checks the validity of the duration considered as a timezone.
     */
    public boolean checkAsTimezone() {
	return (getYear() == 0 && getMonth() == 0 &&
		getDay() == 0  && getSecond() == 0 &&
		getHour() >= -14 && getHour() <= 14 );
    }

    public int getMonths() {
	return (year * 12 + month) * (negative ? -1 : 1);
    }

    public double getSeconds() {
	return (day * 24 * 3600.0 + hour * 3600 + minute * 60 + second)
	       * (negative ? -1 : 1);
    }

    /**
     *	normalize the second, minute, hour, month fields.
     *	Day is not normalized.
     *  
     */
    public void normalize() {
	int carry = (int) (second / 60);
	second -= 60 * carry;

	minute += carry;
	carry = minute / 60;
	minute -= 60 * carry;

	hour += carry;
	carry = hour / 24;
	hour -= 24 * carry;

	day += carry;

	carry = month / 12;
	month -= 12 * carry;

	year += carry;
	
    }


    public int hashCode() {
	long D = Double.doubleToRawLongBits( getSeconds() + getMonths() * 365*86400 );
	return (int) ((D >>> 32) ^ D);
    }

    public boolean equals(Object other) {
	if (other == null || !(other instanceof Duration))
	    return false;
	Duration that = (Duration) other;
	return this.getSeconds() == that.getSeconds()
	    && 	this.getMonths() == that.getMonths();
    }

    /**
     *	An arbitrary comparison assuming that all months have the same duration.
     */
    public int compareTo( Duration that ) {
	int dm = this.getMonths() - that.getMonths();
	if(dm != 0)
	    return Util.comparison(dm);
	return Util.comparison( this.getSeconds() - that.getSeconds() );
    }
}
