/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.xquery.XQueryException;
import net.xfra.qizxopen.xquery.EvalException;
import net.xfra.qizxopen.xquery.fn.UserFunction;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.GlobalVariable;
import net.xfra.qizxopen.xquery.impl.*;

import net.xfra.qizxopen.dm.DocumentManager;
import net.xfra.qizxopen.util.QName;

import java.text.Collator;
import java.util.Date;
import java.io.PrintWriter;

/**
 *	XQuery Evaluation Context. (internal use)
 */
public interface EvalContext
{
    /**
     *	Returns the initial static context.
     */
    StaticContext getStaticContext();
    /**
     *	Returns the calling context, null if not inside a user function.
     */
    EvalContext   getCallerContext();

    /**
     *	Helper creating a context for the called function.
     */
    EvalContext subContext( UserFunction.Call called ) throws EvalException;

    /**
     *	Notes the current evaluation point for trace & debug.
     *	Can also raise an EvalException if a time limit is reached.
     */
    void  at( Expression pt ) throws XQueryException ;

    /**
     *	Returns the document or collection input.
     */
    Value getInput();
    /**
     *	Finds a document by URI. Ensures that a document is loaded once only.
     */
    Value getDocument( String uri ) throws XQueryException;

    /**
     *	Returns the document manager used by this context.
     */
    DocumentManager getDocumentManager();
 
    /**
     *	Defined externally (by default stdout or servlet output)
     */
    PrintWriter getDefaultOutput();

    /**
     *	Gets a message Log, either set explictly or from the static context.
     */
    Log getLog();

    // miscellaneous information:

    /**
     *	 Retrieves a named property from the context.
     */
    Object getProperty(String name);

    /**
     *	Finds a collation by its URI. If not found, an error is raised.
     *	The codepoint collator is represented by a null value.
     * @param uri collation URI. If null, the default collation is returned.
     */
    Collator getCollator( String uri ) throws XQueryException;

    /**
     *	Time at start of query evaluation.
     */
    Date     getCurrentDate();

    /**
     *	implicit time zone as a difference in minutes from GMT (or UTC).
     */
    int      getImplicitTimezone();

    /**
     *	Raises an EvalException that is located to this context.
     */
    Value error( Expression location, EvalException error ) throws EvalException;
    Value error( Expression location, String error ) throws EvalException;

    Expression getCurrentLocation();

    /**
     *	Prints the calling stack to a Log, with amaximum depth.
     */
    void  printStack(Log log, int maxDepth);

    /**
     *	Gets the value of a global variable.
     *	@throws EvalException if the variable has no specified value.
     */
    Value loadGlobal( GlobalVariable var ) throws XQueryException;

    /**
     *	Maximum number of registers for each scalar type (integer, double, string, item)
     */
    int MAX_REGISTER = 4;

    int INT_REGISTER = 0;
    int DOUBLE_REGISTER = INT_REGISTER + MAX_REGISTER;
    int STRING_REGISTER = DOUBLE_REGISTER + MAX_REGISTER;
    int ITEM_REGISTER = STRING_REGISTER + MAX_REGISTER;
    int LAST_REGISTER = ITEM_REGISTER + 2 * MAX_REGISTER;  // twice more item registers.

    /**
     *	Gets the value of a local variable.
     *	The address is computed by the compiler, it can be a simple rank or 
     *	a scalar register.
     */
    Value loadLocal( int address ) throws XQueryException;

    /** Loads a scalar integer local variable */
    long    loadLocalInteger( int address ) throws XQueryException;
    /** Loads a scalar double local variable */
    double  loadLocalDouble( int address ) throws XQueryException;
    /** Loads a scalar String local variable */
    String  loadLocalString( int address ) throws XQueryException;
    /** Loads a scalar local variable */
    Item    loadLocalItem( int address ) throws XQueryException;

    /**
     *	 Evaluates an expression and store the result into a local variable.
     *	 Used for user function calls, let.
     *	 Checks the type on the fly if dynamicType is not null.
     *	 @throws TypeException if improper check
     */
    void  storeLocal( int address, Expression expr, Type dynamicType,
		      Focus focus, EvalContext calling ) throws XQueryException;

    /**
     *   Stores a value or its current item to a local variable [for, some, every].
     *   @param current if true, stores only the current item, else stores
     *   the whole sequence.
     *	 @throws TypeException if improper check
     */
    void  storeLocal( int address, Value value, boolean current ) throws XQueryException;

    /**
     *   Stores the positional variable in [for, some, every].
     */
    void  storeLocalInteger( int address, long value );

} // end of interface EvalContext

