/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.util.Util;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.HashMap;
import java.util.ArrayList;
// XSLT:
import javax.xml.transform.*;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;

/**
 *   Compiles and caches XQuery modules on behalf of a XQueryProcessor.
 * <p>This is a simple implementation that assumes that all modules URIs can be
 * resolved using the same base location (any URL supported by Java).
 * If the physical location ("at") is not specified in the "import module"
 * declaration, then the module URI is used as follows: periods are replaced
 * by '/' and the extension '.xqm' is appended. For example if the base URL is
 * <code>http://myserver.net/xqmodules</code>, the declaration
 * <code>import module mod1 = "myapp.module1"</code> will resolve the module location
 * as <code>http://myserver.net/xqmodules/myapp/module1.xqm</code>
 * <p>
 *  Ensures that a module is loaded only once for the same query (i.e. two
 *  indirect references to a module from the same query must yield the same object).
 * <p>Can be shared by several XQueryProcessors in a multithreaded context.
 *	
 */
public class ModuleManager implements ErrorListener
{
    protected URL baseURL;
    private HashMap modules = new HashMap();

    private TransformerFactory xsltFactory;
    private ArrayList templateCache = new ArrayList();
    private int templateCacheSize = 3;	//TODO: config

    /**
     * Builds a Module Manager with a base location for
     * resolution of module URIs.
     */
    public ModuleManager( URL baseURL ) throws IOException {
	this.baseURL = baseURL;
    }

    /**
     * Builds a Module Manager with a base location for
     * resolution of module URIs.
     */
    public ModuleManager( String baseURI ) throws IOException {
	this.baseURL = Util.uriToURL(baseURI);
    }

    /**
     *	Overridable method for resolving a module name and location to
     *	an actual URL.
     *	The default method is use the base URL of the module manager and
     *	the physical URI of the module (specified by the "at" clause).
     *	If the physical URI is absent, it is produced from the namespace URI
     *	by replacing '.' by '/' and adding the ".xqm" extension.
     *	For example "package1.package2.module1" would be expanded into
     *	"&lt;base-url>/package1/package2/module1.xqm"
     */
    public URL resolveModuleLocation(String namespaceURI, String physicalURI)
	throws IOException {
	if(physicalURI == null)
	    physicalURI = namespaceURI.replace('.', '/') + ".xqm";
	//-System.err.println("resolveModuleLocation "+baseURL+" "+physicalURI);
	return new URL(baseURL, physicalURI);
    }

    /**
     *	Unloads all modules. Main Queries can still refer safely to the modules.
     *	<p>Note: Due to possible dependencies between modules, it would be very
     *	 difficult to unload a module selectively.
     */
    public synchronized void unloadAllModules() {
	modules = new HashMap();
    }

    /**	[used internally by the parser.]     */
    public synchronized Module loadModule( Parser caller,
					   String uri, String location, Log log)
	throws IOException, XQueryException {

	Module mo = (Module) modules.get(uri);
	if(mo == null) {
	    // actual loading:
	    URL url = resolveModuleLocation(uri, location);
	    String text = loadSource(url);
	    Parser p = new Parser(this);
	    // create the module and store it before parsing to avoid looping on
	    // cyclical module references:
	    modules.put(uri, mo = new Module());
	    // parse:
	    p.setupFrom(caller);
	    mo.setBaseURI(baseURL.toString());
	    p.parseLibraryModule( mo, text, url.toString(), log );
	}
	return mo;
    }

    String loadSource( URL url ) throws IOException {
	InputStream in = url.openStream();
        String source = null;
        try {
            source = loadSource( in );
        } finally {
            in.close();
        }
        return source;
    }

    String loadSource(InputStream stream) throws IOException {
	int bufsize = stream.available() + 1;
	char[] buf = new char[bufsize];
	int buflen = 0;
	InputStreamReader reader = new InputStreamReader(stream);
	while (true) {
	    int nread = reader.read(buf, buflen, buf.length - buflen);
	    if (nread < 0)
		nread = 0;
	    buflen = buflen + nread;
	    if (buflen < buf.length)
		break;
	    char[] newbuf = new char[buflen * 2];
	    System.arraycopy(buf, 0, newbuf, 0, buflen);
	    buf = newbuf;
	}
	reader.close();
	return new String(buf, 0, buflen);
    }

    // ---------------- XSLT -----------------------------------------------------

    static class CacheSlot {
	String path;
	Object loaded;
    }
 
    /**
     *	Sets the TransformerFactory used for XSLT transformations.
     */
    public void setXSLTFactory(TransformerFactory value) {
        xsltFactory = value;
    }

    /**
     *	Gets the TransformerFactory used for XSLT transformations.
     *	Useful for configuring the default factory.
     */
    public TransformerFactory getXSLTFactory() {
	if(xsltFactory == null) {
	    xsltFactory = TransformerFactory.newInstance();
	}
        return xsltFactory;
    }

   /**
     *	
     */
    Templates loadTemplates( String path )
	throws TransformerException {
	// simple implementation: block on compilation
	synchronized(templateCache) {
	    CacheSlot e = null;
	    int c = templateCache.size();
	    for( ; --c >= 0; )
		if( (e = (CacheSlot) templateCache.get(c)) != null &&
		    e.path.equals(path))
		    break;
	    if(c >= 0) {
		return (Templates) e.loaded;
	    }
	    // load or reload:
	    e = new CacheSlot();
	    e.path = path;
	    getXSLTFactory();
	    try {
		Source source = null;
		xsltFactory.setErrorListener(this);
                if (path.startsWith("http:") || path.startsWith("file:")) {
                    source = xsltFactory.getURIResolver().resolve(path, null);
                }
		else {
                    source = new StreamSource(path);
                }
		if (source == null)
		    throw new TransformerException("cannot find stylesheet "+
						   path);
		Templates templ = xsltFactory.newTemplates(source);
 		e.loaded = templ;
	    }
	    catch (TransformerConfigurationException ex) {
		throw new TransformerException(ex.getMessage(), ex.getCause());
	    }
	    templateCache.add(0, e);
	    if(templateCache.size() > templateCacheSize)
		templateCache.remove(templateCache.size() - 1);
	    
	    return (Templates) e.loaded;
	}
    }

    public void error(TransformerException exception) 
	throws TransformerException {
	throw  exception;
    }

    public void fatalError(TransformerException exception)
	throws TransformerException {
	throw  exception;
    }

    public void warning(TransformerException exception) { }
} 

