/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.util.Util;
import net.xfra.qizxopen.dm.SAXEventReceiver;
import net.xfra.qizxopen.dm.FONIDocument;
import net.xfra.qizxopen.xquery.dm.Node;
import net.xfra.qizxopen.xquery.dm.FONIDataModel;
import net.xfra.qizxopen.xquery.dt.SingleNode;

import org.xml.sax.*;
import org.xml.sax.ext.LexicalHandler;
import java.io.IOException;
import java.net.URL;

/**
 *  An XQueryProcessor that evaluates a query and yields a document in the form
 *  of SAX2 events. It can be used as a source to other XML/JAXP processors.
 *  <p>Steps to follow to use this object:<ul>
 *  <li>Create a SAXXQueryProcessor
 *  <li>set SAX handlers
 *  <li>compile a Query, install it with setQuery
 *  <li>use parse() to execute the query and send the result (a document) to
 *  the SAX output. Optionally the InputSource can be parsed and used as
 *  input by the query.
 *  </ul>
 */
public class SAXXQueryProcessor extends XQueryProcessor implements XMLReader
{
    private XQuery query;
    private SAXEventReceiver saxout = new SAXEventReceiver();
    private DTDHandler dtdHandler;	// not used
    private EntityResolver entityResolver;	// not used

    static final String FEATURE_NS =
                    "http://xml.org/sax/features/namespaces";
    static final String FEATURE_NSPREFIXES =
                    "http://xml.org/sax/features/namespace-prefixes";
    static final String PROPERTY_LEXICAL =
                    "http://xml.org/sax/properties/lexical-handler";

    public SAXXQueryProcessor( ) {
    }

    public SAXXQueryProcessor( String moduleBaseURI, String baseURI )
	throws IOException {
	super( moduleBaseURI, baseURI );
    }

    public SAXXQueryProcessor( XQueryProcessor master ) {
	super(master);
    }

    /**
     *	Specifies the evaluated query.
     */
    public void setQuery( XQuery query ) {
	this.query = query;
    }

    /**
     *	Executes a query (defined by setQuery) and sends results to the
     *	SAX handlers. The query must evaluate as a document.
     *	The inputSource argument is optional (can be null) and defines
     *	the input() document for the query evaluation.
     *  @param inputSource optional input source. If not null, a document is parsed
     *	from it and made accessible by the query through the input() function.
     *	@throws IOException parsing the input
     *	@throws SAXException parsing the input or wrapping XQuery exceptions
     */
    public void parse(InputSource inputSource)
	throws IOException, SAXException {
	if(docMan == null)
	    throw new IllegalArgumentException("no Document Manager specified");
	if(query == null)
	    throw new IllegalArgumentException("no query specified");
	execute( inputSource == null ? null : docMan.parseDocument(inputSource) );
    }

    /**
     * Executes a query (defined by setQuery) and sends results to the SAX 
     * handlers.The query must evaluate as a document. The systemId argument 
     * defines the input() document for the query evaluation.
     *	@param systemId input source. A document is parsed from it and
     *	made accessible by the query through the input() function.
     *	@throws IOException parsing the input
     *	@throws SAXException parsing the input or wrapping XQuery exceptions
     */
    public void parse( String systemId )
	throws IOException, SAXException {
	if(docMan == null)
	    throw new IllegalArgumentException("no Document Manager specified");
	if(query == null)
	    throw new IllegalArgumentException("no query specified");
	execute( docMan.parseDocument( Util.uriToURL(systemId) ) );
    }

    

    public ContentHandler getContentHandler() {
	return saxout.getContentHandler();
    }
    public void setContentHandler(ContentHandler handler) {
	saxout.setContentHandler(handler);
    }

    public DTDHandler getDTDHandler() {
	return dtdHandler;
    }
    /**
     *	Not used.
     */
    public void setDTDHandler(DTDHandler handler) {
	dtdHandler = handler;
    }

    public EntityResolver getEntityResolver() {
	return entityResolver;
    }
    /**
     *	Not used.
     */
    public void setEntityResolver(EntityResolver resolver) {
	entityResolver = resolver;
    }

    public ErrorHandler getErrorHandler() {
	return saxout.getErrorHandler();
    }
    public void setErrorHandler(ErrorHandler handler) {
	saxout.setErrorHandler(handler);
    }

    public boolean getFeature(String name)
	throws SAXNotRecognizedException, SAXNotSupportedException {
	if(name.equals(FEATURE_NS))
	    return true;
	else if(name.equals(FEATURE_NSPREFIXES))
	    return saxout.nsPrefixes;
	else {
	    
	    throw new SAXNotSupportedException(name);
	}
    }

    public void setFeature(String name, boolean value)
	throws SAXNotRecognizedException, SAXNotSupportedException {
	
	if(name.equals(FEATURE_NS)) {
	    if(!value)
		throw new SAXNotSupportedException(name);	// cant help managing NS!
	}
	else if(name.equals(FEATURE_NSPREFIXES))
	    saxout.nsPrefixes = value;
	else throw new SAXNotSupportedException(name);
    }

    public void setProperty(String name, Object value)
	throws SAXNotRecognizedException, SAXNotSupportedException {
	if(name.equals(PROPERTY_LEXICAL))
	    saxout.setLexicalHandler( (LexicalHandler) value );
	throw new SAXNotSupportedException(name);
    }    

    public Object getProperty(String name)
	throws SAXNotRecognizedException, SAXNotSupportedException {
	if(name.equals(PROPERTY_LEXICAL))
	    return saxout.getLexicalHandler();
	throw new SAXNotSupportedException(name);
    }
    
    void execute(FONIDocument input) throws IOException, SAXException {
	if(query == null)
	    throw new IllegalArgumentException("no query set");
	if(input != null)
	    this.input = new SingleNode(new FONIDataModel(input).getDocumentNode());
	try {
	    // saxout.setTrace(true);
	    executeQuery( query, saxout );
        }
        catch (XQueryException e) {
            SAXParseException saxe = new SAXParseException(e.getMessage(), null,
							   (Exception) e.getCause());
	    if(saxout.getErrorHandler() != null)
		saxout.getErrorHandler().fatalError(saxe);
	    throw saxe;
        }
    }

} // end of class SAXXQueryProcessor

