/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.dm.Node;
import net.xfra.qizxopen.dm.BaseNodeTest;
import net.xfra.qizxopen.xquery.impl.*;
import net.xfra.qizxopen.xquery.dt.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;

import java.util.HashMap;

/**
 *	Superclass of all Type representations.	
 */
public class Type {

    public static final byte ONE_OCC = 0;
    public static final byte OPT_OCC = 1;
    public static final byte MULTI_OCC = 2;
    public static final byte OPTMULTI_OCC = OPT_OCC | MULTI_OCC;

    public static final String ERR_EMPTY_UNEXPECTED = "unexpected empty sequence";
    public static final String ERR_TOO_MANY = "too many items";
    public static final String ERR_TYPE_MISMATCH = "improper item type";

    public static boolean isOptional(int occ) {
	return (occ & OPT_OCC) != 0;
    }
    public static boolean isRepeatable(int occ) {
	return (occ & MULTI_OCC) != 0;
    }

    /**
     *	Static type checking. Tells whether a static expression type can be 
     *  accepted by this formal type.
     */
    public boolean accepts( Type expressionType ) {
	return true;
    }

    /**
     *	Dynamic matching. Tells whether a dynamic value can be accepted by
     *	this type.  This can imply checking both the item types and the
     *	length of the sequence.
     *  @param value value to test.
     *  @return if OK: an expanded value (ArraySequence or SingleItem).
     */
    public Value check( Value value ) throws XQueryException {
	int occ = getOccurrence();
	if(!value.next()) {
	    if( isOptional(occ) )
		return value;
	    throw new TypeException(ERR_EMPTY_UNEXPECTED);
	}
	// Optimization: expand the value only if it is necessary or worth the trouble
	boolean expand = value.worthExpanding();
	Item current = null;
	ArraySequence seq = null;
	int count = 0;
	ItemType itemType = getItemType();
	do {
	    Item item = value.asItem();
	    if(! itemType.acceptsItem( item ) )	// ie current item of value
		throw new TypeException(ERR_TYPE_MISMATCH);
	    ++ count;
	    if(expand)
		if(current == null)
		    current = item;
		else {
		    if(seq == null) {
			seq = new ArraySequence(2);
			seq.addItem( current );
		    }
		    seq.addItem( item );
		}
	} while(value.next());
	if(seq != null)
	    seq.pack();
	Value result = expand ? ((seq != null) ? (Value) seq : new SingleItem(current))
	                      : value;
	switch(occ) {
	    default:	// non opt
		if(count == 1) return result;
		throw new TypeException(count == 0 ? ERR_EMPTY_UNEXPECTED : ERR_TOO_MANY);
	    case OPT_OCC:
		if(count <= 1) return result;
		throw new TypeException(ERR_TOO_MANY);
	    case OPTMULTI_OCC:
		return result;
	    case MULTI_OCC:
		if(count >= 1) return result;
		throw new TypeException(ERR_EMPTY_UNEXPECTED);
	}
    }

    /**
     *	Tests that a value matches a type.
     */
    public boolean test( Value value ) throws XQueryException {
	int occ = getOccurrence();
	if(!value.next())
	    return isOptional(occ);
	int count = 0;
	ItemType itemType = getItemType();
	
	do {
	    if(! itemType.acceptsItem( value.asItem() ) )	// ie current item of value
		return false;
	    ++ count;
	} while(value.next());

	switch(occ) {
	    default:	// non opt
		return count == 1;
	    case OPT_OCC:
		return count <= 1;
	    case MULTI_OCC:
		return count >= 1;
	    case OPTMULTI_OCC:
		return true;
	}
    }

    /**
     *	Dynamic matching of a single item. For optimization of typeswitch.
     */
    public boolean acceptsItem( Item item ) {
	return true;
    }

    /**
     *	Returns a type encompassing types this and that.
     *  TODO? optimize
     */
    public Type unionWith( Type that ) {
	ItemType union = getItemType(), thatType = that.getItemType();
	// NONE or empty() is neutral:
	if(union == Type.NONE)
	    union = thatType;
	else if(thatType != Type.NONE) {
	    if(thatType.accepts(union))
		union = thatType;
	    else //if(! union.accepts(thatType))
		for( ; union != null && !thatType.isDerivedFrom(union); )
		    union = union.getSuperType();
	}
	// else : union is this type
	if(union == null) { /// should not happen
	    System.err.println("cannot find union of types "+getItemType()+" & "+thatType);
	    return null;
	}
	int occ = getOccurrence() | that.getOccurrence();
	switch(occ) {
	case OPT_OCC:
	    return union.opt;
	case MULTI_OCC:
	    return union.plus;
	case OPTMULTI_OCC:
	    return union.star;
	default:
	    return union;
	}
    }

    /**
     *	If this is a sequence type, return the item type, otherwise return the type itself.
     */
    public ItemType getItemType() {
	return Type.ITEM;
    }

    /**
     *	If this is a sequence type, return the occurrence indicator, otherwise
     *	return ONE_OCC, except for anyType which is '*'.
     */
    public int getOccurrence() {
	return this == Type.ANY ? OPTMULTI_OCC : ONE_OCC;
    }

    public QName getName() {
	return QName.get(Namespace.XSD, getShortName());  // redefined for untypedAtomic
    }

    public String getShortName() {
	return "anyType";
    }

    public String toString(StaticContext ctx) {
	return toString();
    }

    public String toString() {
	return "xs:" + getShortName();
    }

    public void dump( ExprDump d ) {
	d.println("Type "+this);
    }

    /**
     *	Conversion of external Java objects to internal values.
     */
    public Value  convertFromObject( Object object ) {
	throw new RuntimeException("Type.convertFromObject "+object);
    }

    /**
     *	Conversion of internal values to external Java objects.
     */
    public Object convertToObject( Expression expr,
				   Focus focus, EvalContext context )
	throws XQueryException {
	return expr.eval(focus, context);
	//return context.error(expr, this+" cannot convert to Java object: "+expr);
    }

    // ----------- predefined types ---------------------------------------------

    static HashMap itemTypeMap = new HashMap();
    public static QName anyType;
    //public static QName anySimpleType;
    public static QName untypedAtomic;

    public static Type ANY;
    public static ItemType NONE;
    public static ItemType ITEM;
    public static NodeType NODE;
    public static NodeType ELEMENT;
    public static NodeType DOCUMENT;
    public static NodeType ATTRIBUTE;
    public static NodeType TEXT;
    public static NodeType PI;
    public static NodeType COMMENT;
    public static NodeType NAMESPACE;

    public static AtomicType ATOM;	// alias for ANY_ATOMIC_TYPE
    public static AtomicType ANY_ATOMIC_TYPE;
    public static AtomicType MOMENT;
    public static AtomicType TIME;
    public static AtomicType DATE;
    public static AtomicType DATE_TIME;
    public static AtomicType G_DAY;
    public static AtomicType G_MONTH;
    public static AtomicType G_YEAR;
    public static AtomicType G_YEAR_MONTH;
    public static AtomicType G_MONTH_DAY;
    public static AtomicType DURATION;
    public static AtomicType UNTYPED_ATOMIC;
    public static AtomicType BOOLEAN;
    public static AtomicType BINARY;
    public static AtomicType HEX_BINARY;
    public static AtomicType BASE64_BINARY;
    public static AtomicType NUMERIC;
    public static AtomicType FLOAT;
    public static AtomicType DOUBLE;
    public static AtomicType DECIMAL;
    public static AtomicType INTEGER;
    public static AtomicType NON_POSITIVE_INTEGER;
    public static AtomicType NEGATIVE_INTEGER;
    public static AtomicType LONG;
    public static AtomicType INT;
    public static AtomicType SHORT;
    public static AtomicType BYTE;
    public static AtomicType NON_NEGATIVE_INTEGER;
    public static AtomicType UNSIGNED_LONG;
    public static AtomicType UNSIGNED_INT;
    public static AtomicType UNSIGNED_SHORT;
    public static AtomicType UNSIGNED_BYTE;
    public static AtomicType POSITIVE_INTEGER;
    public static AtomicType STRING;
    public static AtomicType NORMALIZED_STRING;
    public static AtomicType TOKEN;
    public static AtomicType LANGUAGE;
    public static AtomicType NAME;
    public static AtomicType NCNAME;
    public static AtomicType ID;
    public static AtomicType IDREF;
    public static AtomicType ENTITY;
    public static AtomicType NMTOKEN;
    public static AtomicType NOTATION;
    public static AtomicType QNAME;
    public static AtomicType ANYURI;

    // xdt: types:
    public static AtomicType OBJECT;
    public static AtomicType WRAPPED_OBJECT;	// alias for OBJECT
    public static AtomicType YEAR_MONTH_DURATION;
    public static AtomicType DAY_TIME_DURATION;
    public static AtomicType CHAR;

    /**
     *	Searches a predefined Item Type by name. Returns null if none found.
     */
    public static ItemType findItemType( QName typeName ) {
	return (ItemType) itemTypeMap.get(typeName);
    }

    private static void defItemType( ItemType type, ItemType parent ) {
	type.parent = parent;
	itemTypeMap.put(type.getName(), type);
    }

    static {
	anyType = QName.get(Namespace.XSD, "anyType");
	//anySimpleType = QName.get(Namespace.XSD, "anySimpleType");
	untypedAtomic = QName.get(Namespace.XDT, "untypedAtomic");

	ANY = new Type();
	defItemType(ITEM = new ItemType(), null);
	defItemType(NODE = new NodeType(null), ITEM);

	ELEMENT = new NodeType(new BaseNodeTest(Node.ELEMENT, null, null));
	defItemType(ELEMENT, NODE);

	DOCUMENT = new NodeType(new BaseNodeTest(Node.DOCUMENT, null, null));
	defItemType(DOCUMENT, NODE);

	ATTRIBUTE = new NodeType(new BaseNodeTest(Node.ATTRIBUTE, null, null));
	defItemType(ATTRIBUTE, NODE);

	TEXT = new NodeType(new BaseNodeTest(Node.TEXT, null, null));
	defItemType(TEXT, NODE);

	PI = new NodeType(new BaseNodeTest(Node.PROCESSING_INSTRUCTION,
					   null, null));
	defItemType(PI, NODE);

	COMMENT = new NodeType(new BaseNodeTest(Node.COMMENT, null, null));
	defItemType(COMMENT, NODE);

	NAMESPACE= new NodeType(new BaseNodeTest(Node.NAMESPACE, null, null));
	defItemType(NAMESPACE, NODE);

        defItemType(ATOM = ANY_ATOMIC_TYPE = new AtomicType(), ITEM);
        defItemType(NONE = new EmptyType(), ITEM);	// special for empty
        OBJECT = WRAPPED_OBJECT = new WrappedObjectType(Object.class);
	defItemType(OBJECT, ATOM);
        defItemType(CHAR = new CharType(), UNSIGNED_INT);

        defItemType(MOMENT = new MomentType(), ATOM);	// internal
        defItemType(TIME = new TimeType(), ATOM);
        defItemType(DATE = new DateType(), ATOM);
        defItemType(DATE_TIME = new DateTimeType(), ATOM);
        defItemType(G_DAY = new GDayType(), ATOM);
        defItemType(G_MONTH = new GMonthType(), ATOM);
        defItemType(G_YEAR = new GYearType(), ATOM);
        defItemType(G_YEAR_MONTH = new GYearMonthType(), ATOM);
        defItemType(G_MONTH_DAY = new GMonthDayType(), ATOM);
        defItemType(DURATION = new DurationType(), ATOM);
        defItemType(YEAR_MONTH_DURATION = new YearMonthDurationType(), DURATION);
        defItemType(DAY_TIME_DURATION = new DayTimeDurationType(), DURATION);
        defItemType(UNTYPED_ATOMIC = new UntypedAtomicType(), ATOM);
        defItemType(BOOLEAN = new BooleanType(), ATOM);
        defItemType(BINARY = new BinaryType(), ATOM);
        defItemType(HEX_BINARY = new HexBinaryType(), ATOM);
        defItemType(BASE64_BINARY = new Base64BinaryType(), ATOM);
        defItemType(NUMERIC = new NumericType(), ATOM);	// internal
        defItemType(FLOAT = new FloatType(), ATOM);
        defItemType(DOUBLE = new DoubleType(), ATOM);
        defItemType(DECIMAL = new DecimalType(), ATOM);
        defItemType(INTEGER = new IntegerType(), DECIMAL);
        defItemType(NON_POSITIVE_INTEGER = new NonPositiveIntegerType(), INTEGER);
        defItemType(NEGATIVE_INTEGER = new NegativeIntegerType(), NON_POSITIVE_INTEGER);
        defItemType(LONG = new LongType(), INTEGER);
        defItemType(INT = new IntType(), LONG);
        defItemType(SHORT = new ShortType(), INT);
        defItemType(BYTE = new ByteType(), SHORT);
        defItemType(NON_NEGATIVE_INTEGER = new NonNegativeIntegerType(), INTEGER);
        defItemType(UNSIGNED_LONG = new UnsignedLongType(), NON_NEGATIVE_INTEGER);
        defItemType(UNSIGNED_INT = new UnsignedIntType(), UNSIGNED_LONG);
        defItemType(UNSIGNED_SHORT = new UnsignedShortType(), UNSIGNED_INT);
        defItemType(UNSIGNED_BYTE = new UnsignedByteType(), UNSIGNED_SHORT);
        defItemType(POSITIVE_INTEGER = new PositiveIntegerType(), NON_NEGATIVE_INTEGER);
        defItemType(STRING = new StringType(), ATOM);
        defItemType(NORMALIZED_STRING = new NormalizedStringType(), STRING);
        defItemType(TOKEN = new TokenType(), NORMALIZED_STRING);
        defItemType(LANGUAGE = new LanguageType(), TOKEN);
        defItemType(NAME = new NameType(), TOKEN);
        defItemType(NCNAME = new NCNameType(), NAME);
        defItemType(ID = new IDType(), NCNAME);
        defItemType(IDREF = new IDREFType(), NCNAME);
        defItemType(ENTITY = new ENTITYType(), NCNAME);
        defItemType(NMTOKEN = new NMTOKENType(), TOKEN);
        defItemType(NOTATION = new NotationType(), ATOM);
        defItemType(QNAME = new QNameType(), ATOM);
        defItemType(ANYURI = new AnyURIType(), ATOM);
    }
}
