/*
 *	Qizx/Open version 0.4p2
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery;

import net.xfra.qizxopen.xquery.dt.BaseValue;
import net.xfra.qizxopen.util.QName;
import java.text.Collator;

/**
 *	Represents the result of the evaluation of a XML Query, which is a sequence of Items.
 *	
 *	<p>The next() method must be repeatedly invoked to check that an item is available.
 *	<p>When next() returns true, the current item value can be retrieved through 
 *	specialized methods asX() (defined in super-interface Item) according to its type.
 *	<p>Value has often the dual aspect of a sequence and an Item. It is not immutable,
 *	because of the next() method. It can be regenerated by bornAgain().
 */
public interface Value extends Item
{
    /**
     *	Attempts to get the next atomic item. If true is returned, the item value is 
     *  available through one of the specialized accessors.
     */
    boolean  next() throws XQueryException;

    /**
     *	Attempts to get the next atomic item, without expanding collections.
     */
    boolean  nextCollection() throws XQueryException;

    /**
     *	Gets the current item (undefined result if end of sequence reached).
     */
    Item     asItem() throws TypeException;

    /**
     *	Gets the current item with atomization.
     *	If the current item is a Node, it is transparently transformed into
     *	a sequence of atomic values (in the future non Basic implementation), 
     *  or simply to its string value (XQ Basic).
     */
    Item     asAtom() throws TypeException;

    /**
     *	Returns a new version of the sequence <i>in its initial state</i>.
     *  Value objects are in principle immutable, but due to the iterative implementation
     *  style (see the {@link #next()} method), this is not actually true. Therefore when
     *  a value is used several times (in particular the value of a variable), there is a
     *  need for "regeneration". <p>NOTE: this needs not be a deep copy, because only the
     *  state of the iterator is concerned, not the underlying data.
     */
    abstract Value  bornAgain();

    /**
     *	Returns true if it is more efficient to keep the value as an iterator, and not
     *	to expand it into an array sequence. This method is used when a value is stored
     *	into variable. It should return true only when the value is cheap to compute AND
     *	doesnt involve Node constructors (because of a problem of Node identity).
     */
    abstract boolean worthExpanding();

    /**
     *	Boosts iterations that dont need the item value (count(), skipping)
     */
    abstract void setLazy(boolean lazy);

    /**
     *	Optimized evaluation of count() or last().
     *	Assumes the iterator to be in initial state.
     *	@param context evaluation context: used for debugging and timeout detection.
     */
    abstract long quickCount(EvalContext context) throws XQueryException;

    /**
     *	The empty sequence (single instance)
     */
    static Value empty = new Empty();

    /**
     *	The empty sequence.
     */
    static class Empty extends BaseValue {

	public ItemType  getType() {
	    return Type.ATOM;
	}

	public boolean next() {
	    return false;	// by essence
	}

	public Value  bornAgain() {
	    return this;
	}

	public boolean  isNode() {
	    return false;	// should not be called
	}

	public Item asItem() throws TypeException {
	    return null;	// whatever: should not be called
	}

	public Item asAtom() throws TypeException {
	    return asItem();	// should not be called
	}

	public int compareTo( Item that, Collator c, int implicitTimeZone) {
	    return INCOMPARABLE;
	}
    }
} // end of interface Value
